package com.devicenative.dna.datasource.loader;

import android.app.admin.DevicePolicyManager;
import android.app.usage.UsageStats;
import android.app.usage.UsageStatsManager;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.pm.ApplicationInfo;
import android.content.pm.LauncherActivityInfo;
import android.content.pm.LauncherApps;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.ResolveInfo;
import android.content.res.Configuration;
import android.content.res.Resources;
import android.os.Build;
import android.os.UserHandle;
import android.os.UserManager;
import android.provider.AlarmClock;
import android.os.RemoteException;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;

import com.devicenative.dna.db.DNAAppRecord;
import com.devicenative.dna.db.DNADatabaseInterface;
import com.devicenative.dna.db.DNADatabaseQueue;
import com.devicenative.dna.utils.DNAConstants;
import com.devicenative.dna.utils.DNALogger;
import com.devicenative.dna.utils.DNAPreferences;
import com.devicenative.dna.utils.DNAStatsLogger;
import com.devicenative.dna.utils.DNAUtils;

import org.json.JSONObject;

public class DNAAppLoader extends DNALoader {

    public DNAAppLoader(Context context) {
        super(context);
    }

    @Override
    public void run() {
        DNADatabaseQueue.enqueue(context.get(), () -> {
            loadAllApps();
        });
    }

    private void loadAllApps() {
        List<DNAAppRecord> apps = new ArrayList<>();

        Context ctx = context.get();
        DNAPreferences prefs = DNAPreferences.getInstance(ctx);

        UserManager manager = (UserManager) ctx.getSystemService(Context.USER_SERVICE);
        LauncherApps launcher = (LauncherApps) ctx.getSystemService(Context.LAUNCHER_APPS_SERVICE);

        PackageManager pManager = ctx.getPackageManager();
        Map<String, String> tagsMap = getTagsMap(pManager);

        UsageStatsManager usageStatsManager = (UsageStatsManager) ctx.getSystemService(Context.USAGE_STATS_SERVICE);
        boolean isUsageStatsEnabled = false;

        Calendar cal = Calendar.getInstance();
        cal.add(Calendar.YEAR, -2);

        Map<String, UsageStats> statsMap = new HashMap<>();
        try {
            statsMap = usageStatsManager.queryAndAggregateUsageStats(
                    cal.getTimeInMillis(),
                    System.currentTimeMillis());
        } catch (RuntimeException e) {
            DNALogger.e("DNAAppLoader: queryAndAggregateUsageStats failed: " + e.getMessage());
        }

        for (android.os.UserHandle profile : manager.getUserProfiles()) {

            if (!manager.isUserRunning(profile) ||
                    !manager.isUserUnlocked(profile)) {
                continue;
            }

            boolean isUserInQuietMode = manager.isQuietModeEnabled(profile);

            List<LauncherActivityInfo> launcherActivities;
            try {
                launcherActivities = launcher.getActivityList(null, profile);
            } catch (RuntimeException e) {
                DNALogger.e("DNAAppLoader: getActivityList failed for " + profile + ": " + e.getMessage());
                continue;
            }

            for (LauncherActivityInfo activityInfo : launcherActivities) {
                if (activityInfo == null) {
                    continue;
                }
                if (Thread.currentThread().isInterrupted()) {
                    return;
                }

                ApplicationInfo appInfo = activityInfo.getApplicationInfo();

                if (!launcher.isPackageEnabled(appInfo.packageName, profile)) {
                    continue;
                }

                if (!launcher.isActivityEnabled(activityInfo.getComponentName(), profile)) {
                    continue;
                }

                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
                    if (launcher.shouldHideFromSuggestions(appInfo.packageName, profile)) {
                        prefs.setPackageHidden(appInfo.packageName, profile.toString(), true);
                    } else if (prefs.isPackageHidden(appInfo.packageName, profile.toString())) {
                        prefs.setPackageHidden(appInfo.packageName, profile.toString(), false);
                    }
                }

                if (DNAUtils.isArchived(appInfo) || DNAUtils.isSuspended(appInfo) || isUserInQuietMode) {
                    prefs.setPackageHidden(appInfo.packageName, profile.toString(), true);
                } else if (prefs.isPackageHidden(appInfo.packageName, profile.toString())) {
                    prefs.setPackageHidden(appInfo.packageName, profile.toString(), false);
                }

                DNAAppRecord app = new DNAAppRecord();
                app.packageName = appInfo.packageName;
                app.userId = profile.toString();
                if (activityInfo.getLabel() != null) {
                    app.name = String.valueOf(activityInfo.getLabel());
                } else {
                    app.name = appInfo.loadLabel(pManager).toString();
                }
                app.component = activityInfo.getName();
                app.installedAt = activityInfo.getFirstInstallTime();
                app.updatedAt = System.currentTimeMillis();

                if (tagsMap.containsKey(app.packageName)) {
                    app.aliases = tagsMap.get(app.packageName);
                }

                if (statsMap.containsKey(app.packageName)) {
                    isUsageStatsEnabled = true;
                    app.lastUsedAt = Objects.requireNonNull(statsMap.get(app.packageName)).getLastTimeUsed();
                    app.timeUsedInLast3Y = Objects.requireNonNull(statsMap.get(app.packageName)).getTotalTimeInForeground()/1000;
                }

                apps.add(app);
            }
        }

        if (isUsageStatsEnabled) {
            prefs.setFirstUsageRead(true);
        }

        List<DNAAppRecord> existingApps = DNADatabaseInterface.getAllAppRecords(ctx);
        List<DNAAppRecord> appsToInsertOrUpdate = new ArrayList<>();

        for (DNAAppRecord app : apps) {
            boolean found = false;
            for (DNAAppRecord existingApp : existingApps) {
                if (existingApp.packageName.equals(app.packageName) &&
                        existingApp.userId.equals(app.userId) &&
                        existingApp.uninstalledAt == 0 &&
                        (existingApp.component == null || Objects.equals(existingApp.component, app.component)) &&
                        Objects.equals(existingApp.name, app.name) &&
                        Objects.equals(existingApp.aliases, app.aliases)) {
                    found = true;
                    break;
                }
            }
            if (!found) {
                appsToInsertOrUpdate.add(app);
            }
        }

        for (DNAAppRecord existingApp : existingApps) {
            boolean found = false;
            for (DNAAppRecord app : apps) {
                if (existingApp.packageName.equals(app.packageName) && Objects.equals(existingApp.userId, app.userId)) {
                    found = true;
                    break;
                }
            }
            if (!found & existingApp.uninstalledAt == 0) {
                existingApp.uninstalledAt = System.currentTimeMillis();
                appsToInsertOrUpdate.add(existingApp);
            }
        }

        DNADatabaseInterface.insertOrUpdateAppRecords(ctx, appsToInsertOrUpdate);

        apps.clear();
        appsToInsertOrUpdate.clear();
        existingApps.clear();

        runnableFinished();
    }

    private Map<String, String> getTagsMap(PackageManager pManager) {
        Map<String, String> tagsMap = new HashMap<>();

        // Fetching apps and assigning tags
        mapAppsToTags(tagsMap, pManager, new Intent(Intent.ACTION_DIAL), DNAConstants.IntentAlias.PHONE_ALIAS);
        mapAppsToTags(tagsMap, pManager, getCategoryIntent(Intent.CATEGORY_APP_CONTACTS), DNAConstants.IntentAlias.CONTACTS_ALIAS);
        mapAppsToTags(tagsMap, pManager, getCategoryIntent(Intent.CATEGORY_APP_BROWSER), DNAConstants.IntentAlias.BROWSER_ALIAS);
        mapAppsToTags(tagsMap, pManager, getCategoryIntent(Intent.CATEGORY_APP_EMAIL), DNAConstants.IntentAlias.EMAIL_ALIAS);
        mapAppsToTags(tagsMap, pManager, getCategoryIntent(Intent.CATEGORY_APP_MARKET), DNAConstants.IntentAlias.MARKET_ALIAS);
        mapAppsToTags(tagsMap, pManager, getCategoryIntent(Intent.CATEGORY_APP_MESSAGING), DNAConstants.IntentAlias.MESSAGING_ALIAS);
        mapAppsToTags(tagsMap, pManager, new Intent(AlarmClock.ACTION_SHOW_ALARMS), DNAConstants.IntentAlias.CLOCK_ALIAS);
        mapAppsToTags(tagsMap, pManager, getCategoryIntent(Intent.CATEGORY_APP_MAPS), DNAConstants.IntentAlias.MAPS_ALIAS);
        mapAppsToTags(tagsMap, pManager, new Intent(Intent.ACTION_CAMERA_BUTTON), DNAConstants.IntentAlias.CAMERA_ALIAS);
        mapAppsToTags(tagsMap, pManager, getCategoryIntent(Intent.CATEGORY_APP_MUSIC), DNAConstants.IntentAlias.MUSIC_ALIAS);
        mapAppsToTags(tagsMap, pManager, getCategoryIntent(Intent.CATEGORY_APP_CALENDAR), DNAConstants.IntentAlias.CALENDAR_ALIAS);
        mapAppsToTags(tagsMap, pManager, getCategoryIntent(Intent.CATEGORY_APP_GALLERY), DNAConstants.IntentAlias.GALLERY_ALIAS);
        mapAppsToTags(tagsMap, pManager, getCategoryIntent(Intent.CATEGORY_APP_FILES), DNAConstants.IntentAlias.FILE_MANAGER_ALIAS);

        return tagsMap;
    }

    private Intent getCategoryIntent(String category) {
        Intent intent = new Intent(Intent.ACTION_MAIN, null);
        intent.addCategory(category);
        return intent;
    }

    private void mapAppsToTags(Map<String, String> tagsMap, PackageManager pManager, Intent intentFilter, String alias) {
        List<String> apps = getAppsByCategory(pManager, intentFilter);
        for (String app : apps) {
            tagsMap.merge(app, alias, (oldValue, newValue) -> oldValue.contains(newValue) ? oldValue : oldValue + "," + newValue);
        }
    }
    private List<String> getAppsByCategory(PackageManager pManager, Intent intent) {
        List<ResolveInfo> list = pManager.queryIntentActivities(intent, 0);
        List<String> apps = new ArrayList<>(list.size());
        for (ResolveInfo resolveInfo : list) {
            String pName = resolveInfo.activityInfo.applicationInfo.packageName;
            apps.add(pName);
        }
        return apps;
    }
}
