package com.devicenative.dna;


import android.app.ActivityManager;
import android.app.KeyguardManager;
import android.content.BroadcastReceiver;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.ServiceConnection;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.os.Build;
import android.os.Handler;
import android.os.IBinder;
import android.os.UserHandle;
import android.service.notification.StatusBarNotification;
import android.util.Log;

import com.devicenative.dna.DNADataOrchestrator.LocalBinder;
import com.devicenative.dna.ads.DNAResultServer;
import com.devicenative.dna.db.DNADatabaseInterface;
import com.devicenative.dna.notification.DNANotificationListener;
import com.devicenative.dna.utils.DNAConfigBuilder;
import com.devicenative.dna.utils.DNAConstants;
import com.devicenative.dna.utils.DNALogger;
import com.devicenative.dna.utils.DNAPreferences;
import com.devicenative.dna.utils.DNAStatsLogger;
import com.devicenative.dna.utils.DNAUtils;

import org.json.JSONObject;

import java.util.List;
import java.util.logging.Logger;


/**
 * This primary class is responsible for managing native ads on the device. The methods on
 * this class are all that is needed to start/stop the service and retrieve ads for display.
 * It follows the Singleton pattern to ensure only one instance of this class is created.
 */
public class DeviceNativeAds {

    /* The static instance of the DeviceNativeAds class */
    private static DeviceNativeAds instance_ = null;

    private Context context_;
    private boolean isActive_ = false;

    private DNADataOrchestrator dnaDataOrchestrator_;
    private ServiceConnection dataServiceConnection_;
    private boolean isBindingInProgress_ = false;

    private final DNAPreferences dnaPreferences_;

    private final DNAResultServer resultServer_;

    private static final long UPDATE_DEBOUNCE_INTERVAL_MS = 5000;
    private long lastUpdateTriggerTimestamp = 0;

//    private final DNANotificationListener dnaNotificationListener_;

    /**
     * The configuration class for the DeviceNativeAds class. This is used to set
     * various configuration options for the class. Key parameters include:
     * - debugMode: Enables verbose logging.
     * - disableGAID: Disables the Google Advertising ID.
     * - disableAppUsage: Disables app usage tracking.
     * - countryOverride: Overrides the country code for the device. Use ISO 3166-1 alpha-2 format.
     * - disableAds: Disables ads.
     * - disableSearchInstallAds: Disables search install ads.
     * - disableSearchAds: Disables all search ads.
     * - disableRecomInstallAds: Disables recommended install ads.
     * - disableRecomAds: Disables all recommended ads.
     * - carrierValue: Set the carrier value for the device.
     */
    public static class DNAConfig {
        public Boolean debugMode = null;
        public Boolean disableGAID = null;
        public Boolean disableAppUsage = null;
        public String countryOverride = null;
        public Boolean disableAds = null;
        public Boolean disableSearchInstallAds = null;
        public Boolean disableSearchAds = null;
        public Boolean disableRecomInstallAds = null;
        public Boolean disableRecomAds = null;
        public String carrierValue = null;
    }

    /* The initialization of the DeviceNativeAds class,
    private because of singleton pattern */
    private DeviceNativeAds(Context context) {
        context_ = context;
        dnaPreferences_ = DNAPreferences.getInstance(context);
        resultServer_ = DNAResultServer.getInstance(context);
//        dnaNotificationListener_ = new DNANotificationListener();
        isActive_ = true;
        
        // Create a single, reusable ServiceConnection to prevent memory leaks
        dataServiceConnection_ = new ServiceConnection() {
            @Override
            public void onServiceConnected(ComponentName className, IBinder service) {
                if (!(service instanceof DNADataOrchestrator.LocalBinder)) {
                    DNALogger.i("DeviceNativeAds: Ignoring cross-process binder for " + className);
                    isBindingInProgress_ = false;
                    return;
                }
                LocalBinder binder = (LocalBinder) service;
                dnaDataOrchestrator_ = binder.getService();
                isBindingInProgress_ = false;
                DNALogger.i("DeviceNativeAds: Connected to data orchestrator.");
            }

            @Override
            public void onServiceDisconnected(ComponentName name) {
                DNALogger.w("DeviceNativeAds: Disconnected from " + name + ". Cleaning up connection state.");
                // Important: Clean up state so we can reconnect later.
                dnaDataOrchestrator_ = null;
                isBindingInProgress_ = false;
            }
        };
    }

    /**
     * Static method to get the instance of the DeviceNativeAds class. This
     * is what you will use to get a reference to the static singleton.
     * @param context The application context.
     * @return The instance of the DeviceNativeAds class.
     */
    synchronized public static DeviceNativeAds getInstance(Context context) {
        if (instance_ == null || !instance_.isActive_) {
            instance_ = new DeviceNativeAds(context.getApplicationContext());
        }
        return instance_;
    }

    synchronized private void validateIsActiveAndReinitialize() {
        if (!isActive_) {
            dnaPreferences_.init(context_);
            resultServer_.init(context_);
            isActive_ = true;
        }
    }

    /**
     * Method to initialize the DeviceNativeAds class. This is intended to be called
     * in your Application onCreate method. It initializes the data service among other
     * things.
     * @param deviceKey The device key.
     */
    synchronized public void init(String deviceKey) {
        init(deviceKey, null);
    }

    /**
     * Method to initialize the DeviceNativeAds class. This is intended to be called
     * in your Application onCreate method. It initializes the data service among other
     * things.
     * @param deviceKey The device key.
     * @param config The configuration to use.
     */
    synchronized public void init(String deviceKey, DNAConfig config) {
        DNALogger.i("DeviceNativeAds init called");
         validateIsActiveAndReinitialize();

        dnaPreferences_.setDeviceKey(deviceKey);
        if (config != null) {
            updateConfig(config);
        } else {
            dnaPreferences_.setDebugMode(false);
            dnaPreferences_.setConfigCountryOverride(null);
            dnaPreferences_.setConfigGAIDDisabled(false);
            dnaPreferences_.setConfigAppUsageDisabled(false);
            dnaPreferences_.setConfigAdsDisabled(false);
            dnaPreferences_.setRecomInstallAdsDisabled(false);
            dnaPreferences_.setRecomAdsDisabled(false);
            dnaPreferences_.setSearchInstallAdsDisabled(false);
            dnaPreferences_.setSearchAdsDisabled(false);
        }
        // generate a UUID for this session
        dnaPreferences_.setSessionID(java.util.UUID.randomUUID().toString());
        // get the current package name and set in preferences
        dnaPreferences_.setAppPackageName(context_.getPackageName());
        // get the current app version and set in preferences
        dnaPreferences_.setAndroidVersion(String.valueOf(Build.VERSION.SDK_INT));
        dnaPreferences_.setDeviceModel(android.os.Build.MODEL);
        dnaPreferences_.setDeviceStartTime(Build.TIME);
        dnaPreferences_.setCPUCount(Runtime.getRuntime().availableProcessors());
        ActivityManager activityManager = (ActivityManager) context_.getSystemService(Context.ACTIVITY_SERVICE);
        ActivityManager.MemoryInfo memoryInfo = new ActivityManager.MemoryInfo();
        activityManager.getMemoryInfo(memoryInfo);
        long totalMemory = memoryInfo.totalMem;
        dnaPreferences_.setMaxMemory(totalMemory /(1024 * 1024));
        try {
            PackageInfo packageInfo = context_.getPackageManager().getPackageInfo(context_.getPackageName(), 0);
            dnaPreferences_.setAppVersion(packageInfo.versionName);
        } catch (PackageManager.NameNotFoundException e) {
            DNALogger.e("DeviceNativeAds: Unable to get app version");
        }

        JSONObject statsMetadata = new JSONObject();
        try {
            statsMetadata.put("sId", dnaPreferences_.getSessionID());
        } catch (Exception e) {
            DNALogger.e("DeviceNativeAds: Unable to create stats metadata");
        }
        DNAStatsLogger.logInternal(context_, "Session Start", null, statsMetadata);
        triggerUpdateCheck(false);
    }

    private boolean isDataOrchestratorRunning() {
        ActivityManager manager = (ActivityManager) context_.getSystemService(Context.ACTIVITY_SERVICE);
        if (manager != null) {
            for (ActivityManager.RunningServiceInfo service : manager.getRunningServices(Integer.MAX_VALUE)) {
                if (DNADataOrchestrator.class.getName().equals(service.service.getClassName())) {
                    return true;
                }
            }
        }
        return false;
    }

    private void initAndConnectToDataProvider(final int counter) {
        // Check if we are already connected or in the process of connecting
        if (dnaDataOrchestrator_ != null) {
            DNALogger.i("DeviceNativeAds: Already connected to data orchestrator.");
            return;
        }
        
        if (isBindingInProgress_) {
            DNALogger.i("DeviceNativeAds: Binding already in progress.");
            return;
        }

        DNALogger.i("DeviceNativeAds: Attempting to start and connect to data orchestrator.");
        Intent intent = new Intent(this.context_, DNADataOrchestrator.class);

        try {
            this.context_.startService(intent);
        } catch (IllegalStateException e) {
            DNALogger.w("DeviceNativeAds: Unable to start data service from background. Setting up retry.");

            if (counter > 20) {
                DNALogger.e("DeviceNativeAds: Already tried and failed twenty times to start service. Giving up.");
                return;
            }

            IntentFilter intentFilter = new IntentFilter();
            intentFilter.addAction(Intent.ACTION_SCREEN_ON);
            intentFilter.addAction(Intent.ACTION_USER_PRESENT);
            context_.registerReceiver(new BroadcastReceiver() {
                @Override
                public void onReceive(final Context context, Intent intent) {
                    KeyguardManager myKM = (KeyguardManager) context.getSystemService(Context.KEYGUARD_SERVICE);
                    if (myKM != null && !myKM.inKeyguardRestrictedInputMode()) {
                        context.unregisterReceiver(this);
                        new Handler().postDelayed(() -> {
                            DNALogger.i("DeviceNativeAds: Screen unlocked, retrying to start background services");
                            initAndConnectToDataProvider(counter + 1);
                        }, 10);
                    }
                }
            }, intentFilter);

            return;
        }

        // Use the single, reusable ServiceConnection to prevent memory leaks
        DNALogger.i("DeviceNativeAds: Binding to the service");
        isBindingInProgress_ = true;
        boolean isBinding = this.context_.bindService(intent, dataServiceConnection_, Context.BIND_AUTO_CREATE);

        if (!isBinding) {
            DNALogger.e("DeviceNativeAds: bindService returned false. Service may not be available or binding is not possible at this time.");
            isBindingInProgress_ = false;
        }
    }

    /**
     * Method to destroy the DeviceNativeAds instance. This is intended to be called from
     * your Application onTerminate method to clean up the service.
     */
    synchronized public void destroy() {
        DNALogger.i("DeviceNativeAds destroy called");
        JSONObject statsMetadata = new JSONObject();
        try {
            statsMetadata.put("sId", dnaPreferences_.getSessionID());
        } catch (Exception e) {
            DNALogger.e("DeviceNativeAds: Unable to create stats metadata");
        }
        DNAStatsLogger.logInternal(context_, "Session Stop", null, statsMetadata);
        
        // Clean up service connection
        if (dnaDataOrchestrator_ != null || isBindingInProgress_) {
            try {
                this.context_.unbindService(dataServiceConnection_);
            } catch (IllegalArgumentException e) {
                // Service was not bound, ignore
                DNALogger.i("DeviceNativeAds: Service was not bound during cleanup: " + e.getMessage());
            }
        }

        // Stop provider service
        if (dnaDataOrchestrator_ != null) {
            this.context_.stopService(new Intent(this.context_, DNADataOrchestrator.class));
        }
        
        // Clean up state
        dnaDataOrchestrator_ = null;
        isBindingInProgress_ = false;
        isActive_ = false;
        DNAResultServer.shutDown();
    }

    /**
     * Method to update the configuration of the DeviceNativeAds class. This is intended
     * to be called mid-session to alter the configuration of the service.
     * @param config
     */
    synchronized public void updateConfig(DNAConfig config) {
        triggerUpdateCheck(false);
        if (config != null && dnaPreferences_ != null) {
            if (config.debugMode != null) {
                dnaPreferences_.setDebugMode(config.debugMode);
            }
            if (config.carrierValue != null) {
                dnaPreferences_.setCarrierValue(config.carrierValue);
            }
            if (config.disableGAID != null) {
                dnaPreferences_.setConfigGAIDDisabled(config.disableGAID);
            }
            if (config.disableAppUsage != null) {
                dnaPreferences_.setConfigAppUsageDisabled(config.disableAppUsage);
            }
            if (config.disableAds != null) {
                dnaPreferences_.setConfigAdsDisabled(config.disableAds);
            }

            if (config.disableSearchAds != null) {
                dnaPreferences_.setSearchAdsDisabled(config.disableSearchAds);
            }

            if (config.disableSearchInstallAds != null) {
                dnaPreferences_.setSearchInstallAdsDisabled(config.disableSearchInstallAds);
            }

            if (config.disableRecomAds != null) {
                dnaPreferences_.setRecomAdsDisabled(config.disableRecomAds);
            }

            if (config.disableRecomInstallAds != null) {
                dnaPreferences_.setRecomInstallAdsDisabled(config.disableRecomInstallAds);
            }

            if (config.countryOverride != null) {
                dnaPreferences_.setConfigCountryOverride(config.countryOverride);
            }
        }
    }

    /**
     * Method to handle when a notification is posted. It sends the notification to the service
     * for later use in ad creation and targeting.
     * @param sbn The StatusBarNotification that was posted.
     */
    synchronized public void onNotificationPosted(StatusBarNotification sbn) {
//        if (dnaNotificationListener_ != null) {
//            dnaNotificationListener_.onNotificationPosted(context_, sbn);
//        }
    }

    /**
     * Method to set the visibility of a package in app suggestions. When set to false (hidden),
     * the package will be filtered out from suggested apps results. This setting is persistent
     * across app sessions and applies to all activities within the specified package for the
     * given user.
     * @param packageName The package name to hide or show in suggestions.
     * @param userId The user handle for multi-user support. Can be null for current user.
     * @param visible True to show the package in suggestions, false to hide it.
     */
    synchronized public void setPackageSuggestionVisibility(String packageName, UserHandle userId, boolean visible) {
        triggerUpdateCheck(false);
        if (packageName == null || packageName.isEmpty()) {
            return;
        }
        
        String userIdStr = userId != null ? userId.toString() : null;
        dnaPreferences_.setPackageHidden(packageName, userIdStr, !visible);
        resultServer_.clearCache();
    }

    /**
     * Method to set the visibility of a specific component in app suggestions. When set to false
     * (hidden), the component will be filtered out from suggested apps results. This setting is
     * persistent across app sessions and provides more granular control than package-level hiding.
     * Component-level hiding takes precedence over package-level settings.
     * @param componentName The fully qualified component name to hide or show in suggestions.
     * @param userId The user handle for multi-user support. Can be null for current user.
     * @param visible True to show the component in suggestions, false to hide it.
     */
    synchronized public void setComponentSuggestionVisibility(String componentName, UserHandle userId, boolean visible) {
        triggerUpdateCheck(false);
        if (componentName == null || componentName.isEmpty()) {
            return;
        }
        
        // Parse component name to extract activity name if in format "packageName/activityName#id"
        String activityName = DNAUtils.extractActivityName(componentName);
        
        String userIdStr = userId != null ? userId.toString() : null;
        dnaPreferences_.setComponentHidden(activityName, userIdStr, !visible);
        resultServer_.clearCache();
    }



    /**
     * Method to get organic app recommendations. This method return the organic app recommendations
     * in milliseconds, so it's safe to run on the main thread. Depending on your configuration,
     * ads will be intermixed through the results.
     * @param count The number of organic app recommendations to fetch.
     * @return The DNAResultItems to be displayed.
     */

    synchronized public List<DNAResultItem> getOrganicAppSuggestions(int count) {
        triggerUpdateCheck(false);
        return resultServer_.fetchOrganicAppSuggestions(count, null);
    }

    /**
     * Method to get organic app recommendations. This method return the organic app recommendations
     * in milliseconds, so it's safe to run on the main thread. Depending on your configuration,
     * ads will be intermixed through the results.
     * @param count The number of organic app recommendations to fetch.
     * @param placementTag The placement tag for the ads.
     * @return The DNAResultItems to be displayed.
     */
    synchronized public List<DNAResultItem> getOrganicAppSuggestions(int count, String placementTag) {
        triggerUpdateCheck(false);
        return resultServer_.fetchOrganicAppSuggestions(count, placementTag);
    }

    /**
     * Method to get organic deep link recommendations. This method return the organic dl recommendations
     * in milliseconds, so it's safe to run on the main thread. Depending on your configuration,
     * ads can be intermixed through the results.
     * @count The number of organic deep link recommendations to fetch.
     * @return The DNAResultItems to be displayed.
     */
    synchronized public List<DNAResultItem> getOrganicLinkSuggestions(int count) {
        triggerUpdateCheck(false);
        return resultServer_.fetchOrganicLinkSuggestions(count, null);
    }

    /**
     * Method to get organic deep link recommendations. This method return the organic dl recommendations
     * in milliseconds, so it's safe to run on the main thread. Depending on your configuration,
     * ads can be intermixed through the results.
     * @count The number of organic deep link recommendations to fetch.
     * @param placementTag The placement tag for the ads.
     * @return The DNAResultItems to be displayed.
     */
    synchronized public List<DNAResultItem> getOrganicLinkSuggestions(int count, String placementTag) {
        triggerUpdateCheck(false);
        return resultServer_.fetchOrganicLinkSuggestions(count, placementTag);
    }

    /**
     * Method to fetch a list of the top apps to download, which are likely all advertisements. It will return
     * in milliseconds, so it's safe to run on the main thread.
     * @count The number of hot apps to fetch.
     * @return The DNAResultItems to be displayed.
     */
    synchronized public List<DNAResultItem> getHotAppsList(int count) {
        triggerUpdateCheck(false);
        return resultServer_.fetchHotAppsList(count, null);
    }

    /**
     * Method to fetch a list of the top apps to download, which are likely all advertisements. It will return
     * in milliseconds, so it's safe to run on the main thread.
     * @count The number of hot apps to fetch.
     * @param placementTag The placement tag for the ads.
     * @return The DNAResultItems to be displayed.
     */
    synchronized public List<DNAResultItem> getHotAppsList(int count, String placementTag) {
        triggerUpdateCheck(false);
        return resultServer_.fetchHotAppsList(count, placementTag);
    }

    /**
     * Method to get organic app recommendations for a search. This method return the organic app
     * recommendations in milliseconds, so it's safe to run on the main thread. Depending on your
     * configuration, ads will be intermixed through the results.
     * @param query The search query.
     * @return The DNAResultItems to be displayed.
     */

    synchronized public List<DNAResultItem> getOrganicResultsForSearch(String query) {
        triggerUpdateCheck(false);
        return resultServer_.fetchOrganicResultsForSearch(query, null, true, -1);
    }

    /**
     * Method to get organic app recommendations for a search. This method return the organic app
     * recommendations in milliseconds, so it's safe to run on the main thread. Depending on your
     * configuration, ads will be intermixed through the results.
     * @param query The search query.
     * @param placementTag The placement tag for the ads.
     * @return The DNAResultItems to be displayed.
     */
    synchronized public List<DNAResultItem> getOrganicResultsForSearch(String query, String placementTag) {
        triggerUpdateCheck(false);
        return resultServer_.fetchOrganicResultsForSearch(query, placementTag, true, -1);
    }

    /**
     * Method to get organic app recommendations for a search. This method return the organic app
     * recommendations in milliseconds, so it's safe to run on the main thread. Depending on your
     * configuration, ads will be intermixed through the results.
     * @param query The search query.
     * @param placementTag The placement tag for the ads.
     * @param count The number of results to trigger impressions for
     * @return The DNAResultItems to be displayed.
     */
    synchronized public List<DNAResultItem> getOrganicResultsForSearch(String query, String placementTag, int count) {
        triggerUpdateCheck(false);
        return resultServer_.fetchOrganicResultsForSearch(query, placementTag, true, count);
    }

    /**
     * Method to get an search results for caching. You should not really use this, but if you must, you
     * must also use the fireImpression method below when you actually show the advert because
     * it does not fire an impression automatically. This method return an ad in milliseconds,
     * so it's safe to run on the main thread.
     * @param query The search query.
     * @param placementTag The placement tag for the ads.
     * @return The DNAResultItems to be displayed.
     */
    synchronized public List<DNAResultItem> getOrganicResultsForSearchForCache(String query, String placementTag) {
        triggerUpdateCheck(false);
        return resultServer_.fetchOrganicResultsForSearch(query, placementTag, false, -1);
    }

    /**
     * Method to get ads for display. It will automatically fire an impression immediately if
     * the impressionUrl is populated for the ad. This method return the ads in milliseconds,
     * so it's safe to run on the main thread.
     * @param count The number of ads to fetch.
     * @return The DNAResultItems to be displayed.
     */
    synchronized public List<DNAResultItem> getAdsForDisplay(int count) {
        triggerUpdateCheck(false);
        return resultServer_.fetchAdsForDisplay(count, null);
    }

    /**
     * Method to get ads for display. It will automatically fire an impression immediately if
     * the impressionUrl is populated for the ad. This method return the ads in milliseconds,
     * so it's safe to run on the main thread.
     * @param count The number of ads to fetch.
     * @param placementTag The placement tag for the ads.
     * @return The DNAResultItems to be displayed.
     */
    synchronized public List<DNAResultItem> getAdsForDisplay(int count, String placementTag) {
        triggerUpdateCheck(false);
        return resultServer_.fetchAdsForDisplay(count, placementTag);
    }

    /**
     * Method to get an ad for search. This method return the relevant ads in milliseconds, and
     * automatically fire an impression if the impressionUrl is populated for the ad.
     * @param query The search query.
     * @return The DNAResultItem to be displayed.
     */

    synchronized public List<DNAResultItem> getAdsForSearch(String query) {
        triggerUpdateCheck(false);
        return resultServer_.fetchAdsForSearch(query, null, true);
    }

    /**
     * Method to get an ad for search. This method return the relevant ads in milliseconds, and
     * automatically fire an impression if the impressionUrl is populated for the ad.
     * @param query The search query.
     * @param placementTag The placement tag for the ads.
     * @return The DNAResultItem to be displayed.
     */
    synchronized public List<DNAResultItem> getAdsForSearch(String query, String placementTag) {
        triggerUpdateCheck(false);
        return resultServer_.fetchAdsForSearch(query, placementTag, true);
    }

    /**
     * Method to get an search results for caching. You should not really use this, but if you must, you
     * must also use the fireImpression method below when you actually show the advert because
     * it does not fire an impression automatically. This method return an ad in milliseconds,
     * so it's safe to run on the main thread.
     * @param query The search query.
     * @return The DNAResultItem to be displayed.
     */

    synchronized public List<DNAResultItem> getAdsForSearchForCache(String query) {
        triggerUpdateCheck(false);
        return resultServer_.fetchAdsForSearch(query, null, false);
    }

    /**
     * Method to get an search results for caching. You should not really use this, but if you must, you
     * must also use the fireImpression method below when you actually show the advert because
     * it does not fire an impression automatically. This method return an ad in milliseconds,
     * @param query The search query.
     * @param placementTag The placement tag for the ads.
     * @return The DNAResultItem to be displayed.
     */
    synchronized public List<DNAResultItem> getAdsForSearchForCache(String query, String placementTag) {
        triggerUpdateCheck(false);
        return resultServer_.fetchAdsForSearch(query, placementTag, false);
    }


    /**
     * Method to get an ads for caching. You should not really use this, but if you must, you
     * must also use the fireImpression method below when you actually show the advert because
     * it does not fire an impression automatically. This method return an ad in milliseconds,
     * so it's safe to run on the main thread.
     * @param count The number of ads to fetch.
     * @return The DNAResultItem to be displayed.
     */
    synchronized public List<DNAResultItem> getAdsForCache(int count) {
        triggerUpdateCheck(false);
        return resultServer_.fetchAdsForCache(count, null);
    }

    /**
     * Method to get an ads for caching. You should not really use this, but if you must, you
     * must also use the fireImpression method below when you actually show the advert because
     * it does not fire an impression automatically. This method return an ad in milliseconds,
     * so it's safe to run on the main thread.
     * @param count The number of ads to fetch.
     * @param placementTag The placement tag for the ads.
     * @return The DNAResultItem to be displayed.
     */
    synchronized public List<DNAResultItem> getAdsForCache(int count, String placementTag) {
        triggerUpdateCheck(false);
        return resultServer_.fetchAdsForCache(count, placementTag);
    }

    /**
     * Not recommended without consult. Use getAdForDisplay instead and show the ad immediately.
     * Method to fire an impression, but should ONLY be used if you are caching ads. This
     * executes on a separate thread, and it is safe to fire and forget. Fine to pass null
     * to the clickHandler callback.
     * @param resultItems The list of DNAResultItems for which the impression is to be fired.
     * @param clickHandler The DeviceNativeAdClickHandler instance. Optional. Fine to pass null.
     */
    synchronized public void fireImpressions(List<DNAResultItem> resultItems, DeviceNativeClickHandler clickHandler) {
        triggerUpdateCheck(false);
        resultServer_.registerBatchImpression(resultItems, clickHandler);
    }

    /**
     * Not recommended without consult. Use getAdForDisplay instead and show the ad immediately.
     * Method to fire an impression, but should ONLY be used if you are caching ads. This
     * executes on a separate thread, and it is safe to fire and forget. Fine to pass null
     * to the clickHandler callback.
     * @param resultItems The list of DNAResultItems for which the impression is to be fired.
     * @param clickHandler The DeviceNativeAdClickHandler instance. Optional. Fine to pass null.
     */
    synchronized public void fireImpressions(List<DNAResultItem> resultItems, String placementTag, DeviceNativeClickHandler clickHandler) {
        triggerUpdateCheck(false);
        resultServer_.registerBatchImpression(resultItems, placementTag, false, clickHandler);
    }

    /**
     * Not recommended without consult. Let DNA handle the routing due to the complexity of the routing.
     * Method to fire a click without routing. It assumes you will handle routing yourself. It
     * executes on a separate thread, and it is safe to fire and forget. Fine to pass null
     * to the clickHandler callback.
     * @param resultItem The DNAResultItem for which the click is fired.
     * @param clickHandler The DeviceNativeAdClickHandler instance. Optional. Fine to pass null.
     */
    synchronized public void fireClickWithoutRouting(DNAResultItem resultItem, DeviceNativeClickHandler clickHandler) {
        triggerUpdateCheck(false);
        resultServer_.registerClick(resultItem, clickHandler);
    }

    /**
     * Method to fire a click without routing for non-DNAResultItem objects. It assumes you will 
     * handle routing yourself. It executes on a separate thread, and it is safe to fire and forget.
     * @param packageName The package name of the app being clicked.
     * @param componentName The component name of the app being clicked.
     * @param userHandle The user handle for multi-user support. Can be null for current user.
     */
    synchronized public void fireClickWithoutRouting(String packageName, String componentName, UserHandle userHandle) {
        triggerUpdateCheck(false);
        // Parse component name to extract activity name if in format "packageName/activityName#id" or "ComponentInfo{...}"
        String activityName = DNAUtils.extractActivityName(componentName);
        resultServer_.clearCache();
        resultServer_.registerClick(packageName, activityName, userHandle);
    }

    /**
     * Method to fire a click and route the user to the destination. It executes on a separate
     * thread, and loading could take a second, so it's recommended to show a loading indicator
     * until the callback is fired. Fine to pass null to the clickHandler callback if you don't
     * need to.
     * @param resultItem The DNAResultItem for which the click is fired.
     * @param clickHandler The DeviceNativeAdClickHandler instance.
     */
    synchronized public void fireClickAndRoute(DNAResultItem resultItem, DeviceNativeClickHandler clickHandler) {
        triggerUpdateCheck(false);
        resultServer_.clickAndRoute(resultItem, null, clickHandler);
    }

    /**
     * Method to fire a click and route the user to the destination, but it allows you to override
     * the destination URL. It executes on a separate thread, and loading could take a second, so
     * it's recommended to show a loading indicator until the callback is fired. Fine to pass null
     * to the clickHandler callback if you don't need the callback status.
     * @param resultItem The DNAResultItem for which the click is fired.
     * @param destinationUrlOverride The destination URL to override the one in the result item.
     * @param clickHandler The DeviceNativeAdClickHandler instance.
     */
    synchronized public void fireClickAndRoute(DNAResultItem resultItem, String destinationUrlOverride, DeviceNativeClickHandler clickHandler) {
        triggerUpdateCheck(false);
        resultServer_.clickAndRoute(resultItem, destinationUrlOverride, clickHandler);
    }

    /**
     * Method to trigger a data refresh. This is intended to be called when you want to force
     * a data refresh, for example after a profile change.
     */
    synchronized public void triggerDataRefresh() {
        triggerUpdateCheck(true);
    }

    private void triggerUpdateCheck(boolean force) {
        long now = System.currentTimeMillis();
        if (!force && now - lastUpdateTriggerTimestamp < UPDATE_DEBOUNCE_INTERVAL_MS) {
            DNALogger.i("DeviceNativeAds: Debouncing update trigger.");
            return;
        }
        lastUpdateTriggerTimestamp = now;

        DNALogger.i("DeviceNativeAds: triggerUpdateCheck called with force: " + force);
        initAndConnectToDataProvider(0);

        if (force) {
            try {
                Intent svc = new Intent(context_, DNADataOrchestrator.class)
                        .setAction("com.devicenative.dna.WAKE_AND_SYNC");
                context_.startService(svc);
            } catch (Exception e) {
                DNALogger.e("DeviceNativeAds: Error firing wake broadcast: " + e.getMessage());
            }
        }
    }
}
