package com.devicenative.dna;

import static android.content.pm.LauncherApps.ShortcutQuery.FLAG_MATCH_DYNAMIC;
import static android.content.pm.LauncherApps.ShortcutQuery.FLAG_MATCH_MANIFEST;
import static android.content.pm.LauncherApps.ShortcutQuery.FLAG_MATCH_PINNED;

import android.content.ComponentName;
import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.LauncherActivityInfo;
import android.content.pm.LauncherApps;
import android.content.pm.PackageManager;
import android.content.pm.ShortcutInfo;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.os.Parcelable;
import android.os.Parcel;
import android.os.Process;
import android.os.UserHandle;

import com.devicenative.dna.utils.DNALogger;
import com.devicenative.dna.utils.DNAUtils;

import org.json.JSONObject;

import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * DNAdUnit class represents an ad unit in the application.
 * It contains all the necessary information about an ad unit.
 */
public class DNAResultItem implements Parcelable {
    static public final String TYPE_APP = "app";
    static public final String TYPE_AD = "ad";
    static public final String TYPE_SHORTCUT = "shortcut";
    static public final String TYPE_NOTIFICATION = "notification";
    static public final String TYPE_CONTACT = "contact";
    static public final String TYPE_SEARCH = "search";

    /**
     * The id of the result, just for convenience. A UUID string.
     */
    public String id;

    /**
     * The package name of the result item. Note that there could be multiple
     * results for the same app and package name, so this will not be unique.
     */
    public String packageName;

    /**
     * The class name of the result item. Can be null!
     */
    public String className;

    /**
     * The user handle associated with this package. Can be null!
     */
    public String userHandle;

    /**
     * The user ID associated with this package. This is used to identify the user handle
     */
    public int uId;

    /**
     * A helpful boolean to indicate whether app is installed.
     */
    public boolean isInstalled;

    /**
     * The type of the result item. It can be app, ad, shortcut, notification, contact, or search.
     * Please reference the static variables at the top if checking this field.
     */
    public String resultType;

    /**
     * The app name of the result item. Note that there could be multiple
     * results for the same app and package name, so this will not be unique.
     */
    public String appName;

    /**
     * The result title for the result to be shown to the user
     */
    public String title;

    /**
     * The result description to be shown to the user. Can be null!
     */
    public String description;

    /**
     * The ad creative icon to be shown to the user. Can be null,
     * in which case you should use the app icon from package manager
     */
    public String iconUrl;

    /**
     * The click URL of the ad unit. This will automatically be
     * fired by the SDK when using the click and route method.
     * This can
     */
    public String clickUrl;

    /**
     * The destination domain of the ad unit clickUrl. It's used to
     * check whether the redirects have completed.
     */
    public String destinationUrl;


    /**
     * The number of ratings for the app from Google Play Store.
     */
    public int ratings;

    /**
     * The rating of the app from Google Play Store.
     */
    public double rating;

    /**
     * The number of downloads for the app from Google Play Store.
     */
    public int downloads;

    /**
     * The play store categrory of the app. Note that this can be null.
     */
    public String category;

    /**
     * The impression URL of the ad unit. This will automatically be
     * fired by the SDK when requesting an ad for display.
     */
    public String impressionUrl;

    public int viewThrough;

    public boolean notPlayStore;

    public String source;

    public String placementTag;

    public DNAResultItem parentResultItem;

    public transient JSONObject statMetadata;

    /**
     * Constructor for DNAResultItem class. Meant for internal only use.
     */
    public DNAResultItem(JSONObject rawItem, String type) {
        switch (type) {
            case TYPE_AD:
                this.id = rawItem.optString("id");
                this.packageName = rawItem.optString("packageName");
                this.userHandle = rawItem.optString("userId");
                this.uId = DNAUtils.readPidFromUserHandleString(this.userHandle);
                this.appName = rawItem.optString("appName");
                this.title = rawItem.optString("title");
                this.description = rawItem.optString("description");
                this.iconUrl = rawItem.optString("iconUrl");
                this.clickUrl = rawItem.optString("clickUrl");
                this.destinationUrl = rawItem.optString("destinationUrl");
                this.impressionUrl = rawItem.optString("impressionUrl");
                this.viewThrough = rawItem.optInt("viewThrough");
                this.notPlayStore = rawItem.optBoolean("notPlayStore");
                this.ratings = rawItem.optInt("ratings");
                this.downloads = rawItem.optInt("downloads");
                this.rating = rawItem.optDouble("rating");
                this.category = rawItem.optString("category");
                break;
            case TYPE_APP:
                this.id = rawItem.optString("id");
                this.packageName = rawItem.optString("packageName");
                this.appName = rawItem.optString("appName");
                this.className = rawItem.optString("component");
                this.userHandle = rawItem.optString("userId");
                this.uId = DNAUtils.readPidFromUserHandleString(this.userHandle);
                this.title = rawItem.optString("appName");
                break;
            case TYPE_SHORTCUT:
                this.id = rawItem.optString("id");
                this.packageName = rawItem.optString("packageName");
                this.appName = rawItem.optString("appName");
                this.title = rawItem.optString("title");
                this.description = rawItem.optString("description");
                this.clickUrl = rawItem.optString("intentUri");
                break;
            case TYPE_NOTIFICATION:
                this.id = rawItem.optString("id");
                this.packageName = rawItem.optString("packageName");
                this.appName = rawItem.optString("appName");
                this.title = rawItem.optString("title");
                this.description = rawItem.optString("description");
                break;
        }
        this.resultType = type;
    }

    protected DNAResultItem(Parcel in) {
        id = in.readString();
        packageName = in.readString();
        userHandle = in.readString();
        uId = in.readInt();
        className = in.readString();
        isInstalled = in.readByte() != 0;
        resultType = in.readString();
        appName = in.readString();
        title = in.readString();
        description = in.readString();
        iconUrl = in.readString();
        clickUrl = in.readString();
        destinationUrl = in.readString();
        ratings = in.readInt();
        rating = in.readDouble();
        downloads = in.readInt();
        category = in.readString();
        impressionUrl = in.readString();
        viewThrough = in.readInt();
        notPlayStore = in.readByte() != 0;
        source = in.readString();
        placementTag = in.readString();
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(id);
        dest.writeString(packageName);
        dest.writeString(userHandle);
        dest.writeInt(uId);
        dest.writeString(className);
        dest.writeByte((byte) (isInstalled ? 1 : 0));
        dest.writeString(resultType);
        dest.writeString(appName);
        dest.writeString(title);
        dest.writeString(description);
        dest.writeString(iconUrl);
        dest.writeString(clickUrl);
        dest.writeString(destinationUrl);
        dest.writeInt(ratings);
        dest.writeDouble(rating);
        dest.writeInt(downloads);
        dest.writeString(category);
        dest.writeString(impressionUrl);
        dest.writeInt(viewThrough);
        dest.writeByte((byte) (notPlayStore ? 1 : 0));
        dest.writeString(source);
        dest.writeString(placementTag);
    }

    @Override
    public int describeContents() {
        return 0;
    }

    public static final Creator<DNAResultItem> CREATOR = new Creator<DNAResultItem>() {
        @Override
        public DNAResultItem createFromParcel(Parcel in) {
            return new DNAResultItem(in);
        }

        @Override
        public DNAResultItem[] newArray(int size) {
            return new DNAResultItem[size];
        }
    };

    public void setParentResultItem(DNAResultItem parentResultItem) {
        this.parentResultItem = parentResultItem;
    }

    public void updateAppInstalledState(Context context) {
        LauncherApps la = context.getSystemService(LauncherApps.class);
        if (la == null) {
            isInstalled = false;
            return;
        }

        UserHandle profile;
        if (userHandle != null && !userHandle.isEmpty()) {
            profile  = DNAUtils.getUserHandleFromFormattedString(userHandle);
        } else {
            profile      = Process.myUserHandle();
            userHandle   = profile.toString();     // keep it numeric
            uId          = DNAUtils.readPidFromUserHandleString(userHandle);
        }

        if (!la.isPackageEnabled(packageName, profile)) {
            isInstalled = false;
            return;
        }

        List<LauncherActivityInfo> acts = la.getActivityList(packageName, profile);
        if (className != null && !className.isEmpty()) {
            // We need a specific component
            for (LauncherActivityInfo act : acts) {
                if (!la.isActivityEnabled(act.getComponentName(), profile)) {
                    continue;
                }
                if (className.equals(act.getName())) {
                    isInstalled = true;
                    return;
                }
            }
            isInstalled = false;   // class not found
        } else {
            if (!acts.isEmpty()) {
                for (LauncherActivityInfo act : acts) {
                    if (!la.isActivityEnabled(act.getComponentName(), profile)) {
                        continue;
                    }
                    isInstalled = true;
                    className   = act.getName();
                    return;
                }
            } else {
                isInstalled = false;
            }
        }
    }

    /**
     * This method returns a string representation of the DNAdUnit object.
     * @return A string representation of the DNAdUnit object.
     */
    @Override
    public String toString() {
        return "DNAResultItem: " + this.id + ", " + this.resultType + ", " + this.packageName + ", " + this.appName + ", " + this.title + ", " + this.description;
    }

    public ComponentName getComponentName() {
        if (packageName == null || className == null) {
            return null;
        }
        return new ComponentName(packageName, className);
    }

    /**
     * Optional: This method loads the icon of the ad unit asynchronously. It is a convenience method
     * in case you don't want to implement handling in your own.
     * @param callback The callback to be invoked when the icon is loaded.
     */
    public void loadCreativeDrawableAsync(Context context, final ImageCallback callback) {
        if (resultType.equals(TYPE_SHORTCUT)) {
            Drawable drawable = getShortcutDrawable(context);
            if (drawable != null) {
                callback.onImageLoaded(drawable);
            } else {
                callback.onError("No icon for this shortcut.");
            }
            return;
        }

        if (iconUrl == null || iconUrl.isEmpty()) {
            callback.onError("No icon URL for this ad unit.");
            return;
        }

        ExecutorService executor = Executors.newSingleThreadExecutor();
        executor.submit(new Runnable() {
            @Override
            public void run() {
                try {
                    Drawable drawable = getDrawableFromURL(iconUrl);
                    callback.onImageLoaded(drawable);
                } catch (Exception e) {
                    callback.onError(e.getMessage());
                } finally {
                    executor.shutdown();
                }
            }
        });
    }

    /**
     * Optional: This method loads the icon of the ad unit synchronously. It is a convenience method
     * in case you don't want to implement handling in your own.
     * @return The drawable of the ad unit icon.
     */
    public Drawable loadCreativeDrawable(Context context) {
        if (resultType.equals(TYPE_SHORTCUT)) {
            return getShortcutDrawable(context);
        }

        if (iconUrl == null || iconUrl.isEmpty()) {
            return null;
        }

        return getDrawableFromURL(iconUrl);
    }

    private Drawable getDrawableFromURL(String src) {
        try {
            URL url = new URL(src);
            HttpURLConnection connection = (HttpURLConnection) url.openConnection();
            connection.setDoInput(true);
            connection.connect();
            InputStream input = connection.getInputStream();
            Bitmap bitmap = BitmapFactory.decodeStream(input);
            return new BitmapDrawable(Resources.getSystem(), bitmap);
        } catch (Exception e) {
            return null;
        }
    }

    private Drawable getShortcutDrawable(Context context) {
        try {
            final LauncherApps launcherApps = (LauncherApps) context.getSystemService(Context.LAUNCHER_APPS_SERVICE);

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N_MR1) {
                if (launcherApps == null || !launcherApps.hasShortcutHostPermission()) {
                    return null;
                }

                LauncherApps.ShortcutQuery query = new LauncherApps.ShortcutQuery();
                query.setPackage(packageName);
                query.setShortcutIds(Collections.singletonList(clickUrl));
                query.setQueryFlags(FLAG_MATCH_DYNAMIC | FLAG_MATCH_MANIFEST | FLAG_MATCH_PINNED);

                List<ShortcutInfo> shortcuts = launcherApps.getShortcuts(query, Process.myUserHandle());
                if (shortcuts != null) {
                    for (ShortcutInfo shortcut : shortcuts) {
                        if (shortcut.isEnabled()) {
                            return launcherApps.getShortcutIconDrawable(shortcut, 0);
                        }
                    }
                }
            }
        } catch (Exception e) {
            DNALogger.i("Error loading shortcut icon: " + e.getMessage());
        }
        return null;
    }

    /**
     * This is an interface definition for a callback to be invoked when the icon is loaded.
     */
    public interface ImageCallback {
        void onImageLoaded(Drawable icon);
        void onError(String message);
    }
}
