package com.devicenative.dna;

import android.app.Activity;
import android.app.AlarmManager;
import android.app.Application;
import android.app.PendingIntent;
import android.app.Service;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.os.Binder;
import android.os.Build;
import android.os.Bundle;
import android.os.IBinder;
import android.util.Log;

import com.devicenative.dna.ads.DNAResultServer;
import com.devicenative.dna.broadcast.DNAProfileChangeReceiver;

import com.devicenative.dna.datasource.DNAAppDataSource;
import com.devicenative.dna.datasource.DNAServerSyncDataSource;
import com.devicenative.dna.datasource.DNAShortcutDataSource;
import com.devicenative.dna.datasource.DNAStatsSyncDataUploader;
import com.devicenative.dna.datasource.DNAUsageStatsDataSource;
import com.devicenative.dna.datasource.loader.DNALoaderUpdateListener;
import com.devicenative.dna.utils.DNAConfigBuilder;
import com.devicenative.dna.utils.DNALogger;
import com.devicenative.dna.utils.DNAPreferences;
import com.devicenative.dna.utils.DNAStatsLogger;
import com.devicenative.dna.utils.DNAWakeScheduleReceiver;

import org.json.JSONObject;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;

public class DNADataOrchestrator extends Service implements DNALoaderUpdateListener, DNAWakeScheduleReceiver.Listener, Application.ActivityLifecycleCallbacks {

    private static final String SERVICE_LOCK_FILE = "DNAServiceLock";
    private static FileLock serviceLock = null;

    private DNAAppDataSource appDataSource;
    private DNAServerSyncDataSource serverSyncDataSource;
    private DNAStatsSyncDataUploader statsSyncDataSource;
    private DNAShortcutDataSource shortcutDataSource;
    private DNAUsageStatsDataSource usageStatsDataSource;

    private static DNAWakeScheduleReceiver wakeScheduleReceiver;

    public boolean isLoading = false;
    public boolean allSourcesHaveLoaded = false;

    private boolean isActivityChangingConfigurations = false;
    private int activityReferences = 0;
    private long start;

    private final IBinder binder = new LocalBinder();

    public class LocalBinder extends Binder {
        public DNADataOrchestrator getService() {
            // Return this instance of the provider so that clients can call public methods
            return DNADataOrchestrator.this;
        }
    }

    public void onCreate() {
        super.onCreate();

        DNALogger.i("DNADataOrchestrator: onCreate called");
        if (!acquireServiceLock()) {
            stopSelf();
        }

        start = System.currentTimeMillis();

        DNAProfileChangeReceiver profileChangedHandler = new DNAProfileChangeReceiver();
        profileChangedHandler.register(this.getApplicationContext());

        activityReferences = 1;
        // Register the activity lifecycle observer
        ((Application) this.getApplicationContext()).registerActivityLifecycleCallbacks(this);

        // Register the wake schedule receiver
        wakeScheduleReceiver = new DNAWakeScheduleReceiver(this);
        IntentFilter filter = new IntentFilter("com.devicenative.dna.WAKE_AND_SYNC");

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
            registerReceiver(wakeScheduleReceiver, filter, Context.RECEIVER_EXPORTED);
        } else {
            registerReceiver(wakeScheduleReceiver, filter);
        }
        // Set the alarm and register the receiver
        configureWakeAndSync();

        // Load the data
        appDataSource = new DNAAppDataSource(this);
        serverSyncDataSource = new DNAServerSyncDataSource(this);
        statsSyncDataSource = new DNAStatsSyncDataUploader(this);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            shortcutDataSource = new DNAShortcutDataSource(this);
        }
        usageStatsDataSource = new DNAUsageStatsDataSource(this);
        loadAllData();

        if (DNAPreferences.getInstance(getApplicationContext()).getUserAgent(getApplicationContext()).isEmpty()) {
            try {
                Intent intent = new Intent(this, DNAConfigBuilder.class);
                startService(intent);
            } catch(Exception e) {
                DNALogger.e("DNADataOrchestrator: Error starting DNAConfigBuilder: " + e.getMessage());
            }
        }
    }

    private void loadAllData() {
        if (isLoading) {
            return;
        }
        isLoading = true;
        DNALogger.i("DNADataOrchestrator: Loading all data");
        DNAPreferences prefs = DNAPreferences.getInstance(this);
        long now = System.currentTimeMillis();
        if (appDataSource != null) {
            appDataSource.loadData();
        }
        if (prefs.areAppsLoaded()) {
            appDataSource.setIsLoaded(true);
            handleProviderLoaded();
            usageStatsDataSource.loadData();
        }
        if (serverSyncDataSource != null) {
            long lastSyncTime = prefs.getLastApiSyncTime();
            long timeSinceLastSync = now - lastSyncTime;
            if (timeSinceLastSync > prefs.getRefreshInterval()) {
                serverSyncDataSource.loadData();
            } else {
                serverSyncDataSource.setIsLoaded(true);
                handleProviderLoaded();
            }
        }
        if (statsSyncDataSource != null) {
            long lastSyncTime = prefs.getLastApiSyncTime();
            long timeSinceLastSync = now - lastSyncTime;
            if (timeSinceLastSync > prefs.getRefreshInterval()) {
                statsSyncDataSource.loadData();
            } else {
                statsSyncDataSource.setIsLoaded(true);
                handleProviderLoaded();
            }
        }
        if (shortcutDataSource != null && !prefs.areShortcutsLoaded()) {
            shortcutDataSource.loadData();
        } else if (prefs.areShortcutsLoaded()) {
            shortcutDataSource.setIsLoaded(true);
            handleProviderLoaded();
        }
    }

    private void configureWakeAndSync() {
        DNALogger.i("DNADataOrchestrator: Configuring next wake and sync");
        Intent intent = new Intent(this.getApplicationContext(), DNADataOrchestrator.class)
                .setAction("com.devicenative.dna.WAKE_AND_SYNC");
        AlarmManager alarmManager = (AlarmManager) this.getApplicationContext().getSystemService(Context.ALARM_SERVICE);

        try {
            PendingIntent alarmIntent = PendingIntent.getService(this.getApplicationContext(), 0, intent,
                    PendingIntent.FLAG_UPDATE_CURRENT | PendingIntent.FLAG_IMMUTABLE);

            // Set the alarm to fire at a specific time interval, inexact
            DNAPreferences prefs = DNAPreferences.getInstance(this);
            long interval = prefs.getLocalRefreshInterval();
            alarmManager.setInexactRepeating(AlarmManager.RTC_WAKEUP, System.currentTimeMillis() + interval, interval, alarmIntent);
        } catch (Exception e) {
            DNALogger.e("DNADataOrchestrator: Error setting alarm: " + e.getMessage());
            DNAPreferences prefs = DNAPreferences.getInstance(this);
            JSONObject statsMetadata = new JSONObject();
            try {
                statsMetadata.put("sId", prefs.getSessionID());
                statsMetadata.put("loc", "configureWakeAndSync");
                statsMetadata.put("m", e.getMessage());
            } catch (Exception je) {
                DNALogger.e("DeviceNativeAds: Unable to create stats metadata");
            }
            DNAStatsLogger.logInternal(this, "Error", null, statsMetadata);
        }
    }

    private void handleProviderLoaded() {
        DNALogger.i("DNADataOrchestrator: Provider loaded triggered");
        if (this.allSourcesHaveLoaded) {
            return;
        }

        if (!appDataSource.isLoaded() ||
                !serverSyncDataSource.isLoaded() ||
                (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O && !shortcutDataSource.isLoaded()) ||
                !usageStatsDataSource.isLoaded()) {
            return;
        }

        long time = System.currentTimeMillis() - start;
        DNALogger.i("DNADataOrchestrator: Time to load all providers: " + time + "ms");

        this.allSourcesHaveLoaded = true;
        this.isLoading = false;

        DNAResultServer adServer = DNAResultServer.getInstance(this);
        adServer.prefetchSuggestions();
    }

    @Override
    public void onBroadcastReceived(Intent intent) {
        if (intent != null) {
            DNALogger.i("DNADataOrchestrator: Wake broadcast received: " + intent.getAction());
        } else {
            DNALogger.i("DNADataOrchestrator: Wake broadcast received: null");
        }
        DNAPreferences prefs = DNAPreferences.getInstance(this);
        long now = System.currentTimeMillis();

        prefs.setLastWakeTime(now);
        start = System.currentTimeMillis();

        if (serverSyncDataSource != null && !this.isLoading) {
            long lastSyncTime = prefs.getLastApiSyncTime();
            long timeSinceLastSync = now - lastSyncTime;
            if (timeSinceLastSync > prefs.getRefreshInterval()) {
                this.isLoading = true;
                this.allSourcesHaveLoaded = false;
                serverSyncDataSource.setIsLoaded(false);
                serverSyncDataSource.loadData();

                if (statsSyncDataSource != null) {
                    statsSyncDataSource.setIsLoaded(false);
                    statsSyncDataSource.loadData();
                }

                if (appDataSource != null) {
                    appDataSource.setIsLoaded(false);
                    appDataSource.loadData();
                }
            } else if (prefs.getAppsNeedReload()) {
                if (appDataSource != null) {
                    appDataSource.setIsLoaded(false);
                    appDataSource.loadData();
                }
            }
        }
        if (usageStatsDataSource != null) {
            long lastUsageSyncTime = prefs.getLastUsageSyncTime();
            long timeSinceLastUsageSync = now - lastUsageSyncTime;
            if (timeSinceLastUsageSync > DNAPreferences.USAGE_SYNC_INTERVAL) {
                this.allSourcesHaveLoaded = false;
                usageStatsDataSource.setIsLoaded(false);
                usageStatsDataSource.loadData();
            }
        }
    }

    @Override
    public void onLoaderUpdate(String loaderName, int updateType) {
        DNALogger.i("DNADataOrchestrator: Loader update triggered: " + loaderName + " " + updateType);
        if (loaderName.equals(serverSyncDataSource.getClass().getSimpleName()) && updateType == DNALoaderUpdateListener.UPDATE_TYPE_FINISH) {
            configureWakeAndSync();
        }

        if (loaderName.equals(appDataSource.getClass().getSimpleName()) && updateType == DNALoaderUpdateListener.UPDATE_TYPE_FINISH) {
            DNAPreferences prefs = DNAPreferences.getInstance(this);
            DNALogger.i("DNADataOrchestrator: Setting apps loaded");
            prefs.setAppsLoaded();
            prefs.setAppsReloaded();

            // now that apps are loaded, load usage stats
            if (usageStatsDataSource != null) {
                usageStatsDataSource.loadData();
            }
        }

        if (updateType == DNALoaderUpdateListener.UPDATE_TYPE_FINISH) {
            this.handleProviderLoaded();
        }
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
//        if (intent != null && "com.devicenative.dna.WAKE_AND_SYNC".equals(intent.getAction())) {
        DNALogger.i("DNADataOrchestrator: onStartCommand with WAKE_AND_SYNC");
        onBroadcastReceived(intent);
//        }
        return START_NOT_STICKY;
    }

    @Override
    public IBinder onBind(Intent intent) {
        return this.binder;
    }

    private boolean acquireServiceLock() {
        File lockFile = new File(getFilesDir(), SERVICE_LOCK_FILE);
        try {
            FileChannel channel = new RandomAccessFile(lockFile, "rw").getChannel();
            serviceLock = channel.tryLock();
            return serviceLock != null;
        } catch (IOException | OverlappingFileLockException e) {
            return false;
        }
    }

    private void releaseServiceLock() {
        if (serviceLock != null) {
            try {
                serviceLock.release();
                serviceLock = null;
            } catch (IOException e) {
                DNALogger.i("DNADataOrchestrator: Error releasing service lock: " + e.getMessage());
            }
        }
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        DNALogger.i("DNADataOrchestrator: onDestroy called");
        unregisterReceiver(wakeScheduleReceiver);
        releaseServiceLock();
    }

    @Override
    public void onActivityCreated(Activity activity, Bundle savedInstanceState) { }

    @Override
    public void onActivityStarted(Activity activity) {
        DNALogger.i("DNADataOrchestrator: Activity started, and reference count is " + activityReferences + " and isChangingConfigurations is " + isActivityChangingConfigurations);
        if (activityReferences == 0 && !isActivityChangingConfigurations && this.allSourcesHaveLoaded) {
            onBroadcastReceived(new Intent("com.devicenative.dna.WAKE_AND_SYNC"));
        }
        activityReferences++;
    }
    @Override
    public void onActivityResumed(Activity activity) { }

    @Override
    public void onActivityPaused(Activity activity) { }

    @Override
    public void onActivityStopped(Activity activity) {
        DNALogger.i("DNADataOrchestrator: Activity stopped, and reference count is " + activityReferences + " and isChangingConfigurations is " + isActivityChangingConfigurations);
        isActivityChangingConfigurations = activity.isChangingConfigurations();
        activityReferences = Math.max(0, activityReferences - 1);
    }
    @Override
    public void onActivitySaveInstanceState(Activity activity, Bundle outState) { }
    @Override
    public void onActivityDestroyed(Activity activity) { }
}

