/*
 * Decompiled with CFR 0.152.
 */
package datadog.trace.util.json;

import datadog.slf4j.Logger;
import datadog.slf4j.LoggerFactory;
import datadog.trace.util.json.JsonPath;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

public class JsonPathParser {
    private static final char OPEN_BRACKET = '[';
    private static final char CLOSE_BRACKET = ']';
    private static final char OPEN_BRACE = '(';
    private static final char ASTERISK = '*';
    private static final char PERIOD = '.';
    private static final char SPACE = ' ';
    private static final char COMMA = ',';
    private static final char SINGLE_QUOTE = '\'';
    private static final char DOUBLE_QUOTE = '\"';
    private static final char ESC = '\\';
    private static final Logger log = LoggerFactory.getLogger(JsonPathParser.class);

    public static List<JsonPath> parseJsonPaths(List<String> rules) {
        if (null == rules || rules.isEmpty()) {
            return Collections.emptyList();
        }
        ArrayList<JsonPath> result = new ArrayList<JsonPath>(rules.size());
        for (String rule : rules) {
            try {
                JsonPath jp = JsonPathParser.parse(rule);
                result.add(jp);
            }
            catch (Exception ex) {
                log.warn("Failed to parse redaction rule '{}'. {}. Skipping rule.", (Object)rule, (Object)ex.getMessage());
            }
        }
        return result;
    }

    public static JsonPath parse(String path) throws ParseError {
        Cursor cur = new Cursor(path);
        if (!path.startsWith("$.") && !path.startsWith("$[")) {
            cur.fail("JsonPath must start with '$.' or '$['");
        }
        JsonPath.Builder builder = JsonPath.Builder.start();
        cur.incPos(1);
        while (cur.isWithinLimits()) {
            if (cur.is('[')) {
                boolean ok = JsonPathParser.tryToParsePropertyInBrackets(cur, builder) || JsonPathParser.tryToParseIndex(cur, builder) || JsonPathParser.tryToParseWildcard(cur, builder);
                if (ok) continue;
                cur.fail("Expecting in brackets a property, an array index, or a wildcard.");
                continue;
            }
            if (cur.is('.')) {
                if (cur.nextIs('.')) {
                    builder.anyDesc();
                    cur.incPos(1);
                } else if (cur.nextIs('[')) {
                    cur.fail("'.' can't go before '['");
                }
                if (!cur.hasMoreCharacters()) {
                    cur.fail("Path must not end with a '.'");
                } else {
                    cur.incPos(1);
                }
                if (!cur.is('.')) continue;
                cur.fail("More than two '.' in a row");
                continue;
            }
            if (cur.is('*')) {
                if (cur.nextIs('*')) {
                    cur.fail("More than one '*' in a row");
                }
                cur.incPos(1);
                builder.anyChild();
                continue;
            }
            JsonPathParser.parseKey(cur, builder);
        }
        return builder.build();
    }

    private static void parseKey(Cursor cur, JsonPath.Builder builder) throws ParseError {
        int start;
        int i = start = cur.position();
        int end = 0;
        while (cur.isWithinLimits(i)) {
            char c = cur.charAt(i);
            if (c == ' ') {
                cur.failAt(i, "No spaces allowed in property names.");
            } else {
                if (c == '.' || c == '[') {
                    end = i;
                    break;
                }
                if (c == '(') {
                    cur.failAt(i, "Expressions are not supported.");
                }
            }
            ++i;
        }
        if (end == 0) {
            end = cur.len();
        }
        cur.setPos(end);
        String property = cur.subSequence(start, end).toString();
        builder.name(property);
    }

    private static boolean tryToParseWildcard(Cursor cur, JsonPath.Builder builder) throws ParseError {
        if (!cur.nextIsIgnoreSpaces(cur.pos, '*')) {
            return false;
        }
        int asteriskPos = cur.findCharSkipSpaces(cur.pos, '*');
        if (!cur.nextIsIgnoreSpaces(asteriskPos, ']')) {
            int offset = asteriskPos + 1;
            cur.failAt(offset, "Expected ']'");
        }
        int closedAt = cur.findCharSkipSpaces(asteriskPos, ']');
        cur.setPos(closedAt + 1);
        builder.anyChild();
        return true;
    }

    private static boolean tryToParseIndex(Cursor cur, JsonPath.Builder builder) throws ParseError {
        int start = cur.position() + 1;
        int end = cur.indexOf(start, ']');
        if (end < 0) {
            return false;
        }
        String expr = cur.subSequence(start, end).toString().trim();
        for (int i = 0; i < expr.length(); ++i) {
            char c = expr.charAt(i);
            if (Character.isDigit(c)) continue;
            return false;
        }
        try {
            int index = Integer.parseInt(expr);
            builder.index(index);
        }
        catch (NumberFormatException ex) {
            cur.fail("Invalid array index. Must be an integer.");
        }
        cur.setPos(end + 1);
        return true;
    }

    private static boolean tryToParsePropertyInBrackets(Cursor cur, JsonPath.Builder builder) throws ParseError {
        int endBracketIndex;
        int startPosition;
        char quote = cur.findNonSpace();
        if (quote != '\'' && quote != '\"') {
            return false;
        }
        String property = null;
        int readPosition = startPosition = cur.position() + 1;
        int endPosition = 0;
        boolean inProperty = false;
        while (cur.isWithinLimits(readPosition)) {
            char c = cur.charAt(readPosition);
            if ('\\' == c) {
                cur.failAt(readPosition, "Escape character is not supported in property name.");
            } else if (c == ',') {
                String message = inProperty ? "Comma is not allowed in property name." : "Multiple properties are not supported.";
                cur.failAt(readPosition, message);
            } else {
                if (c == ']' && !inProperty) break;
                if (c == quote) {
                    if (inProperty) {
                        endPosition = readPosition;
                        property = cur.subSequence(startPosition, endPosition).toString();
                        inProperty = false;
                    } else {
                        startPosition = readPosition + 1;
                        inProperty = true;
                    }
                }
            }
            ++readPosition;
        }
        if (inProperty) {
            cur.fail("Property has not been closed - missing closing " + quote);
        }
        if ((endBracketIndex = cur.findCharSkipSpaces(endPosition, ']')) < 0) {
            cur.fail("Property has not been closed - missing closing ']'");
        }
        cur.setPos(++endBracketIndex);
        builder.name(property);
        return true;
    }

    private static class Cursor {
        private final CharSequence data;
        private int pos;
        private final int endPos;

        public Cursor(CharSequence data) {
            this.data = data;
            this.pos = 0;
            this.endPos = data.length() - 1;
        }

        public int len() {
            return this.endPos + 1;
        }

        public char charAt(int idx) {
            return this.data.charAt(idx);
        }

        public boolean is(char c) {
            return this.data.charAt(this.pos) == c;
        }

        public boolean nextIs(char c) {
            return this.isWithinLimits(this.pos + 1) && this.data.charAt(this.pos + 1) == c;
        }

        public void incPos(int count) {
            this.pos += count;
        }

        public void setPos(int newPos) {
            this.pos = newPos;
        }

        public int position() {
            return this.pos;
        }

        public int findCharSkipSpaces(int startAt, char c) {
            int p;
            int n = p = startAt == this.endPos ? startAt : this.skipSpaces(startAt + 1);
            if (this.charAt(p) == c) {
                return p;
            }
            return -1;
        }

        private int skipSpaces(int p) {
            while (this.isWithinLimits(p + 1) && this.charAt(p) == ' ') {
                ++p;
            }
            return p;
        }

        public int indexOf(int start, char c) {
            int p = start;
            while (this.isWithinLimits(p)) {
                if (this.charAt(p) == c) {
                    return p;
                }
                ++p;
            }
            return -1;
        }

        public boolean nextIsIgnoreSpaces(int start, char c) {
            int p = this.skipSpaces(start + 1);
            return this.isWithinLimits(p) && this.charAt(p) == c;
        }

        public char findNonSpace() {
            return this.findNonSpace(this.pos);
        }

        public char findNonSpace(int start) {
            int p = this.skipSpaces(start + 1);
            if (this.isWithinLimits(p)) {
                return this.charAt(p);
            }
            return ' ';
        }

        private boolean isWithinLimits() {
            return this.pos <= this.endPos;
        }

        public boolean hasMoreCharacters() {
            return this.isWithinLimits(this.pos + 1);
        }

        public boolean isWithinLimits(int p) {
            return p >= 0 && p <= this.endPos;
        }

        public CharSequence subSequence(int start, int end) {
            return this.data.subSequence(start, end);
        }

        private void fail(String message) throws ParseError {
            this.failAt(this.pos, message);
        }

        private void failAt(int position, String message) throws ParseError {
            throw new ParseError(this.data, position, message);
        }
    }

    public static final class ParseError
    extends Exception {
        public final int position;
        public final String error;

        public ParseError(CharSequence path, int position, String error) {
            super("Failed to parse JsonPath \"" + path + "\" at " + position + ". " + error);
            this.position = position;
            this.error = error;
        }
    }
}

