package com.dada.smart.user.visitor;

import android.support.annotation.NonNull;

import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * view遍历的任务执行器，确保单个 ViewVisitor 中同时只有1一个遍历任务被执行，
 * 同时该ViewVisitor有新的遍历任务出现时，会尽快结束老任务
 */
public class VisitorExecutor {
    private static ThreadPoolExecutor poolExecutor = new ThreadPoolExecutor(1, 1, 0, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(1), new ThreadPoolExecutor.DiscardOldestPolicy());
    private static VisitorExecutor visitorExecutor = new VisitorExecutor();

    public static VisitorExecutor getInstance() {
        return visitorExecutor;
    }

    private TaskInfo taskInfo;

    public void execute(ViewVisitor viewVisitor, @NonNull Task task) {
        TaskInfo newTaskInfo = new TaskInfo(viewVisitor, task);
        cancelTaskIfNeed(newTaskInfo);
        poolExecutor.execute(newTaskInfo);
    }

    /**
     * 确保新的view遍历触发时，能及时结束旧遍历，尽快开始新遍历
     *
     * @param newTaskInfo
     */
    private void cancelTaskIfNeed(TaskInfo newTaskInfo) {
        if (taskInfo != null && taskInfo.viewVisitor.createTime <= newTaskInfo.viewVisitor.createTime) {
            taskInfo.isCancelled = true;
        }
        taskInfo = newTaskInfo;
    }

    public interface Task {
        void run(TaskInfo taskInfo);
    }

    public class TaskInfo implements Runnable {
        ViewVisitor viewVisitor;
        boolean isCancelled;
        Task task;

        public TaskInfo(@NonNull ViewVisitor viewVisitor, @NonNull Task task) {
            this.viewVisitor = viewVisitor;
            this.task = task;
        }

        @Override
        public void run() {
            task.run(this);
        }
    }
}
