package com.dada.smart.user.event;

import android.content.SharedPreferences;
import android.support.annotation.NonNull;
import android.support.annotation.VisibleForTesting;

import com.dada.smart.common.Utils;
import com.dada.smart.user.BuildConfig;
import com.dada.smart.user.http.Client;
import com.dada.smart.user.log.Log;
import com.dada.smart.user.log.LogDao;
import com.dada.smart.user.util.DataUtil;
import com.orhanobut.logger.Logger;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * Created by tomkeyzhang on 17/5/18.
 * 获取需要监控的事件信息，记录及发送事件日志
 */
public class EventLogRepository {
    private final int CODE_OK = 0;
    private final String PREF_APP_START_TIME = "app_start_time";
    //单一线程池
    private Executor singleExecutor;
    //普通线程池
    private Executor commonExecutor;
    private Client client;
    private LogDao logDao;
    private String server;
    private Map<String, Events> eventMap;
    private Map<String, List<Event>> pvEventMap;

    private RequestParam requestParam;

    public interface RequestParam {
        Map<String, Object> params();
    }

    private EventLogRepository(String server) {
        this.server = server;
        this.eventMap = new HashMap<>();
        this.pvEventMap = new HashMap<>();
    }

    /**
     * 触发事件
     *
     * @param eventId
     * @param typeId
     * @param refPageName
     */
    public void onEvent(final long eventId, final long typeId, final String refPageName, final String extra) {
        singleExecutor.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    logDao.insert(new Log(eventId, System.currentTimeMillis(), typeId, refPageName, extra));
                    Logger.v("onEvent:" + typeId + "-" + eventId + "-" + extra);
                } catch (Exception e) {
                    //CursorWindowAllocationException无法声明
                    //IllegalStateException
                    //SQLiteDatabaseCorruptException
                    e.printStackTrace();
                }
            }
        });
    }

    public void sendAppStartEventIfNeed(SharedPreferences preferences, long currentMills, long appId) {
        long appFirstStartMillis = preferences.getLong(PREF_APP_START_TIME, 0L);
        //        Logger.v("AppStart Log appFirstStartMillis="+appFirstStartMillis+" currentMills="+currentMills);
        if (!DataUtil.isTheSameDay(appFirstStartMillis, currentMills)) {
            onEvent(0, Event.TYPE_APP_START, "", "");
            sendEvents(appId);
            preferences.edit().putLong(PREF_APP_START_TIME, currentMills).apply();
            //            Logger.v("AppStart Log");
        }
    }

    /**
     * 发送事件数据
     *
     * @param appId
     */
    public void sendEvents(final long appId) {
        singleExecutor.execute(new Runnable() {
            @Override
            public void run() {
                synchronized (EventLogRepository.this) {
                    try {
                        List<Log> logs = logDao.get(20);
                        if (logs.isEmpty())
                            return;
                        Map<String, Object> map = new HashMap<>();
                        if (requestParam != null)
                            map.putAll(requestParam.params());
                        map.put("appId", appId);
                        map.put("events", convertLog(logs));
                        String url = server + "/event/log/upload";
                        Client.Call call = client.makeCall(Client.Method.POST, url, map);

                        JSONObject object = new JSONObject(call.execute());
                        if (CODE_OK == object.optLong("code")) {
                            logDao.delete(logs);
                            Logger.v("delete logs:" + logs.size());
                        }
                    } catch (JSONException e) {
                        e.printStackTrace();
                    } catch (RuntimeException e) {
                        //CursorWindowAllocationException无法声明
                        //IllegalStateException
                        //SQLiteDatabaseCorruptException
                        e.printStackTrace();
                    }
                }

            }
        });
    }

    /**
     * 获取事件配置
     *
     * @param appId
     */
    public void fetchConfig(final long appId) {
        commonExecutor.execute(new Runnable() {
            @Override
            public void run() {
                Map<String, Object> param = new HashMap<>();
                param.put("appId", appId);
                param.put("sdk", BuildConfig.VERSION);
                String url = server + "/event/config/list";
                Client.Call call = client.makeCall(Client.Method.GET, url, param);
                fetchActivityEvents(call.cache());
                fetchActivityEvents(call.execute());
            }
        });
    }


    /**
     * 获取指定界面的所有非pv事件
     *
     * @param activityName
     * @return
     */
    public Events getEvents(String activityName) {
        return eventMap.get(activityName);
    }

    /**
     * 获取指定界面的所有pv事件
     *
     * @param activityName
     * @return
     */
    public List<Event> getPvEvents(String activityName) {
        return pvEventMap.get(activityName);
    }

    public boolean hasEvent(String activityName) {
        return (eventMap.containsKey(activityName)  || pvEventMap.containsKey(activityName));
    }

    private JSONArray convertLog(List<Log> logs) throws JSONException {
        JSONArray array = new JSONArray();
        for (int i = 0; i < logs.size(); i++) {
            JSONObject object = new JSONObject();
            Log log = logs.get(i);
            object.put("id", log.getEventId());
            object.put("typeId", log.getEventTypeId());
            object.put("refPageIdentifier", log.getRefPageIdentifier());
            object.put("createTime", log.getTriggerTime());
            object.put("extra", log.getExtra());
            array.put(object);
        }
        return array;
    }

    /**
     * 解析出所有事件
     */
    private void fetchActivityEvents(String result) {
        //        Logger.d(result);
        if (Utils.isEmpty(result))
            return;
        try {
            JSONObject object = new JSONObject(result);
            if (CODE_OK == object.optLong("code", -1)) {
                JSONArray array = object.getJSONArray("body");
                for (int i = 0; i < array.length(); i++) {
                    JSONObject activity = array.getJSONObject(i);
                    String pageIdentifier = activity.optString("pageIdentifier");
                    JSONArray eventArray = activity.getJSONArray("events");
                    List<Event> events = new ArrayList<>();
                    List<Event> pvEvents = new ArrayList<>();
                    boolean inMainWindow = true;
                    for (int j = 0; j < eventArray.length(); j++) {
                        Event event = Event.fromJson(eventArray.getJSONObject(j));
                        if (event.isActivityEvent()) {//== equals
                            pvEvents.add(event);
                        } else if (event.isViewEvent()) {
                            //只要有一个不在主window就不再变更值
                            if (inMainWindow) {
                                inMainWindow = event.isInMainWindow();
                            }
                            events.add(event);
                        }
                    }
                    if (!events.isEmpty()) {
                        eventMap.put(pageIdentifier, new Events(inMainWindow, events));
                    }
                    if (!pvEvents.isEmpty()) {
                        pvEventMap.put(pageIdentifier, pvEvents);
                    }
                }
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }
    }

    @VisibleForTesting
    public Map<String, Events> getEventMap() {
        return eventMap;
    }

    @VisibleForTesting
    public Map<String, List<Event>> getPvEventMap() {
        return pvEventMap;
    }

    public static class Builder {
        private EventLogRepository repository;

        public Builder(@NonNull String server) {
            repository = new EventLogRepository(server);
        }

        public Builder client(Client client) {
            repository.client = client;
            return this;
        }

        public Builder logDao(LogDao logDao) {
            repository.logDao = logDao;
            return this;
        }

        public Builder requestParam(RequestParam requestParam) {
            repository.requestParam = requestParam;
            return this;
        }

        public Builder singleExecutor(Executor executor) {
            repository.singleExecutor = executor;
            return this;
        }

        public Builder commonExecutor(Executor executor) {
            repository.commonExecutor = executor;
            return this;
        }


        public EventLogRepository build() {
            if (repository.singleExecutor == null) {
                repository.singleExecutor = Executors.newSingleThreadExecutor();
            }
            if (repository.commonExecutor == null) {
                repository.commonExecutor = new ThreadPoolExecutor(2, 20, 20L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(64), new ThreadPoolExecutor.DiscardOldestPolicy());
            }
            if (repository.client == null)
                throw new RuntimeException("client must be set!");

            if (repository.logDao == null)
                throw new RuntimeException("logDao must be set!");

            return repository;
        }
    }

}
