package com.dada.smart.common;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Pattern;

/**
 * Smart-Expression 表达式语言解析器
 */
public class SmartExpression {
    private static final Pattern varNamePattern = Pattern.compile("\\p{Lower}\\w*");
    private static final String UNKNOWN = "unknown";
    public static final String CONSTRUCTOR_METHOD = "constructor";
    public static final String TYPE_ASSIGN = "assign";
    public static final String TYPE_EQUAL = "equal";

    private final static Pattern digitPattern = Pattern.compile("^[0-9]*$");
    private String[] expressions;

    public SmartExpression(String expressions) {
        this.expressions = expressions.replaceAll("\n", "").split(";");
    }

    private Result getResult(Object base, String expression) {
        Result result = new Result(expression);
        if (result.isParseOk())
            valueGet(base, result);
        return result;
    }

    private void valueGet(Object base, Result result) {
        try {
            Object value = base;
            //enable=<android.support.v4.app.NotificationManagerCompat>from(this).areNotificationsEnabled()

            for (String exp : result.right.split("\\.")) {
                if (exp.contains("(") && exp.contains(")")) {
                    value = methodGet(value, result.getBaseClassName(), exp);
                } else {
                    value = propertyGet(value, result.getBaseClassName(), exp);
                }
            }
            result.success = true;
            result.value = value.toString();
        } catch (Throwable e) {
            e.printStackTrace();
            result.success = false;
            result.value = e.getMessage();
        }
        //        System.out.println(result);
    }

    public List<Result> getResult(Object base) {
        List<Result> results = new ArrayList<>();
        for (String expression : expressions) {
            if (!Utils.isEmpty(expression) && base != null)
                results.add(getResult(base, expression));
        }
        return results;
    }


    private Object propertyGet(Object base, String className, String property) {
        ReflectUtils reflectUtils;
        if (!Utils.isEmpty(className)) {//自定义执行的目标对象
            reflectUtils = ReflectUtils.reflect(className);
        } else {//在当前base对象中执行
            reflectUtils = ReflectUtils.reflect(base);
        }
        return reflectUtils.field(property).get();
    }

    private Object methodGet(Object base, String className, String method) {
        String[] array = method.split("\\(");
        String methodName = array[0];
        Object[] args = convertParam(base, array[1].replaceAll("\\)", ""));

        ReflectUtils reflectUtils;
        if (!Utils.isEmpty(className)) {//自定义执行的目标对象
            reflectUtils = ReflectUtils.reflect(className);
        } else {//在当前base对象中执行
            reflectUtils = ReflectUtils.reflect(base);
        }
        //执行构造方法创建对象
        if (CONSTRUCTOR_METHOD.equals(methodName)) {
            return reflectUtils.newInstance(args).get();
        } else {
            return reflectUtils.method(methodName, args).get();
        }
    }

    //<android.support.v4.app.NotificationManagerCompat>from(this).areNotificationsEnabled()
    //    private Object methodGet(Object base, String className, String method) {
    //        return ReflectUtils.reflect(className).method().get();
    //    }

    private Object[] convertParam(Object base, String paramsString) {
        if (Utils.isEmpty(paramsString))
            return new Object[0];
        String[] params = paramsString.split(",");
        //        System.out.println(params.length);
        Object[] args = new Object[params.length];
        for (int i = 0; i < params.length; i++) {
            String param = params[i];
            //如果是数字则转化为整数，其他都认为是字符串类型
            if (digitPattern.matcher(param).matches()) {
                args[i] = Integer.parseInt(param);
            } else if ("this".equals(param)) {
                args[i] = base;
            } else {
                args[i] = param;
            }
            //            System.out.println(args[i]+"-"+args[i].getClass());
        }
        return args;
    }

    public static List<Result> fetchResult(Object base, String expressions) {
        if (Utils.isEmpty(expressions) || base == null)
            return new ArrayList<>();
        return new SmartExpression(expressions).getResult(base);
    }

    public static JSONArray fetchAsArray(Object base, String expressions) {
        List<SmartExpression.Result> results = SmartExpression.fetchResult(base, expressions);
        JSONArray array = new JSONArray();
        for (SmartExpression.Result result : results) {
            JSONObject object = new JSONObject();
            try {
                object.put("expression", result.getExpression());
                object.put("success", result.isSuccess());
                if (result.isTypeEqual()) {
                    object.put("value", result.isSuccess() ? String.valueOf(result.getLeft().equals(result.getValue())) : result.getValue());
                } else if (result.isTypeAssign()) {
                    object.put("value", result.getValue());
                }
            } catch (JSONException e) {
                e.printStackTrace();
            }
            array.put(object);
        }
        return array;
    }

    public static class Result {

        private String expression;
        private String left = UNKNOWN;
        private String right = UNKNOWN;
        private String type = UNKNOWN;
        private String value;
        private String baseClassName;
        private boolean success = false;

        private Result(String expression) {
            this.expression = expression;
            try {
                String[] array;
                if (expression.contains("==")) {
                    array = expression.split("==");
                    type = TYPE_EQUAL;
                } else {
                    array = expression.split("=");
                    type = TYPE_ASSIGN;
                    if (!varNamePattern.matcher(array[0]).matches()) {
                        value = "左侧变量名称不合法";
                        return;
                    }
                }
                this.left = array[0];
                this.right = array[1];
                //enable=<android.support.v4.app.NotificationManagerCompat>from(this).areNotificationsEnabled()
                if (right.startsWith("<") && right.contains(">")) {
                    array = right.replaceFirst("<", "").split(">");
                    baseClassName = array[0];
                    right = array[1];
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        public boolean isParseOk() {
            return !UNKNOWN.equals(left) && !UNKNOWN.equals(right) && !UNKNOWN.equals(type);
        }


        //        public Result(String left, String right) {
        //            this.left = left;
        //            this.right = right;
        //        }
        //
        //        public Result(String left, String right, String type) {
        //            this.left = left;
        //            this.right = right;
        //            this.type = type;
        //        }


        public String getBaseClassName() {
            String clzz = baseClassName;
            baseClassName = null;
            return clzz;
        }

        public boolean isSuccess() {
            return success;
        }

        public String getLeft() {
            return left;
        }

        public void setLeft(String left) {
            this.left = left;
        }

        public String getRight() {
            return right;
        }

        public void setRight(String right) {
            this.right = right;
        }

        public String getType() {
            return type;
        }

        public void setType(String type) {
            this.type = type;
        }

        public String getValue() {
            return value;
        }

        public void setValue(String value) {
            this.value = value;
        }

        public String getExpression() {
            return expression;
        }

        public boolean isLeftEqualsValue() {
            return left != null && value != null && TYPE_EQUAL.equals(type) && left.equals(value);
        }

        public boolean isTypeEqual() {
            return TYPE_EQUAL.equals(type);
        }

        public boolean isTypeAssign() {
            return TYPE_ASSIGN.equals(type);
        }

        @Override
        public String toString() {
            //            if (TYPE_ASSIGN.equals(type)) {
            //                return left + "=" + right;
            //            } else if (TYPE_EQUAL.equals(type)) {
            //                return left + "==" + right;
            //            }
            return expression + "," + value;
        }

    }
}
