package com.dada.response.watcher.watcher;


import android.content.Context;
import android.content.SharedPreferences;
import android.text.TextUtils;

import com.dada.monitor.network.MonitorNetworkUtil;
import com.dada.monitor.network.http.pojo.DaDaResponseBody;
import com.dada.monitor.network.util.Json;
import com.dada.response.watcher.ResponseWatchUtil;
import com.dada.response.watcher.entity.NetStatus;
import com.dada.response.watcher.entity.NetStatusCache;
import com.dada.response.watcher.inter.TagInterface;
import com.dada.response.watcher.util.ArrayUtils;
import com.dada.response.watcher.util.ResponseWatchDebug;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.locks.ReentrantLock;

/**
 * Created by whh on 2019/5/31
 */
public class ResponseHelper {

    public static final String TAG = "ResponseHelper";
    private static final String SPF_RESPONSE_REPOSITORY = "spf_response_repository";
    private static final String SPF_NETSTATUS = "netstatus";


    private static ResponseHelper instance;
    private NetStatusCache netStatus;
    private SharedPreferences netStatusPre;

    private Executor executor = Executors.newSingleThreadExecutor();
    private TagInterface tagInterface;
    private final ReentrantLock lock = new ReentrantLock();

    private static MonitorNetworkUtil networkUtil;

    public static ResponseHelper getInstance() {
        if (instance == null) {
            instance = new ResponseHelper();
        }
        return instance;
    }

    /**
     * 获取 MonitorNetworkUtil 对象
     */
    public MonitorNetworkUtil getNetworkUtil() {
        if (null == networkUtil) {
            networkUtil = MonitorNetworkUtil.getInstance();
        }
        return networkUtil;
    }

    private ResponseHelper() {
        netStatusPre = ResponseWatchUtil.getContext().getSharedPreferences(SPF_RESPONSE_REPOSITORY, Context.MODE_PRIVATE);
        String netStatusStr = netStatusPre.getString(SPF_NETSTATUS, "");
        if (!TextUtils.isEmpty(netStatusStr)) {
            //同步数据
            try {
                netStatus = Json.fromJson(netStatusStr, NetStatusCache.class);
            } catch (Exception e) {
                //ignore
            }
            if (null == netStatus) {
                netStatus = new NetStatusCache();
            }
            ResponseWatchDebug.d(TAG, "netStatus  " + netStatus.toString());
        } else {
            ResponseWatchDebug.d(TAG, "SharedPreferences new NetStatusCache");
            netStatus = new NetStatusCache();
        }
    }

    public synchronized void addWatchedData(String domain, String uri, String status, long tookMs) {
        ResponseWatchDebug.d(TAG, "addWatchedData  domain: " + domain + " uri: " + uri + " status: " + status + " ms: " + tookMs);
        //使用压缩的数据类型，不能对 netStatus 加锁，因为在上传的时候需要对 netStatus 仍然是可以进行写入的
        final Map<String, NetStatusCache.Fields> statusMap = netStatus.statusMap;
        String key = String.format(Locale.CHINA, "%s;%s;%s", domain, uri, status);

        if (statusMap.containsKey(key)) {
            NetStatusCache.Fields fields = statusMap.get(key);
            if (null != fields) {
                long count = fields.getCount();
                long sum = fields.getSum();
                statusMap.put(key, fields.setCount(++count).setSum(sum + tookMs));
                syncSharedPreferences();
            }
        } else {
            statusMap.put(key, new NetStatusCache.Fields().setCount(1L).setSum(tookMs));
            syncSharedPreferences();
        }
    }

    void saveAndUpload() {
        try {
            sendResponseLog();
        } catch (Exception e) {
            e.printStackTrace();
            ResponseWatchDebug.d(TAG, "发送失败: " + e.getMessage());
        }
    }

    private void sendResponseLog() throws Exception {
        lock.lock();
        NetStatus netStatusToUpload = unZipNetStatus();
        ResponseWatchDebug.d(TAG, "转换完类型");
        if (null == netStatusToUpload || ArrayUtils.isEmpty(netStatusToUpload.getMetrics()))
            return;

        //备份
        NetStatusCache netStatusCopy = (NetStatusCache) netStatus.clone();
        ResponseWatchDebug.d(TAG, "备份的netStatus ： " + netStatusCopy.toString());
        lock.unlock();

        ResponseWatchDebug.d(TAG, "真正开始发送 netStatusToUpload " + netStatusToUpload.toString());
        //发送网络请求的时候不能进行加锁，不然会导致发送的时候其他的监听无法监听，进行拥堵
        //有个问题，如果此网络请求很慢，拿到的 copy 不对,会导致重置的时候有两个重置，重置的结果不对
        DaDaResponseBody responseBody = getNetworkUtil().postUrlsyn("api/app/metrics/collect", Json.toJson(netStatusToUpload));

        //测试代码,测试当 sleep 的时候，继续发送请求，然后接口返回 ok 的时候是否会重置之前的
        //        Thread.sleep(5000);

        if (null != responseBody && responseBody.isOk()) {
            //清空
            resetNetStatusCache(netStatusCopy);
        } else {
            if (null != responseBody) {
                ResponseWatchDebug.d(TAG, "失败 code " + responseBody.getErrorCode() + " message" + responseBody.getErrorMsg());
            } else {
                ResponseWatchDebug.d(TAG, "网络不通");
            }
        }
    }

    public void setTagInterface(TagInterface tagInterface) {
        this.tagInterface = tagInterface;
    }

    //解压 NetStatus
    private NetStatus unZipNetStatus() {
        //使用压缩的数据类型，不能对 netStatus 加锁，因为在上传的时候需要对 netStatus 仍然是可以进行写入的
        final Map<String, NetStatusCache.Fields> statusMap = netStatus.statusMap;
        if (statusMap.isEmpty())
            return null;

        NetStatus.Tag tag = new NetStatus.Tag()
                .setApp(tagInterface.getAppName())
                .setCity(tagInterface.getCityCode())
                .setDeviceId(tagInterface.getDeviceId())
                .setNetwork(tagInterface.getNetWork())
                .setOperator(tagInterface.getOperator());

        List<NetStatus.Metric> metrics = new ArrayList<>();
        for (String key : statusMap.keySet()) {
            NetStatusCache.Fields field = statusMap.get(key);
            if (null == field || 0L == field.getCount())
                continue;

            String[] keyValue = key.split(";");
            if (keyValue.length < 3)
                continue;

            String domain = keyValue[0];
            String uri = keyValue[1];
            String status = keyValue[2];

            long count = field.getCount();
            long sum = field.getSum();
            NetStatus.Metric.Fields fields = new NetStatus.Metric.Fields()
                    .setCount(field.getCount())
                    .setSum(field.getSum());

            fields.setMean((int) ((double) sum / (double) count));

            NetStatus.Metric.Tag tags = new NetStatus.Metric.Tag()
                    .setDomain(domain)
                    .setStatus(status)
                    .setUri(uri);
            NetStatus.Metric metric = new NetStatus.Metric()
                    .setFields(fields)
                    .setTags(tags);
            metrics.add(metric);
        }

        return new NetStatus()
                .setTime(System.currentTimeMillis())
                .setTags(tag)
                .setMetrics(metrics);
    }

    /**
     * 使用 netStatusCopy 重置内存中的 netStatus
     */
    private synchronized void resetNetStatusCache(NetStatusCache netStatusCopy) {
        if (null == netStatusCopy || netStatusCopy.statusMap.isEmpty()) {
            return;
        }

        ResponseWatchDebug.d(TAG, "重置：备份的 netStatusCopy： " + netStatusCopy.toString());
        ResponseWatchDebug.d(TAG, "重置：内存中的 netStatus： " + netStatus.toString());
        Map<String, NetStatusCache.Fields> netStatusZipsCopy = netStatusCopy.statusMap;
        Map<String, NetStatusCache.Fields> netStatusZipsCache = netStatus.statusMap;

        for (Map.Entry<String, NetStatusCache.Fields> entryCopy : netStatusZipsCopy.entrySet()) {
            for (Map.Entry<String, NetStatusCache.Fields> entry : netStatusZipsCache.entrySet()) {
                String keyCopy = entryCopy.getKey();
                String key = entry.getKey();
                NetStatusCache.Fields fieldsCopy = entryCopy.getValue();
                NetStatusCache.Fields fields = entry.getValue();
                if (null == keyCopy || null == key || null == fieldsCopy || null == fields)
                    return;

                //相同的key，重置 count 和 sum 的值
                if (keyCopy.equals(key)) {
                    long resetCount = fields.getCount() - fieldsCopy.getCount();
                    long resetSum = fields.getSum() - fieldsCopy.getSum();
                    entry.setValue(new NetStatusCache.Fields().setCount(resetCount).setSum(resetSum));
                }
            }
        }

        syncSharedPreferences();
    }

    /**
     * 将内存中的 netStatus 同步到缓存中
     * 缓存时机：netStatus 改变的时候，也就是所有 netStatus 修改的时候进行同步
     */
    private synchronized void syncSharedPreferences() {
        executor.execute(() -> {
            try {
                netStatusPre.edit().putString(SPF_NETSTATUS, Json.toJson(netStatus)).apply();
            } catch (Exception e) {
                ResponseWatchDebug.d(TAG, "SharedPreferences save exception: " + e.getMessage());
            }
        });
    }
}
