//
// Prediction.java
//
// Copyright (c) 2019 Couchbase, Inc.  All rights reserved.
//
// Licensed under the Couchbase License Agreement (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
// https://info.couchbase.com/rs/302-GJY-034/images/2017-10-30_License_Agreement.pdf
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//

package com.couchbase.lite;

import android.support.annotation.NonNull;

import com.couchbase.lite.internal.support.Log;
import com.couchbase.litecore.C4Database;
import com.couchbase.litecore.C4Prediction;
import com.couchbase.litecore.C4PredictiveModel;
import com.couchbase.litecore.LiteCoreException;
import com.couchbase.litecore.fleece.FLEncoder;
import com.couchbase.litecore.fleece.FLSliceResult;
import com.couchbase.litecore.fleece.FLValue;
import com.couchbase.litecore.fleece.MRoot;

import java.util.HashMap;
import java.util.Map;

/**
 * <b>ENTERPRISE EDITION API : UNCOMMITTED</b><br/><br/>
 * <p>
 * The prediction model manager for registering and unregistering predictive models.
 */
public final class Prediction {

    //---------------------------------------------
    // Member variables
    //---------------------------------------------

    private Map<String, Object> models;

    //---------------------------------------------
    // API - public methods
    //---------------------------------------------

    /**
     * Register a predictive model by the given name.
     * @param name The name of the predictive model.
     * @param model The predictive model.
     */
    public synchronized void registerModel(@NonNull String name, @NonNull PredictiveModel model) {
        if (name == null)
            throw new IllegalArgumentException("name cannot be null.");
        if (model == null)
            throw new IllegalArgumentException("model cannot be null.");

        if (models == null)
             models = new HashMap<String, Object>();

        if (models.containsKey(name))
            C4Prediction.unregister(name);

        C4PredictiveModel c4Model = new C4PredictiveModelImpl(model);
        C4Prediction.register(name, c4Model);
        models.put(name, c4Model);

    }

    /**
     * Unregister the predictive model of the given name.
     * @param name The name of the predictive model.
     */
    public synchronized void unregisterModel(@NonNull String name) {
        if (name == null)
            throw new IllegalArgumentException("name cannot be null.");

        if (models == null)
            return;

        C4Prediction.unregister(name);
        models.remove(name);
    }

    //---------------------------------------------
    // Package level access
    //---------------------------------------------

    Prediction() { }

    //---------------------------------------------
    // Private level access
    //---------------------------------------------

    private class C4PredictiveModelImpl implements C4PredictiveModel {

        private PredictiveModel model;

        C4PredictiveModelImpl(PredictiveModel model) {
            this.model = model;
        }

        @Override
        public long predict(long input, long c4db) {
            DocContext context = new DocContext(new Database(new C4Database(c4db).retain()), null);
            MRoot root = new MRoot(context, new FLValue(input), false);
            Dictionary dict = (Dictionary) root.asNative();
            Dictionary output = model.predict(dict);
            return encode(output).getHandle();
        }

        private FLSliceResult encode(Dictionary prediction) {
            if (prediction != null) {
                FLEncoder encoder = new FLEncoder();
                prediction.encodeTo(encoder);
                try {
                    return encoder.finish2().retain();
                } catch (LiteCoreException e) {
                    Log.e(LogDomain.QUERY, "Error when encoding a predictive result", e);
                }
            }
            return new FLSliceResult().retain();
        }

    }

}
