//
// MessageEndpointListener.java
//
// Copyright (c) 2018 Couchbase, Inc.  All rights reserved.
//
// Licensed under the Couchbase License Agreement (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
// https://info.couchbase.com/rs/302-GJY-034/images/2017-10-30_License_Agreement.pdf
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
package com.couchbase.lite;

import android.support.annotation.NonNull;

import com.couchbase.lite.internal.support.Log;
import com.couchbase.litecore.C4DocumentEnded;
import com.couchbase.litecore.C4Replicator;
import com.couchbase.litecore.C4ReplicatorListener;
import com.couchbase.litecore.C4ReplicatorStatus;
import com.couchbase.litecore.LiteCoreException;
import com.couchbase.litecore.fleece.FLEncoder;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;

import static com.couchbase.litecore.C4ReplicatorMode.kC4Passive;
import static com.couchbase.litecore.C4ReplicatorStatus.C4ReplicatorActivityLevel.kC4Connecting;
import static com.couchbase.litecore.C4ReplicatorStatus.C4ReplicatorActivityLevel.kC4Stopped;
import static com.couchbase.litecore.C4Socket.kC4ReplicatorOptionNoIncomingConflicts;

/**
 * MessageEndpointListener to serve incoming message endpoint connection.
 */
public class MessageEndpointListener {
    private static final LogDomain DOMAIN = LogDomain.NETWORK;

    //---------------------------------------------
    // member variables
    //---------------------------------------------

    private Object lock = new Object();
    private MessageEndpointListenerConfiguration config;
    private ScheduledExecutorService handler;
    private Map<C4Replicator, MessageEndpointConnection> replicators;
    private ChangeNotifier<MessageEndpointListenerChange> changeNotifier;

    public MessageEndpointListener(@NonNull MessageEndpointListenerConfiguration config) {
        if (config == null)
            throw new IllegalArgumentException("config cannot be null.");
        this.config = config;
        this.replicators = new HashMap<>();
        this.changeNotifier = new ChangeNotifier<MessageEndpointListenerChange>();
        this.handler = Executors.newSingleThreadScheduledExecutor(new ThreadFactory() {
            @Override
            public Thread newThread(Runnable target) {
                return new Thread(target, "MessageEndpointListenerThread");
            }
        });
    }

    /**
     * The active connections from peers.
     *
     * @return
     */
    List<MessageEndpointConnection> getConnections() {
        synchronized (lock) {
            return new ArrayList<>(replicators.values());
        }
    }

    /**
     * Accept a new connection.
     *
     * @param connection
     */
    public void accept(@NonNull MessageEndpointConnection connection) {
        if (connection == null)
            throw new IllegalArgumentException("connection cannot be null.");

        MessageSocket socket = new MessageSocket(connection, config.getProtocolType());

        FLEncoder encoder = new FLEncoder();
        encoder.beginDict(1);
        encoder.writeKey(kC4ReplicatorOptionNoIncomingConflicts);
        encoder.writeValue(true);
        encoder.endDict();
        byte[] options = null;
        try {
            options = encoder.finish();
        } catch (LiteCoreException e) {
            Log.e(DOMAIN, "Failed to encode", e);
        } finally {
            encoder.free();
        }

        C4ReplicatorListener statusListener = new C4ReplicatorListener() {
            @Override
            public void statusChanged(
                    final C4Replicator repl,
                    final C4ReplicatorStatus status,
                    final Object context) {
                final MessageEndpointListener listener = (MessageEndpointListener) context;
                handler.execute(new Runnable() {
                    @Override
                    public void run() {
                        listener.statusChanged(repl, status);
                    }
                });
            }

            @Override
            public void documentEnded(
                    final C4Replicator repl,
                    final boolean isPush,
                    final C4DocumentEnded[] document,
                    final Object context) { /* Not used */ }
        };


        C4Replicator replicator = null;
        C4ReplicatorStatus status;
        synchronized (config.getDatabase().getLock()) {
            try {
                replicator = config.getDatabase().getC4Database().createReplicator(
                        socket, kC4Passive, kC4Passive, options, statusListener, this);
                status = new C4ReplicatorStatus(kC4Connecting);
            } catch (LiteCoreException e) {
                status = new C4ReplicatorStatus(kC4Stopped, e.domain, e.code);
            }
        }

        if (replicator != null) {
            synchronized (lock) {
                replicators.put(replicator, connection);
            }
        }

        changeNotifier.postChange(new MessageEndpointListenerChange(connection,status));
    }

    /**
     * Close the given connection.
     *
     * @param connection
     */
    public void close(@NonNull MessageEndpointConnection connection) {
        if (connection == null)
            throw new IllegalArgumentException("connection cannot be null.");

        synchronized (lock) {
            for (C4Replicator replicator : replicators.keySet()) {
                if (connection == replicators.get(replicator)) {
                    replicator.stop();
                    break;
                }
            }
        }
    }

    /**
     * Close all active connections.
     */
    public void closeAll() {
        synchronized (lock) {
            for (C4Replicator replicator : replicators.keySet()) {
                replicator.stop();
            }
        }
    }

    /**
     * Add a change listener.
     *
     * @param listener
     * @return
     */
    @NonNull
    public ListenerToken addChangeListener(@NonNull MessageEndpointListenerChangeListener listener) {
        return addChangeListener(null, listener);
    }

    /**
     * Add a change listener with the given dispatch queue.
     *
     * @param queue
     * @param listener
     * @return
     */
    @NonNull
    public ListenerToken addChangeListener(Executor queue, @NonNull MessageEndpointListenerChangeListener listener) {
        if (listener == null)
            throw new IllegalArgumentException("listener cannot be null.");

        return changeNotifier.addChangeListener(queue, listener);
    }

    /**
     * Remove a change listener.
     *
     * @param token
     */
    public void removeChangeListener(@NonNull ListenerToken token) {
        if (token == null)
            throw new IllegalArgumentException("token cannot be null.");

        changeNotifier.removeChangeListener(token);
    }

    //---------------------------------------------
    // Private member methods (in class only)
    //---------------------------------------------

    private void statusChanged(C4Replicator replicator, C4ReplicatorStatus status) {
        MessageEndpointConnection connection = null;
        synchronized (lock) {
            connection = replicators.get(replicator);
            if (status.getActivityLevel() == kC4Stopped) {
                replicators.remove(replicator);
            }
        }

        if (connection != null) {
            changeNotifier.postChange(new MessageEndpointListenerChange(connection, status));
        }

        if (status.getActivityLevel() == kC4Stopped) {
            replicator.free();
        }
    }
}

