//
// Database.java
//
// Copyright (c) 2018 Couchbase, Inc.  All rights reserved.
//
// Licensed under the Couchbase License Agreement (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
// https://info.couchbase.com/rs/302-GJY-034/images/2017-10-30_License_Agreement.pdf
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
package com.couchbase.lite;

import android.support.annotation.NonNull;

import com.couchbase.litecore.C4Constants;
import com.couchbase.litecore.C4Database;
import com.couchbase.litecore.LiteCoreException;

/**
 * A Couchbase Lite database.
 */
public final class Database extends AbstractDatabase {

    static {
        prediction = new Prediction();
    }

    /**
     * Construct a  AbstractDatabase with a given name and database config.
     * If the database does not yet exist, it will be created, unless the `readOnly` option is used.
     *
     * @param name   The name of the database. May NOT contain capital letters!
     * @param config The database config, Note: null config parameter is not allowed with Android platform
     * @throws CouchbaseLiteException Throws an exception if any error occurs during the open operation.
     */
    public Database(@NonNull String name, @NonNull DatabaseConfiguration config) throws CouchbaseLiteException {
        super(name, config);
    }

    /**
     * <b>ENTERPRISE EDITION API</b><br/></br>
     * <p>
     * Changes the database's encryption key, or removes encryption if the new key is null.
     *
     * @param encryptionKey The encryption key
     * @throws CouchbaseLiteException
     */
    public void changeEncryptionKey(EncryptionKey encryptionKey) throws CouchbaseLiteException {
        synchronized (lock) {
            mustBeOpen();
            int keyType = encryptionKey == null || encryptionKey.getKey() == null ?
                    C4Constants.C4EncryptionAlgorithm.kC4EncryptionNone :
                    C4Constants.C4EncryptionAlgorithm.kC4EncryptionAES256;
            try {
                c4db.rekey(keyType, encryptionKey != null ? encryptionKey.getKey() : null);
            } catch (LiteCoreException e) {
                throw CBLStatus.convertException(e);
            }
        }
    }

    /**
     * <b>ENTERPRISE EDITION API : UNCOMMITTED</b><br/><br/>
     * <p>
     * The predictive model manager for registering and unregistering predictive models.
     */
    @NonNull
    public static final Prediction prediction;

    //---------------------------------------------
    // Implementing abstract methods for Encryption
    //---------------------------------------------

    int getEncryptionAlgorithm() {
        return (config.getEncryptionKey() == null || config.getEncryptionKey().getKey() == null)
                ? C4Constants.C4EncryptionAlgorithm.kC4EncryptionNone
                : C4Constants.C4EncryptionAlgorithm.kC4EncryptionAES256;
    }

    byte[] getEncryptionKey() {
        return config.getEncryptionKey() == null ? null : config.getEncryptionKey().getKey();
    }

    //---------------------------------------------
    // Package
    //---------------------------------------------

    Database(C4Database c4db) {
        super(c4db);
    }

}
