/*
 * Author: Jude Pereira
 * Copyright (c) 2014
 */

package com.clevertap.android.sdk;

import android.annotation.SuppressLint;
import android.Manifest;
import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.support.v4.content.ContextCompat;
import android.bluetooth.BluetoothAdapter;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.telephony.TelephonyManager;
import com.clevertap.android.sdk.exceptions.CleverTapPermissionsNotSatisfied;

import com.google.android.gms.ads.identifier.AdvertisingIdClient;
import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.GoogleApiAvailability;
//import com.google.android.gms.common.GooglePlayServicesUtil;

import java.util.ArrayList;
import java.util.UUID;

/**
 * Provides various methods to access the device.
 */
final class DeviceInfo {
    private static final String GUID_PREFIX = "__";

    private static String provisionalGUID = null;

    private static final Object deviceIDLock = new Object();
    private static final Object adIDLock = new Object();

    private static Context context = null;

    private static String googleAdID = null;

    private static boolean limitAdTracking = false;

    static String getGoogleAdID() {
        synchronized (adIDLock) {
            return googleAdID;
        }
    }

    static boolean isLimitAdTrackingEnabled() {
        synchronized (adIDLock) {
            return limitAdTracking;
        }
    }

    /**
     * CleverTapAPI calls this in its singleton constructor.
     */
    static void initializeWithContext(Context context) {
        DeviceInfo.context = context;
        initDeviceIDAsync();
    }

    private static void generateProvisionalGUID() {
        synchronized (deviceIDLock) {
            if (provisionalGUID == null) {
                provisionalGUID = generateGUID();
            }
        }
    }

    private static void initDeviceIDAsync() {
        // generate a provisional while we do the rest async
        generateProvisionalGUID();

        CleverTapAPI.postAsyncSafely("DeviceInfo#generateDeviceID", new Runnable() {
            @Override
            public void run() {
                // grab and cache the googleAdID in any event if available
                // if we already have a deviceID we won't user ad id as the guid
                cacheGoogleAdID();

                // if we already have a device ID use it and just notify
                // otherwise generate one, either from ad id if available or the provisional
                String deviceID = getDeviceID();
                if (deviceID != null && deviceID.trim().length() > 2) {
                    notifyNewDeviceID(deviceID);
                } else {
                    generateDeviceID();
                }
            }
        });
    }

    private static void cacheGoogleAdID() {

        String advertisingID;
        try {
            AdvertisingIdClient.Info adInfo;
            adInfo = AdvertisingIdClient.getAdvertisingIdInfo(context);
            advertisingID = adInfo.getId();
            synchronized (adIDLock) {
                limitAdTracking = adInfo.isLimitAdTrackingEnabled();
            }
        } catch (Throwable t) {
            advertisingID = null;
        }

        if (advertisingID != null && advertisingID.trim().length() > 2) {
            synchronized (adIDLock) {
                googleAdID = advertisingID.replace("-", "");
            }
        }
    }

    private static void generateDeviceID() {
        String generatedDeviceID;

        // try google ad id first
        // if no ad id then make provisional guid permanent
        if (googleAdID != null) {
            synchronized (adIDLock) {
                generatedDeviceID = Constants.GUID_PREFIX_GOOGLE_AD_ID + googleAdID;
            }
        } else {
            Logger.v("Failed with Advertising ID");

            synchronized (deviceIDLock) {
                generatedDeviceID = provisionalGUID;
                Logger.v("Made provisional ID permanent");
            }
        }

        if (generatedDeviceID != null && generatedDeviceID.trim().length() > 2) {
            forceUpdateDeviceId(generatedDeviceID);
        } else {
            Logger.v("Unable to generate device ID");
        }
    }

    static String getAttributionID() {
        String deviceID = getDeviceID();
        synchronized (deviceIDLock) {
            return (deviceID != null && deviceID.trim().length() > 2) ? deviceID : provisionalGUID;
        }
    }

    static String getDeviceID() {
        synchronized (deviceIDLock) {
            return StorageHelper.getString(context, Constants.DEVICE_ID_TAG, null);
        }
    }

    private static void notifyNewDeviceID(final String deviceID) {
        CleverTapAPI.postAsyncSafely("DeviceInfo#notifyNewDeviceID", new Runnable() {
            @Override
            public void run() {
                try {
                    CleverTapAPI.notifyUserProfileInitialized(deviceID);
                } catch (Throwable t) {
                    // no-op
                }
            }
        });
    }

    private static String generateGUID() {
        return GUID_PREFIX + UUID.randomUUID().toString().replace("-", "");
    }


    static void forceNewDeviceID() {
        String deviceID = generateGUID();
        forceUpdateDeviceId(deviceID);
    }

    /**
     * Force updates the device ID, with the ID specified.
     * <p>
     * This is used internally by the SDK, there is no need to call this explicitly.
     * </p>
     *
     * @param id      The new device ID
     */
    @SuppressLint("CommitPrefEdits")
    static void forceUpdateDeviceId(String id) {
        Logger.v("Force updating the device ID to " + id);
        synchronized (deviceIDLock) {
            StorageHelper.putString(context, Constants.DEVICE_ID_TAG, id);
        }
        notifyNewDeviceID(id);
    }

    /**
     * Tests whether a particular permission is available or not.
     *
     * @param context    The Android {@link Context}
     * @param permission The fully qualified Android permission name
     * @throws CleverTapPermissionsNotSatisfied
     */
    static void testPermission(final Context context, String permission)
            throws CleverTapPermissionsNotSatisfied {
        if (!hasPermission(context, permission))
            throw new CleverTapPermissionsNotSatisfied("Permission required: " + permission);
        else
            Logger.v("Permissions added");
    }

    @SuppressWarnings("WeakerAccess")
    static boolean hasPermission(final Context context, String permission) {
        try {
            return PackageManager.PERMISSION_GRANTED == ContextCompat.checkSelfPermission(context, permission);
        } catch (Throwable t) {
            return false;
        }
    }

    /**
     * Returns the integer identifier for the default app icon.
     *
     * @param context The Android context
     * @return The integer identifier for the image resource
     */
    static int getAppIconAsIntId(final Context context) {
        ApplicationInfo ai = context.getApplicationInfo();
        return ai.icon;
    }

    /**
     * Push Service Detection Handling
     */

    private static ArrayList<PushType> enabledPushTypes = null;
    private static String GCMSenderID = null;
    private static Boolean isFirebasePresent = null;
    private static Boolean areGoogleServicesAvailable = null;
    private static final String FIREBASE_CLASS_NAME = "com.google.firebase.messaging.FirebaseMessaging";

    private static boolean isGCMAvailable() {
        return (isGooglePlayServicesAvailable() && getGCMSenderID() != null);
    }

    private static boolean isFCMAvailable() {
        if (isFirebasePresent == null) {
            try {
                Class.forName(FIREBASE_CLASS_NAME);
                isFirebasePresent = true;
                Logger.v("FCM installed");
            } catch (ClassNotFoundException e) {
                isFirebasePresent = false;
                Logger.v("FCM unavailable");
            }
        }
        return (isGooglePlayServicesAvailable() && isFirebasePresent);
    }

    static ArrayList<PushType> getEnabledPushTypes() {
        if (enabledPushTypes == null) {
            enabledPushTypes = new ArrayList<PushType>();

            // only return one of fcm and gcm , preferring fcm
            boolean fcmAvail = isFCMAvailable();
            if (fcmAvail) {
                enabledPushTypes.add(PushType.FCM);
            }

            if (!fcmAvail && isGCMAvailable()) {
                enabledPushTypes.add(PushType.GCM);
            }

        }
        return enabledPushTypes;
    }

    static boolean isGooglePlayServicesAvailable() {
        if (areGoogleServicesAvailable == null) {
            try {
                GoogleApiAvailability googleAPI = GoogleApiAvailability.getInstance();
                int result = googleAPI.isGooglePlayServicesAvailable(context);
                //int result = GooglePlayServicesUtil.isGooglePlayServicesAvailable(context);
                areGoogleServicesAvailable = (result == ConnectionResult.SUCCESS);

                if (areGoogleServicesAvailable) {
                    Logger.v("Google Play services availabile");
                } else {
                    Logger.d("Google Play services not available");
                }

            } catch (Throwable t) {
                Logger.d("Error checking Google Play services availability", t);
                areGoogleServicesAvailable = false;
            }
        }
        return areGoogleServicesAvailable;
    }

    static String getGCMSenderID() {
        if (GCMSenderID == null) {
            String senderID = null;
            try {
                senderID = ManifestMetaData.getMetaData(context, Constants.LABEL_SENDER_ID);
                if (senderID != null) {
                    senderID = senderID.replace("id:", "");
                } else {
                    Logger.v("GCM sender ID not found");
                }
            }  catch (Throwable t) {
                Logger.v("Error retrieving GCM sender ID", t);
            }
            GCMSenderID = senderID;
        }
        return GCMSenderID;
    }

    static String getCarrier() {
        try {
            TelephonyManager tm = (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);
            String carrier = tm.getSimOperatorName();
            ValidationResult vr = Validator.cleanObjectValue(carrier, Validator.ValidationContext.Profile);
            if (vr.getErrorCode() == 0)
                return (String) vr.getObject();
        } catch (Throwable ignore) {
        }
        return "";
    }

    static String getCountryCode() {
        try {
            TelephonyManager tm = (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);
            return tm.getSimCountryIso();
        } catch (Throwable ignore) {
            return "";
        }
    }

    static Boolean isWifiConnected() {
        Boolean ret = null;

        if (PackageManager.PERMISSION_GRANTED == context.checkCallingOrSelfPermission(Manifest.permission.ACCESS_NETWORK_STATE)) {
            ConnectivityManager connManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
            NetworkInfo networkInfo = connManager.getActiveNetworkInfo();
            ret = (networkInfo != null && networkInfo.getType() == ConnectivityManager.TYPE_WIFI && networkInfo.isConnected());
        }

        return ret;
    }

    @SuppressLint("MissingPermission")
    static Boolean isBluetoothEnabled() {
        Boolean isBluetoothEnabled = null;
        try {
            BluetoothAdapter bluetoothAdapter = BluetoothAdapter.getDefaultAdapter();
            if (bluetoothAdapter != null) {
                isBluetoothEnabled = bluetoothAdapter.isEnabled();
            }
        } catch (SecurityException e) {
            // do nothing since we don't have permissions
        } catch (NoClassDefFoundError e) {
            // Some phones doesn't have this class. Just ignore it
        }
        return isBluetoothEnabled;
    }

    static String getBluetoothVersion() {
        String bluetoothVersion = null;
        if (android.os.Build.VERSION.SDK_INT >= 8) {
            bluetoothVersion = "none";
            if(android.os.Build.VERSION.SDK_INT >= 18 &&
                    context.getPackageManager().hasSystemFeature(PackageManager.FEATURE_BLUETOOTH_LE)) {
                bluetoothVersion = "ble";
            } else if(context.getPackageManager().hasSystemFeature(PackageManager.FEATURE_BLUETOOTH)) {
                bluetoothVersion = "classic";
            }
        }
        return bluetoothVersion;
    }

    static String getNetworkType() {
        TelephonyManager mTelephonyManager = (TelephonyManager)
                context.getSystemService(Context.TELEPHONY_SERVICE);
        int networkType = mTelephonyManager.getNetworkType();
        switch (networkType) {
            case TelephonyManager.NETWORK_TYPE_GPRS:
            case TelephonyManager.NETWORK_TYPE_EDGE:
            case TelephonyManager.NETWORK_TYPE_CDMA:
            case TelephonyManager.NETWORK_TYPE_1xRTT:
            case TelephonyManager.NETWORK_TYPE_IDEN:
                return "2G";
            case TelephonyManager.NETWORK_TYPE_UMTS:
            case TelephonyManager.NETWORK_TYPE_EVDO_0:
            case TelephonyManager.NETWORK_TYPE_EVDO_A:
            case TelephonyManager.NETWORK_TYPE_HSDPA:
            case TelephonyManager.NETWORK_TYPE_HSUPA:
            case TelephonyManager.NETWORK_TYPE_HSPA:
            case TelephonyManager.NETWORK_TYPE_EVDO_B:
            case TelephonyManager.NETWORK_TYPE_EHRPD:
            case TelephonyManager.NETWORK_TYPE_HSPAP:
                return "3G";
            case TelephonyManager.NETWORK_TYPE_LTE:
                return "4G";
            default:
                return null;
        }
    }

}
