package com.clevertap.android.sdk;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.SharedPreferences;

import com.google.firebase.iid.FirebaseInstanceId;


public class FcmManager {

    private static Context context = null;
    private static final Object LOCK = new Object();

    /**
     * CleverTapAPI calls this in its singleton constructor.
     */
    static void initializeWithContext(Context context) {
        FcmManager.context = context;
        doFCMRefresh();
        // may be null here on install
        // in that case will get kicked off from the FcmTokenListenerService
    }

    // called from FcmTokenListenerService
    static void onTokenRefresh() {
        doFCMRefresh();
    }

    static String getDeviceToken() {
        return getCachedFCMToken();
    }

    // all happens serial async

    private static void doFCMRefresh() {
        CleverTapAPI.postAsyncSafely("FcmManager#doFCMRefresh", new Runnable() {
            @Override
            public void run() {
                try {

                    if (!isGooglePlayServicesAvailable()) {
                        Logger.d("FcmManager: Play Services unavailable, unable to request FCM token");
                        return;
                    }

                    String freshToken = FCMGetFreshToken();
                    if (freshToken == null) return;

                    cacheFCMToken(freshToken);

                    // better safe to always force a push from here
                    pushDeviceToken(freshToken, true, true);

                    try {
                        CleverTapAPI.getInstance(context).deviceTokenDidRefresh(freshToken, PushType.FCM);
                    } catch (Throwable t) {
                        //no-op
                    }
                } catch (Throwable t) {
                    Logger.v("FcmManager: FCM Token error", t);
                }
            }
        });
    }

    // helpers

    private static boolean isGooglePlayServicesAvailable() {
        return DeviceInfo.isGooglePlayServicesAvailable();
    }

    /**
     * request token from FCM
     */
    private static String FCMGetFreshToken() {
        Logger.v("FcmManager: Requesting a FCM token");
        String token = null;
        try {
            token = FirebaseInstanceId.getInstance().getToken();
            Logger.i("FCM token : "+token);
        } catch (Throwable t) {
            Logger.v("FcmManager: Error requesting FCM token", t);
        }
        return token;
    }

    private static SharedPreferences getPreferences() {
        try {
            return (context == null) ? null : StorageHelper.getPreferences(context);
        } catch (Throwable t) {
            return null;
        }
    }

    private static final String FCM_PROPERTY_REG_ID = "fcm_token";

    private static String getCachedFCMToken() {
        SharedPreferences prefs = getPreferences();
        return (prefs == null) ? null : prefs.getString(FCM_PROPERTY_REG_ID, null);
    }

    @SuppressLint("CommitPrefEdits")
    private static void cacheFCMToken(String token) {
        try {
            if (token == null || alreadyHaveFCMToken(token)) return;

            final SharedPreferences prefs = getPreferences();
            if (prefs == null) return;

            SharedPreferences.Editor editor = prefs.edit();
            editor.putString(FCM_PROPERTY_REG_ID, token);
            StorageHelper.persist(editor);
        } catch (Throwable t) {
            Logger.v("FcmManager: Unable to cache FCM Token", t);
        }
    }

    private static boolean alreadyHaveFCMToken(final String newToken) {
        if (newToken == null) return false;

        String cachedToken = getCachedFCMToken();
        return (cachedToken != null && cachedToken.equals(newToken));
    }

    private static boolean havePushedDeviceToken = false;

    synchronized static void pushDeviceToken(String token, final boolean register, final boolean forceUpdate) {
        synchronized (LOCK) {
            if (havePushedDeviceToken && !forceUpdate) {
                Logger.v("FcmManager: skipping device token push - already sent.");
                return;
            }

            try {
                token = (token != null) ? token : getCachedFCMToken();
                if (token == null) return;
                DataHandler.pushDeviceToken(context, token, register, PushType.FCM);
                havePushedDeviceToken = true;
            } catch (Throwable t) {
                Logger.v("FcmManager: pushing device token failed", t);
            }
        }
    }
}
