/*
 * Author: Jude Pereira
 * Copyright (c) 2014
 */

package com.clevertap.android.sdk;


import android.annotation.SuppressLint;
import android.app.*;
import android.bluetooth.BluetoothClass;
import android.content.ContentResolver;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.graphics.Bitmap;
import android.location.Location;
import android.location.LocationManager;
import android.media.RingtoneManager;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.support.v4.app.NotificationCompat;

import com.clevertap.android.sdk.exceptions.CleverTapMetaDataNotFoundException;
import com.clevertap.android.sdk.exceptions.CleverTapPermissionsNotSatisfied;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.lang.ref.WeakReference;
import java.net.URLDecoder;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import static android.content.Context.NOTIFICATION_SERVICE;

/**
 * Manipulates the CleverTap SDK.
 */
@SuppressWarnings("WeakerAccess")
public final class CleverTapAPI {

    public enum LogLevel{
        OFF(-1),
        INFO(0),
        DEBUG(2);

        private final int value;

        LogLevel(final int newValue) {
            value = newValue;
        }

        public int intValue() { return value; }
    }

    // Static fields
    public static final String CHARGED_EVENT = "Charged";
    public static final String NOTIFICATION_TAG = "wzrk_pn";
    private static final String CACHED_GUIDS_KEY = "cachedGUIDsKey";
    private static final String MULTI_USER_PREFIX = "mt_";
    private static final Handler handlerUsingMainLooper = new Handler(Looper.getMainLooper());
    private static final ExecutorService es = Executors.newFixedThreadPool(1);
    private static int debugLevel = CleverTapAPI.LogLevel.INFO.intValue();
    private static boolean appForeground = false;
    private static CleverTapAPI ourInstance = null;
    static Runnable pendingInappRunnable = null;
    private static int lastLocationPingTime = 0;
    private InAppNotificationListener inAppNotificationListener;

    static int activityCount = 0;

    private static ArrayList<ValidationResult> pendingValidationResults = new ArrayList<ValidationResult>();
    private static final Boolean pendingValidationResultsLock = true;

    private static String parseInstallationId = null;

    private static boolean appLaunchPushed = false;
    private static final Object appLaunchPushedLock = new Object();

    private final static int EMPTY_NOTIFICATION_ID = -1000;

    static String localAccountID = null, localToken = null, localRegion = null;

    static String getAccountID(final Context context) {
        if (localAccountID == null) {
            localAccountID =  ManifestMetaData.getMetaData(context, Constants.LABEL_ACCOUNT_ID);
        }
        return localAccountID;
    }

    static String getAccountToken(final Context context) {
        if (localToken == null) {
            localToken =  ManifestMetaData.getMetaData(context, Constants.LABEL_TOKEN);
        }
        return localToken;
    }

    static String getAccountRegion(final Context context) {
        if (localRegion == null) {
            localRegion =  ManifestMetaData.getMetaData(context, Constants.LABEL_REGION);
        }
        return localRegion;
    }

    static boolean isAppLaunchPushed() {
        synchronized (appLaunchPushedLock) {
            return appLaunchPushed;
        }
    }

    static void setAppLaunchPushed(boolean pushed) {
        synchronized (appLaunchPushedLock) {
            appLaunchPushed = pushed;
        }
    }

    private static WeakReference<Activity> currentActivity;

    static Activity getCurrentActivity() {
        return (currentActivity == null) ? null : currentActivity.get();
    }

    static void setCurrentActivity(Activity activity) {
        currentActivity = (activity == null) ? null : new WeakReference<Activity>(activity);
    }

    static String getCurrentActivityName() {
        Activity current = getCurrentActivity();
        return (current != null) ? current.getLocalClassName() : null;
    }

    static String getParseInstallationId() {
        return parseInstallationId;
    }

    private static String getGoogleAdID() {
        return DeviceInfo.getGoogleAdID();
    }

    private static boolean isLimitAdTrackingEnabled() {
        return DeviceInfo.isLimitAdTrackingEnabled();
    }

    // Non static fields
    public final EventHandler event;
    public final DataHandler data;
    public final ProfileHandler profile;
    public final SessionHandler session;
    private Runnable sessionTimeoutRunnable = null;
    private final Context context;
    private long appLastSeen;
    private SyncListener syncListener = null;
    private Location locationFromUser = null;

    Context getContext() {
        return context;
    }

    public void setInAppNotificationListener(InAppNotificationListener inAppNotificationListener) {
        this.inAppNotificationListener = inAppNotificationListener;
    }

    public InAppNotificationListener getInAppNotificationListener() {
        return inAppNotificationListener;
    }

    static ValidationResult popValidationResult() {
        // really a shift
        ValidationResult vr = null;

        synchronized (pendingValidationResultsLock) {
            try {
                if (!pendingValidationResults.isEmpty()) {
                    vr = pendingValidationResults.remove(0);
                }
            } catch (Exception e) {
                // no-op
            }
        }
        return vr;
    }

    static void pushValidationResult(ValidationResult vr) {
        synchronized (pendingValidationResultsLock) {
            try {
                int len = pendingValidationResults.size();
                if (len > 50) {
                    ArrayList<ValidationResult> trimmed = new ArrayList<ValidationResult>();
                    // Trim down the list to 40, so that this loop needn't run for the next 10 events
                    // Hence, skip the first 10 elements
                    for (int i = 10; i < len; i++)
                        trimmed.add(pendingValidationResults.get(i));
                    trimmed.add(vr);
                    pendingValidationResults = trimmed;
                } else {
                    pendingValidationResults.add(vr);
                }
            } catch (Exception e) {
                // no-op
            }
        }
    }

    /**
     * Returns the generic handler object which is used to post
     * runnables. The returned value will never be null.
     *
     * @return The static generic handler
     * @see Handler
     */
    static Handler getHandlerUsingMainLooper() {
        return handlerUsingMainLooper;
    }

    private static long EXECUTOR_THREAD_ID = 0;

    /**
     * Use this to safely post a runnable to the async handler.
     * It adds try/catch blocks around the runnable and the handler itself.
     */
    @SuppressWarnings("UnusedParameters")
    static void postAsyncSafely(final String name, final Runnable runnable) {
        try {
            final boolean executeSync = Thread.currentThread().getId() == EXECUTOR_THREAD_ID;

            if (executeSync) {
                runnable.run();
            } else {
                es.submit(new Runnable() {
                    @Override
                    public void run() {
                        EXECUTOR_THREAD_ID = Thread.currentThread().getId();
                        try {
                            //Logger.logFine("Executor service: Starting task - " + name);
                            //final long start = System.currentTimeMillis();
                            runnable.run();
                            //final long time = System.currentTimeMillis() - start;
                            //Logger.logFine("Executor service: Task completed successfully in " + time + "ms (" + name + ")");
                        } catch (Throwable t) {
                            Logger.v("Executor service: Failed to complete the scheduled task", t);
                        }
                    }
                });
            }
        } catch (Throwable t) {
            Logger.v("Failed to submit task to the executor service", t);
        }
    }

    /**
     * Returns whether or not the app is in the foreground.
     *
     * @return The foreground status
     */
    static boolean isAppForeground() {
        return appForeground;
    }

    /**
     * This method is used internally.
     */
    public static void setAppForeground(boolean appForeground) {
        CleverTapAPI.appForeground = appForeground;
    }

    public void setSyncListener(SyncListener syncListener) {
        this.syncListener = syncListener;
    }

    public SyncListener getSyncListener() {
        return syncListener;

    }

    public static void changeCredentials(String accountID, String token) {
        changeCredentials(accountID, token, null);
    }

    public static void changeCredentials(String accountID, String token, String region) {

        if (localAccountID != null && localToken != null) {
            Logger.i("CleverTap SDK already initialized with accountID: " + localAccountID + " accountToken: " + localToken + ". Subsequent call to changeCredentials is ignored");
            return;
        }

        if (accountID != null && accountID.trim().length() > 0) {
            localAccountID = accountID;
        }

        if (token != null && token.trim().length() > 0) {
            localToken = token;
        }

        if (region != null && region.trim().length() > 0) {
            localRegion = region;
        }
    }

    /**
     * Returns an instance of the CleverTap SDK.
     *
     * @param context The Android context
     * @return The {@link CleverTapAPI} object
     * @throws CleverTapMetaDataNotFoundException
     * @throws CleverTapPermissionsNotSatisfied
     */
    public static synchronized CleverTapAPI getInstance(final Context context)
            throws CleverTapMetaDataNotFoundException, CleverTapPermissionsNotSatisfied {
        if (ourInstance == null && context != null) {

            // wake up the data store - loads the local profile - good to do this first
            LocalDataStore.initializeWithContext(context);

            // handles the guid initialization logic
            DeviceInfo.initializeWithContext(context);

            ourInstance = new CleverTapAPI(context.getApplicationContext());

        }

        return ourInstance;
    }

    /**
     * @return Application's {@code SharedPreferences}.
     */
    private SharedPreferences getPreferences(Context context) {
        return StorageHelper.getPreferences(context);
    }

    private void testPermissions(final Context context) throws CleverTapPermissionsNotSatisfied {
        DeviceInfo.testPermission(context, "android.permission.INTERNET");
    }

    private CleverTapAPI(final Context context) throws CleverTapMetaDataNotFoundException, CleverTapPermissionsNotSatisfied {
        this.context = context;
        event = new EventHandler(context);
        profile = new ProfileHandler(context);
        data = new DataHandler(context);
        session = new SessionHandler(context);

        String accountID = getAccountID(context);
        if (accountID == null) {
            Logger.i("CleverTap SDK cannot be initialized: accountID is missing");
            throw new CleverTapMetaDataNotFoundException("CleverTap accountID is missing");
        }

        String accountToken = getAccountToken(context);
        if (accountToken == null) {
            Logger.i("CleverTap SDK cannot be initialized: account Token is missing");
            throw new CleverTapMetaDataNotFoundException("CleverTap account Token is missing");
        }

        testPermissions(context);

        String accountRegion = getAccountRegion(context);
        if (accountRegion == null) {
            accountRegion = "Default"; // just for logging purposes
        }


        Logger.i("CleverTap SDK initialized with accountId: "+ accountID + " accountToken: " + accountToken + " accountRegion: " + accountRegion);

        manifestAsyncValidation();
    }

    //Run manifest validation in async
    private void manifestAsyncValidation(){
        postAsyncSafely("Manifest Validation", new Runnable() {
            @Override
            public void run() {
               ManifestValidator.validate(context);
            }
        });
    }


    // multi-user handling

    private JSONObject getCachedGUIDs() {
        JSONObject cache = null;
        String json = StorageHelper.getString(context, CACHED_GUIDS_KEY, null);
        if (json != null) {
            try {
                cache = new JSONObject(json);
            } catch (Throwable t) {
                // no-op
                Logger.v("Error reading guid cache: " + t.toString());
            }
        }

        return (cache != null) ? cache : new JSONObject();
    }

    private void setCachedGUIDs(JSONObject cachedGUIDs) {
        if (cachedGUIDs == null) return;
        try {
            StorageHelper.putString(context, CACHED_GUIDS_KEY, cachedGUIDs.toString());
        } catch (Throwable t) {
            // no-op
            Logger.v("Error persisting guid cache: " + t.toString());
        }
    }

    private String getGUIDForIdentifier(String key, String identifier) {
        if (key == null || identifier == null) return null;

        String cacheKey = key + "_" + identifier;
        JSONObject cache = getCachedGUIDs();
        try {
            return cache.getString(cacheKey);
        } catch (Throwable t) {
            Logger.v("Error reading guid cache: " + t.toString());
            return null;
        }
    }

    void cacheGUIDForIdentifier(String guid, String key, String identifier) {
        if (guid == null || key == null || identifier == null) return;

        String cacheKey = key + "_" + identifier;
        JSONObject cache = getCachedGUIDs();
        try {
            cache.put(cacheKey, guid);
            setCachedGUIDs(cache);
        } catch (Throwable t) {
            // no-op
            Logger.v("Error caching guid: " + t.toString());
        }
    }

    private boolean isAnonymousDevice() {
        JSONObject cachedGUIDs = getCachedGUIDs();
        return cachedGUIDs.length() <= 0;
    }

    private boolean deviceIsMultiUser() {
        JSONObject cachedGUIDs = getCachedGUIDs();
        return cachedGUIDs.length() > 1;
    }

    private static String processingUserLoginIdentifier = null;
    private static final Boolean processingUserLoginLock = true;

    private boolean isProcessUserLoginWithIdentifier(String identifier) {
        synchronized (processingUserLoginLock) {
            return processingUserLoginIdentifier != null && processingUserLoginIdentifier.equals(identifier);
        }
    }

    /**
     * Creates a separate and distinct user profile identified by one or more of Identity,
     * Email, FBID or GPID values,
     * and populated with the key-values included in the profile map argument.
     * <p>
     * If your app is used by multiple users, you can use this method to assign them each a
     * unique profile to track them separately.
     * <p>
     * If instead you wish to assign multiple Identity, Email, FBID and/or GPID values to the same
     * user profile,
     * use profile.push rather than this method.
     * <p>
     * If none of Identity, Email, FBID or GPID is included in the profile map,
     * all profile map values will be associated with the current user profile.
     * <p>
     * When initially installed on this device, your app is assigned an "anonymous" profile.
     * The first time you identify a user on this device (whether via onUserLogin or profilePush),
     * the "anonymous" history on the device will be associated with the newly identified user.
     * <p>
     * Then, use this method to switch between subsequent separate identified users.
     * <p>
     * Please note that switching from one identified user to another is a costly operation
     * in that the current session for the previous user is automatically closed
     * and data relating to the old user removed, and a new session is started
     * for the new user and data for that user refreshed via a network call to CleverTap.
     * In addition, any global frequency caps are reset as part of the switch.
     *
     * @param profile The map keyed by the type of identity, with the value as the identity
     */

    public void onUserLogin(final Map<String, Object> profile) {
        if (profile == null) return;

        try {
            final String currentGUID = getCleverTapID();
            if (currentGUID == null) return;

            boolean haveIdentifier = false;
            String cachedGUID = null;

            // check for valid identifier keys
            // use the first one we find
            for (String key : profile.keySet()) {
                Object value = profile.get(key);
                if (Constants.PROFILE_IDENTIFIER_KEYS.contains(key)) {
                    try {
                        String identifier = value.toString();
                        if (identifier != null && identifier.length() > 0) {
                            haveIdentifier = true;
                            cachedGUID = getGUIDForIdentifier(key, identifier);
                            if (cachedGUID != null) break;
                        }
                    } catch (Throwable t) {
                        // no-op
                    }
                }
            }

            // if no valid identifier provided or there are no identified users on the device; just push on the current profile
            if (!haveIdentifier || isAnonymousDevice()) {
                Logger.d("onUserLogin: no identifier provided or device is anonymous, pushing on current user profile");
                this.profile.push(profile);
                return;
            }

            // if identifier maps to current guid, push on current profile
            if (cachedGUID != null && cachedGUID.equals(currentGUID)) {
                Logger.d("onUserLogin: " + profile.toString() + " maps to current device id " + currentGUID + " pushing on current profile");
                this.profile.push(profile);
                return;
            }

            // stringify profile to use as dupe blocker
            String profileToString = profile.toString();

            // as processing happens async block concurrent onUserLogin requests with the same profile, as our cache is set async
            if (isProcessUserLoginWithIdentifier(profileToString)) {
                Logger.d("Already processing onUserLogin for " + profileToString);
                return;
            }

            // create new guid if necessary and reset
            // block any concurrent onUserLogin call for the same profile
            synchronized (processingUserLoginLock) {
                processingUserLoginIdentifier = profileToString;
            }

            Logger.v("onUserLogin: queuing reset profile for " + profileToString
                    + " with Cached GUID " + ((cachedGUID != null) ? cachedGUID : "NULL"));

            final ProfileHandler profileHandler = this.profile;

            final String guid = cachedGUID;

            postAsyncSafely("resetProfile", new Runnable() {
                @Override
                public void run() {
                    try {

                        // unregister the device token on the current user
                        forcePushDeviceToken(false);

                        // try and flush and then reset the queues
                        CommsManager.flushQueueSync(context);
                        QueueManager.clearQueues(context);

                        // clear out the old data
                        LocalDataStore.changeUser(context);
                        InAppFCManager.changeUser(context);
                        activityCount = 1;
                        forceDestroySession();

                        // either force restore the cached GUID or generate a new one
                        if (guid != null) {
                            DeviceInfo.forceUpdateDeviceId(guid);
                        } else {
                            DeviceInfo.forceNewDeviceID();
                        }

                        forcePushAppLaunchedEvent("onLoginUser");
                        profileHandler.push(profile);
                        forcePushDeviceToken(true);
                        synchronized (processingUserLoginLock) {
                            processingUserLoginIdentifier = null;
                        }
                    } catch (Throwable t) {
                        Logger.v("Reset Profile error", t);
                    }
                }
            });

        } catch (Throwable t) {
            Logger.v("onUserLogin failed", t);
        }
    }

    static void notifyUserProfileInitialized(String deviceID) {
        if (ourInstance == null) return;
        ourInstance._notifyProfileInitialized(deviceID);
    }

    private void _notifyProfileInitialized(String deviceID) {
        deviceID = (deviceID != null) ? deviceID : getCleverTapID();

        if (deviceID == null) return;

        final SyncListener sl;
        try {
            sl = getSyncListener();
            if (sl != null) {
                sl.profileDidInitialize(deviceID);
            }
        } catch (Throwable t) {
            // Ignore
        }
    }

    @SuppressLint("CommitPrefEdits")
    private void pushDailyEventsAsync() {
        postAsyncSafely("CleverTapAPI#pushDailyEventsAsync", new Runnable() {
            @Override
            public void run() {
                try {
                    Logger.v("Queuing daily events");
                    profile.pushBasicProfile(null);
                } catch (Throwable t) {
                    Logger.v("Daily profile sync failed", t);
                }
            }
        });
    }

    /**
     * Enables or disables debugging. If enabled, see debug messages in Android's logcat utility.
     * Debug messages are tagged as CleverTap.
     *
     * @param level Can be one of the following:  -1 (disables all debugging), 0 (default, shows minimal SDK integration related logging),
     *              1(shows debug output)
     */
    public static void setDebugLevel(int level) {
            debugLevel = level;
    }

    /**
     * Enables or disables debugging. If enabled, see debug messages in Android's logcat utility.
     * Debug messages are tagged as CleverTap.
     *
     * @param level Can be one of the following: LogLevel.OFF (disables all debugging), LogLevel.INFO (default, shows minimal SDK integration related logging),
     *              LogLevel.DEBUG(shows debug output)
     */
    public static void setDebugLevel(LogLevel level) {
        debugLevel = level.intValue();
    }

    public static int getDebugLevel() {
        return debugLevel;
    }

    /**
     * Tells CleverTap that the current Android activity was paused. Add a call to this
     * method in the onPause() method of every activity.
     *
     * @param activity The calling activity
     */
    public void activityPaused(Activity activity) {
        setAppForeground(false);
        // Just update it
        setCurrentActivity(activity);
        appLastSeen = System.currentTimeMillis();
        // Create the runnable, if it is null
        if (sessionTimeoutRunnable == null)
            sessionTimeoutRunnable = new Runnable() {
                @Override
                public void run() {
                    checkTimeoutSession();
                }
            };
        // Remove any existing session timeout runnable object
        getHandlerUsingMainLooper().removeCallbacks(sessionTimeoutRunnable);
        getHandlerUsingMainLooper().postDelayed(sessionTimeoutRunnable,
                Constants.SESSION_LENGTH_MINS * 60 * 1000);
        Logger.v("Foreground activity gone to background");
    }


    private static HashSet<String> inappActivityExclude = null;

    /*package*/ boolean canShowInAppOnActivity() {
        updateBlacklistedActivitySet();

        for (String blacklistedActivity : inappActivityExclude) {
            String currentActivityName = getCurrentActivityName();
            if (currentActivityName != null && currentActivityName.contains(blacklistedActivity)) {
                return false;
            }
        }

        return true;
    }

    private void updateBlacklistedActivitySet() {
        if (inappActivityExclude == null) {
            inappActivityExclude = new HashSet<String>();
            try {
                String activities = ManifestMetaData.getMetaData(context, Constants.LABEL_INAPP_EXCLUDE);
                if (activities != null) {
                    String[] split = activities.split(",");
                    for (String a : split) {
                        inappActivityExclude.add(a.trim());
                    }
                }
            } catch (Throwable t) {
                // Ignore
            }
            Logger.d("In-app notifications will not be shown on " + Arrays.toString(inappActivityExclude.toArray()));
        }
    }

    /**
     * Notifies CleverTap that a new activity was created. Add a call to this method
     * in the onCreate() method of every activity.
     */
    private void notifyActivityChanged(Activity activity) {
        setCurrentActivity(activity);
        activityCount++;

        //noinspection ConstantConditions
        if (activity != null) {
            Logger.v("Activity changed: " + activity.getLocalClassName());
        }

        if (Constants.ENABLE_INAPP) {
            // We MUST loop through the set and do a contains on each and every entry there
            final boolean canShow = canShowInAppOnActivity();

            if (canShow) {
                if (pendingInappRunnable != null) {
                    Logger.v("Found a pending inapp runnable. Scheduling it");
                    getHandlerUsingMainLooper().postDelayed(pendingInappRunnable, 200);
                    pendingInappRunnable = null;
                } else {
                    // Show an in-app notification, if available
                    InAppManager.showNotificationIfAvailable(context);
                }
            } else {
                Logger.d("In-app notifications will not be shown for this activity ("
                        + (activity != null ? activity.getLocalClassName() : "") + ")");
            }
        }

        pushDailyEventsAsync();

        event.pushDeviceDetails();
        SessionManager.activityChanged(getCurrentActivityName());
    }

    /**
     * Tells CleverTap that the current Android activity was resumed. Add a call to this
     * method in the onResume() method of every activity.
     *
     * @param activity The calling activity
     */
    public void activityResumed(Activity activity) {
        setAppForeground(true);
        Activity current = getCurrentActivity();
        boolean newLaunch = current == null;
        // Check if this is a different activity from the last one
        String currentActivityName = getCurrentActivityName();
        if (currentActivityName == null
                || !currentActivityName.equals(activity.getLocalClassName()))
            notifyActivityChanged(activity);

        Logger.v("Background activity in foreground");

        // Initialize push and start a lazy App Launched
        if (newLaunch) {
            initPush();

            getHandlerUsingMainLooper().postDelayed(new Runnable() {
                @Override
                public void run() {
                    // Raise the App Launched event
                    pushAppLaunchedEvent("delayed generic handler");
                }
            }, 500);
        }
    }

    /*package*/ JSONObject getAppLaunchedFields() {
        try {
            final JSONObject evtData = new JSONObject();
            PackageInfo pInfo = context.getPackageManager().getPackageInfo(context.getPackageName(), 0);
            evtData.put("Build", pInfo.versionCode + "");
            evtData.put("Version", pInfo.versionName);
            evtData.put("OS Version", android.os.Build.VERSION.RELEASE);
            evtData.put("SDK Version", BuildInfo.SDK_SVN_REVISION);

            if (locationFromUser != null) {
                evtData.put("Latitude", locationFromUser.getLatitude());
                evtData.put("Longitude", locationFromUser.getLongitude());
            }

            // send up googleAdID
            if (getGoogleAdID() != null) {
                String baseAdIDKey = "GoogleAdID";
                String adIDKey = deviceIsMultiUser() ? MULTI_USER_PREFIX + baseAdIDKey : baseAdIDKey;
                evtData.put(adIDKey, getGoogleAdID());
                evtData.put("GoogleAdIDLimit", isLimitAdTrackingEnabled());
            }

            try {
                // Device data
                String make = Build.MANUFACTURER;
                String model = Build.MODEL;
                model = model.replace(make, "");
                evtData.put("Make", make.trim());
                evtData.put("Model", model.trim());
                evtData.put("Carrier", DeviceInfo.getCarrier());

                final Boolean isWifi = DeviceInfo.isWifiConnected();
                if (isWifi != null) {
                    evtData.put("wifi", isWifi);
                }

                final Boolean isBluetoothEnabled = DeviceInfo.isBluetoothEnabled();
                if (isBluetoothEnabled != null) {
                    evtData.put("BluetoothEnabled", isBluetoothEnabled);
                }

                final String bluetoothVersion = DeviceInfo.getBluetoothVersion();
                if (bluetoothVersion != null) {
                    evtData.put("BluetoothVersion", bluetoothVersion);
                }

                final String radio = DeviceInfo.getNetworkType();
                if(radio != null){
                    evtData.put("Radio", radio);
                }
            } catch (Throwable t) {
                // Ignore
            }

            return evtData;
        } catch (Throwable t) {
            Logger.v("Failed to construct App Launched event", t);
            return new JSONObject();
        }
    }

    // SessionManager/session management

    private void checkTimeoutSession() {
        long now = System.currentTimeMillis();
        if (!isAppForeground()
                && (now - appLastSeen) > Constants.SESSION_LENGTH_MINS * 60 * 1000) {
            Logger.v("Session Timed Out");
            forceDestroySession();
            setCurrentActivity(null);
        }
    }

    private void forceDestroySession() {
        setAppLaunchPushed(false);
        SessionManager.destroySession();
    }

    private void forcePushAppLaunchedEvent(String source) {
        setAppLaunchPushed(false);
        pushAppLaunchedEvent(source);
    }

    synchronized void pushAppLaunchedEvent(String source) {
        if (isAppLaunchPushed()) {
            Logger.v("App Launched has already been triggered. Will not trigger it; source = " + source);
            return;
        } else {
            Logger.v("Firing App Launched event; source = " + source);
        }
        setAppLaunchPushed(true);
        JSONObject event = new JSONObject();
        try {
            event.put("evtName", Constants.APP_LAUNCHED_EVENT);
            event.put("evtData", getAppLaunchedFields());
        } catch (Throwable t) {
            // We won't get here
        }
        QueueManager.queueEvent(context, event, Constants.RAISED_EVENT);
    }

    /**
     * Sends all the events in the event queue.
     */
    public void flush() {
        CommsManager.flushQueueAsync(context);
    }


    // Push Handling

    private static ArrayList<PushType> enabledPushTypes = null;

    private void initPush() {
        enabledPushTypes = DeviceInfo.getEnabledPushTypes();
        if (enabledPushTypes == null) return;
        for (PushType pushType : enabledPushTypes) {
            switch (pushType) {
                case GCM:
                    GcmManager.initializeWithContext(context);
                    break;
                case FCM:
                    FcmManager.initializeWithContext(context);
                    break;
                default:
                    //no-op
                    break;
            }
        }
    }

    /**
     * push the device token outside of the normal course
     */
    private void forcePushDeviceToken(final boolean register) {
        pushDeviceToken(register, true);
    }

    private void pushDeviceToken(final boolean register, final boolean force) {
        if (enabledPushTypes == null) return;
        for (PushType pushType : enabledPushTypes) {
            switch (pushType) {
                case GCM:
                    GcmManager.pushDeviceToken(null, register, force);
                    break;
                case FCM:
                    FcmManager.pushDeviceToken(null, register, force);
                    break;
                default:
                    //no-op
                    break;
            }
        }
    }

    /**
     * Checks whether this notification is from CleverTap.
     *
     * @param extras The payload from the GCM intent
     * @return See {@link NotificationInfo}
     */
    public static NotificationInfo getNotificationInfo(final Bundle extras) {
        if (extras == null) return new NotificationInfo(false, false);

        boolean fromCleverTap = extras.containsKey(NOTIFICATION_TAG);
        boolean shouldRender = fromCleverTap && extras.containsKey("nm");
        return new NotificationInfo(fromCleverTap, shouldRender);
    }

    /**
     * Launches an asynchronous task to create the notification channel from CleverTap
     * <p/>
     * Use this method when implementing your own FCM/GCM handling mechanism. Refer to the
     * SDK documentation for usage scenarios and examples.
     *
     * @param context A reference to an Android context
     * @param channelId A String for setting the id of the notification channel
     * @param channelName  A String for setting the name of the notification channel
     * @param channelDescription A String for setting the description of the notification channel
     * @param importance An Integer value setting the importance of the notifications sent in this channel
     * @param showBadge An boolean value as to whether this channel shows a badge
     *
     */
    public static void createNotificationChannel(final Context context, final String channelId, final CharSequence channelName, final String channelDescription, final int importance, final boolean showBadge) {

        try {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                postAsyncSafely("createNotificationChannel", new Runnable() {
                    @Override
                    public void run() {

                        NotificationManager notificationManager = (NotificationManager) context.getSystemService(NOTIFICATION_SERVICE);
                        NotificationChannel notificationChannel = new NotificationChannel(channelId, channelName, importance);
                        notificationChannel.setDescription(channelDescription);
                        notificationChannel.setShowBadge(showBadge);
                        notificationManager.createNotificationChannel(notificationChannel);
                        Logger.i("Notification channel " + channelName.toString() + " has been created");

                    }
                });
            }
        }catch (Throwable t){
            Logger.v("Failure creating Notification Channel",t);
        }

    }

    /**
     * Launches an asynchronous task to create the notification channel from CleverTap
     * <p/>
     * Use this method when implementing your own FCM/GCM handling mechanism and creating
     * notification channel groups. Refer to the
     * SDK documentation for usage scenarios and examples.
     *
     * @param context A reference to an Android context
     * @param channelId A String for setting the id of the notification channel
     * @param channelName  A String for setting the name of the notification channel
     * @param channelDescription A String for setting the description of the notification channel
     * @param importance An Integer value setting the importance of the notifications sent in this
     *                   channel
     * @param groupId A String for setting the notification channel as a part of a notification
     *                group
     * @param showBadge An boolean value as to whether this channel shows a badge
     */
    public static void createNotificationChannel(final Context context, final String channelId, final CharSequence channelName, final String channelDescription, final int importance, final String groupId, final boolean showBadge) {

        try {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                postAsyncSafely("creatingNotificationChannel", new Runnable() {
                    @Override
                    public void run() {

                        NotificationManager notificationManager = (NotificationManager) context.getSystemService(NOTIFICATION_SERVICE);
                        NotificationChannel notificationChannel = new NotificationChannel(channelId, channelName, importance);
                        notificationChannel.setDescription(channelDescription);
                        notificationChannel.setGroup(groupId);
                        notificationChannel.setShowBadge(showBadge);
                        notificationManager.createNotificationChannel(notificationChannel);
                        Logger.i("Notification channel " + channelName.toString() + " has been created");

                    }
                });
            }
        }catch(Throwable t){
            Logger.v("Failure creating Notification Channel",t);
        }

    }

    /**
     * Launches an asynchronous task to create the notification channel group from CleverTap
     * <p/>
     * Use this method when implementing your own FCM/GCM handling mechanism. Refer to the
     * SDK documentation for usage scenarios and examples.
     *
     * @param context A reference to an Android context
     * @param groupId A String for setting the id of the notification channel group
     * @param groupName  A String for setting the name of the notification channel group
     */
    public static void createNotificationChannelGroup(final Context context, final String groupId, final CharSequence groupName) {

        try {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                postAsyncSafely("creatingNotificationChannelGroup", new Runnable() {
                    @Override
                    public void run() {

                        NotificationManager notificationManager = (NotificationManager) context.getSystemService(NOTIFICATION_SERVICE);
                        notificationManager.createNotificationChannelGroup(new NotificationChannelGroup(groupId, groupName));
                        Logger.i("Notification channel group " + groupName.toString() + " has been created");

                    }
                });
            }
        }catch(Throwable t){
            Logger.v("Failure creating Notification Channel Group",t);
        }

    }

    /**
     * Launches an asynchronous task to delete the notification channel from CleverTap
     * <p/>
     * Use this method when implementing your own FCM/GCM handling mechanism. Refer to the
     * SDK documentation for usage scenarios and examples.
     *
     * @param context A reference to an Android context
     * @param channelId A String for setting the id of the notification channel
     */
    public static void deleteNotificationChannel(final Context context, final String channelId) {

        try {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                postAsyncSafely("deletingNotificationChannel", new Runnable() {
                    @Override
                    public void run() {

                        NotificationManager notificationManager = (NotificationManager) context.getSystemService(NOTIFICATION_SERVICE);
                        notificationManager.deleteNotificationChannel(channelId);
                        Logger.i("Notification channel " + channelId.toString() + " has been deleted");

                    }
                });
            }
        }catch(Throwable t){
            Logger.v("Failure deleting Notification Channel",t);
        }

    }

    /**
     * Launches an asynchronous task to delete the notification channel from CleverTap
     * <p/>
     * Use this method when implementing your own FCM/GCM handling mechanism. Refer to the
     * SDK documentation for usage scenarios and examples.
     *
     * @param context A reference to an Android context
     * @param groupId A String for setting the id of the notification channel group
     */
    public static void deleteNotificationChannelGroup(final Context context, final String groupId) {

        try {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                postAsyncSafely("deletingNotificationChannelGroup", new Runnable() {
                    @Override
                    public void run() {

                        NotificationManager notificationManager = (NotificationManager) context.getSystemService(NOTIFICATION_SERVICE);
                        notificationManager.deleteNotificationChannelGroup(groupId);
                        Logger.i("Notification channel group " + groupId.toString() + " has been deleted");

                    }
                });
            }
        }catch(Throwable t){
            Logger.v("Failure deleting Notification Channel Group",t);
        }

    }



    /**
     * Launches an asynchronous task to download the notification icon from CleverTap,
     * and create the Android notification.
     * <p/>
     * Use this method when implementing your own FCM/GCM handling mechanism. Refer to the
     * SDK documentation for usage scenarios and examples.
     *
     * @param context A reference to an Android context
     * @param extras  The {@link Bundle} object received by the broadcast receiver
     */
    public static void createNotification(final Context context, final Bundle extras) {
        createNotification(context,extras, EMPTY_NOTIFICATION_ID);

    }

    /**
     * Launches an asynchronous task to download the notification icon from CleverTap,
     * and create the Android notification.
     * <p/>
     * If your app is using CleverTap SDK's built in FCM/GCM message handling,
     * this method does not need to be called explicitly.
     * <p/>
     * Use this method when implementing your own FCM/GCM handling mechanism. Refer to the
     * SDK documentation for usage scenarios and examples.
     *
     * @param context A reference to an Android context
     * @param extras  The {@link Bundle} object received by the broadcast receiver
     * @param notificationId A custom id to build a notification
     */
    public static void createNotification(final Context context, final Bundle extras, final int notificationId) {
        //noinspection ConstantConditions
        if (extras == null || extras.get(NOTIFICATION_TAG) == null) {
            return;
        }

        try {
            postAsyncSafely("CleverTapAPI#createNotification", new Runnable() {
                @Override
                public void run() {
                    try {
                        Logger.d("Handling notification: " + extras.toString());
                        // Check if this is a test notification
                        if (extras.containsKey(Constants.DEBUG_KEY)
                                && "y".equals(extras.getString(Constants.DEBUG_KEY))) {
                            int r = (int) (Math.random() * 10);
                            if (r != 8) {
                                // Discard acknowledging this notif
                                return;
                            }
                            JSONObject event = new JSONObject();
                            try {
                                JSONObject actions = new JSONObject();
                                for (String x : extras.keySet()) {
                                    Object value = extras.get(x);
                                    actions.put(x, value);
                                }
                                event.put("evtName", "wzrk_d");
                                event.put("evtData", actions);
                                QueueManager.queueEvent(context, event, Constants.RAISED_EVENT);
                            } catch (JSONException ignored) {
                                // Won't happen
                            }
                            // Drop further processing
                            return;
                        }
                        String silent = extras.getString("wzrk_sn");
                        silent = (silent != null) ? silent : "";
                        String notifTitle = extras.getString("nt");
                        notifTitle = (notifTitle != null) ? notifTitle : "";
                        String notifMessage = extras.getString("nm");
                        notifMessage = (notifMessage != null) ? notifMessage : "";

                        if(silent.equals("true")){
                            if(notifMessage.isEmpty() && notifTitle.isEmpty())
                                triggerSilentNotification(context,extras);
                            else {
                                triggerSilentNotification(context,extras);
                                triggerNotification(context,extras,notifMessage,notifTitle, notificationId);
                            }
                        }else if (silent.equals("false") || silent.isEmpty()){
                            if (notifMessage.isEmpty())
                                return;
                            else if (notifTitle.isEmpty()){
                                notifTitle = context.getApplicationInfo().name;
                                triggerNotification(context,extras,notifMessage,notifTitle, notificationId);
                            }else{
                                triggerNotification(context,extras,notifMessage,notifTitle, notificationId);
                            }
                        }



                    } catch (Throwable t) {
                        // Occurs if the notification image was null
                        // Let's return, as we couldn't get a handle on the app's icon
                        // Some devices throw a PackageManager* exception too
                        Logger.d("Couldn't render notification: ", t);
                    }
                }
            });

        } catch (Throwable t) {
            Logger.d("Failed to process push notification", t);
        }

    }

    /* Trigger a silent Notification */
    public static void triggerSilentNotification(Context context, Bundle extras){
        Logger.d("Handling silent notification: " + extras.toString());
        // TODO where is SilentNotificationReceiver ??
        Intent broadcast  = new Intent("com.clevertap.android.sdk.SilentNotificationReceiver");
        broadcast.putExtras(extras);
        context.sendBroadcast(broadcast);
    }

    /* Trigger Push Notification */
    public static void triggerNotification(Context context, Bundle extras, String notifMessage, String notifTitle, int notificationId){
        String icoPath = extras.getString("ico");
        Intent launchIntent = new Intent(context, CTPushNotificationReceiver.class);

        PendingIntent pIntent;

        // Take all the properties from the notif and add it to the intent
        launchIntent.putExtras(extras);
        launchIntent.removeExtra("wzrk_acts");
        launchIntent.setFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP | Intent.FLAG_ACTIVITY_SINGLE_TOP);
        pIntent = PendingIntent.getBroadcast(context, (int) System.currentTimeMillis(),
                launchIntent, PendingIntent.FLAG_UPDATE_CURRENT);

        NotificationCompat.Style style;
        String bigPictureUrl = extras.getString("wzrk_bp");
        if (bigPictureUrl != null && bigPictureUrl.startsWith("http")) {
            try {
                Bitmap bpMap = Utils.getNotificationBitmap(bigPictureUrl, false, context);

                //noinspection ConstantConditions
                if (bpMap == null)
                    throw new Exception("Failed to fetch big picture!");

                style = new NotificationCompat.BigPictureStyle()
                        .setSummaryText(notifMessage)
                        .bigPicture(bpMap);
            } catch (Throwable t) {
                style = new NotificationCompat.BigTextStyle()
                        .bigText(notifMessage);
                Logger.v("Falling back to big text notification, couldn't fetch big picture", t);
            }
        } else {
            style = new NotificationCompat.BigTextStyle()
                    .bigText(notifMessage);
        }

        int smallIcon;
        try {
            String x = ManifestMetaData.getMetaData(context, Constants.LABEL_NOTIFICATION_ICON);
            if (x == null) throw new IllegalArgumentException();
            smallIcon = context.getResources().getIdentifier(x, "drawable", context.getPackageName());
            if (smallIcon == 0) throw new IllegalArgumentException();
        } catch (Throwable t) {
            smallIcon = DeviceInfo.getAppIconAsIntId(context);
        }

        int priorityInt = NotificationCompat.PRIORITY_DEFAULT;
        String priority = extras.getString("pr");
        if (priority != null) {
            if (priority.equals("high")) {
                priorityInt = NotificationCompat.PRIORITY_HIGH;
            }
            if (priority.equals("max")) {
                priorityInt = NotificationCompat.PRIORITY_MAX;
            }
        }

        // if we have not user set notificationID then try collapse key
        if (notificationId == EMPTY_NOTIFICATION_ID) {
            try {
                Object collapse_key = extras.get("wzrk_ck");
                if(collapse_key != null) {
                    if (collapse_key instanceof Number) {
                        notificationId = ((Number) collapse_key).intValue();
                    } else if (collapse_key instanceof String) {
                        try {
                            notificationId = Integer.parseInt(collapse_key.toString());
                            Logger.d("Converting collapse_key: " + collapse_key + " to notificationId int: " + notificationId);
                        } catch (NumberFormatException e) {
                            notificationId = (collapse_key.toString().hashCode());
                            Logger.d("Converting collapse_key: " + collapse_key + " to notificationId int: " + notificationId);
                        }
                    }
                }
            } catch (NumberFormatException e) {
                // no-op
            }
        } else {
            Logger.d("Have user provided notificationId: " + notificationId + " won't use collapse_key (if any) as basis for notificationId");
        }

        // if after trying collapse_key notification is still empty set to random int
        if (notificationId == EMPTY_NOTIFICATION_ID) {
            notificationId = (int) (Math.random() * 100);
            Logger.d("Setting random notificationId: " + notificationId);
        }

        NotificationCompat.Builder nb;

        if (context.getApplicationInfo().targetSdkVersion >= Build.VERSION_CODES.O){
            String channelId = extras.getString("wzrk_cid", "");
            if (channelId.isEmpty()) {
                Logger.d("ChannelId is empty for notification: " + extras.toString());
            }
            nb = new NotificationCompat.Builder(context,channelId);

            // choices here are Notification.BADGE_ICON_NONE = 0, Notification.BADGE_ICON_SMALL = 1, Notification.BADGE_ICON_LARGE = 2.  Default is  Notification.BADGE_ICON_LARGE
            String badgeIconParam = extras.getString("wzrk_bi", null);
            if (badgeIconParam != null) {
                try {
                    int badgeIconType = Integer.parseInt(badgeIconParam);
                    if (badgeIconType >=0) {
                        nb.setBadgeIconType(badgeIconType);
                    }
                } catch (Throwable t) {
                    // no-op
                }
            }

            String badgeCountParam = extras.getString("wzrk_bc", null);
            if (badgeCountParam != null) {
                try {
                    int badgeCount = Integer.parseInt(badgeCountParam);
                    if (badgeCount >= 0) {
                        nb.setNumber(badgeCount);
                    }
                } catch (Throwable t) {
                    // no-op
                }
            }
        } else {
            nb = new NotificationCompat.Builder(context);
        }

        nb.setContentTitle(notifTitle)
                .setContentText(notifMessage)
                .setContentIntent(pIntent)
                .setAutoCancel(true)
                .setStyle(style)
                .setPriority(priorityInt)
                .setSmallIcon(smallIcon);

        try {
            nb.setLargeIcon(Utils.getNotificationBitmap(icoPath, true, context));
        } catch (PackageManager.NameNotFoundException e) {
            Logger.d("Error setting large notification icon: ", e);
        }

        try {
            if (extras.containsKey("wzrk_sound")) {
                Uri soundUri = null;

                Object o = extras.get("wzrk_sound");

                if ((o instanceof Boolean && (Boolean) o)) {
                    soundUri = RingtoneManager.getDefaultUri(RingtoneManager.TYPE_NOTIFICATION);
                } else if (o instanceof String) {
                    String s = (String) o;
                    if (s.equals("true")) {
                        soundUri = RingtoneManager.getDefaultUri(RingtoneManager.TYPE_NOTIFICATION);
                    } else if (!s.isEmpty()) {
                        if (s.contains(".mp3") || s.contains(".ogg") || s.contains(".wav")) {
                            s = s.substring(0, (s.length() - 4));
                        }
                        soundUri = Uri.parse(ContentResolver.SCHEME_ANDROID_RESOURCE + "://" + context.getPackageName() + "/raw/" + s);
                    }
                }

                if (soundUri != null) {
                    nb.setSound(soundUri);
                }
            }
        } catch (Throwable t) {
            Logger.d("Could not process sound parameter", t);
        }

        // add actions if any
        JSONArray actions = null;
        String actionsString = extras.getString("wzrk_acts");
        if (actionsString != null) {
            try {
                actions = new JSONArray(actionsString);
            } catch (Throwable t) {
                Logger.d("error parsing notification actions: " + t.getLocalizedMessage());
            }
        }

        boolean isCTIntentServiceAvailable = isServiceAvailable(context,
                CTNotificationIntentService.MAIN_ACTION, CTNotificationIntentService.class);

        if (actions != null && actions.length() > 0) {
            for (int i = 0; i < actions.length(); i++) {
                try {
                    JSONObject action = actions.getJSONObject(i);
                    String label = action.optString("l");
                    String dl = action.optString("dl");
                    String ico = action.optString("ico");
                    String id = action.optString("id");
                    boolean autoCancel = action.optBoolean("ac", true);
                    if (label.isEmpty() || id.isEmpty()) {
                        Logger.d("not adding push notification action: action label or id missing");
                        continue;
                    }
                    int icon = 0;
                    if (!ico.isEmpty()) {
                        try {
                            icon = context.getResources().getIdentifier(ico, "drawable", context.getPackageName());
                        } catch (Throwable t) {
                            Logger.d("unable to add notification action icon: " + t.getLocalizedMessage());
                        }
                    }

                    boolean sendToCTIntentService = (autoCancel && isCTIntentServiceAvailable);

                    Intent actionLaunchIntent;
                    if (sendToCTIntentService) {
                        actionLaunchIntent = new Intent(CTNotificationIntentService.MAIN_ACTION);
                        actionLaunchIntent.putExtra("ct_type", CTNotificationIntentService.TYPE_BUTTON_CLICK);
                        if (!dl.isEmpty()) {
                            actionLaunchIntent.putExtra("dl", dl);
                        }
                    } else {
                        if (!dl.isEmpty()) {
                            actionLaunchIntent = new Intent(Intent.ACTION_VIEW, Uri.parse(dl));
                        } else {
                            actionLaunchIntent = context.getPackageManager().getLaunchIntentForPackage(context.getPackageName());
                        }
                    }

                    actionLaunchIntent.putExtras(extras);
                    actionLaunchIntent.removeExtra("wzrk_acts");
                    actionLaunchIntent.putExtra("actionId", id);
                    actionLaunchIntent.putExtra("autoCancel", autoCancel);
                    actionLaunchIntent.putExtra("wzrk_c2a", id);
                    actionLaunchIntent.putExtra("notificationId", notificationId);

                    actionLaunchIntent.setFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP | Intent.FLAG_ACTIVITY_SINGLE_TOP);

                    PendingIntent actionIntent;
                    int requestCode = ((int) System.currentTimeMillis()) + i;
                    if (sendToCTIntentService) {
                        actionIntent = PendingIntent.getService(context, requestCode,
                                actionLaunchIntent, PendingIntent.FLAG_UPDATE_CURRENT);
                    } else {
                        actionIntent = PendingIntent.getActivity(context, requestCode,
                                actionLaunchIntent, PendingIntent.FLAG_UPDATE_CURRENT);
                    }
                    nb.addAction(icon, label, actionIntent);

                } catch (Throwable t) {
                    Logger.d("error adding notification action : " + t.getLocalizedMessage());
                }
            }
        }

        Notification n = nb.build();

        Logger.d("Building notification: " +n.toString() + ", with notificationId: "+String.valueOf(notificationId));

        NotificationManager notificationManager =
                (NotificationManager) context.getSystemService(NOTIFICATION_SERVICE);

        notificationManager.notify(notificationId, n);
    }

    /**
     * This method is used internally.
     */
    public void pushDeepLink(Uri uri) {
        pushDeepLink(uri, false);
    }

    synchronized void pushDeepLink(Uri uri, boolean install) {
        if (uri == null)
            return;

        try {
            JSONObject referrer = UriHelper.getUrchinFromUri(uri);
            referrer.put("referrer", uri.toString());
            if (install) {
                referrer.put("install", true);
            }
            event.pushDeviceDetailsWithExtras(referrer);
        } catch (Throwable t) {
            Logger.v("Failed to push deep link", t);
        }
    }

    public synchronized void pushInstallReferrer(String source, String medium, String campaign) {
        if (source == null && medium == null && campaign == null) return;
        try {
            // If already pushed, don't send it again
            int status = StorageHelper.getInt(context, "app_install_status", 0);
            if (status != 0) {
                Logger.d("Install referrer has already been set. Will not override it");
                return;
            }
            StorageHelper.putInt(context, "app_install_status", 1);

            if (source != null) source = Uri.encode(source);
            if (medium != null) medium = Uri.encode(medium);
            if (campaign != null) campaign = Uri.encode(campaign);

            String uriStr = "wzrk://track?install=true";
            if (source != null) uriStr += "&utm_source=" + source;
            if (medium != null) uriStr += "&utm_medium=" + medium;
            if (campaign != null) uriStr += "&utm_campaign=" + campaign;

            Uri uri = Uri.parse(uriStr);
            pushDeepLink(uri, true);
        } catch (Throwable t) {
            Logger.v("Failed to push install referrer", t);
        }
    }

    private static final HashMap<String, Integer> installReferrerMap = new HashMap<String, Integer>(8);

    public void enablePersonalization() {
        LocalDataStore.setPersonalisationEnabled(context, true);
    }

    public void disablePersonalization() {
        LocalDataStore.setPersonalisationEnabled(context, false);
    }

    public void pushInstallReferrer(Intent intent) {
        try {
            final Bundle extras = intent.getExtras();
            // Preliminary checks
            if (extras == null || !extras.containsKey("referrer")) {
                return;
            }
            final String url;
            try {
                url = URLDecoder.decode(extras.getString("referrer"), "UTF-8");

                Logger.v("Referrer received: " + url);
            } catch (Throwable e) {
                // Could not decode
                return;
            }
            if (url == null) {
                return;
            }
            int now = (int) (System.currentTimeMillis() / 1000);

            if (installReferrerMap.containsKey(url) && now - installReferrerMap.get(url) < 10) {
                Logger.v("Skipping install referrer due to duplicate within 10 seconds");
                return;
            }

            installReferrerMap.put(url, now);

            Uri uri = Uri.parse("wzrk://track?install=true&" + url);

            pushDeepLink(uri, true);
        } catch (Throwable t) {
            // Weird
        }
    }

    /**
     * get the current device location
     * requires Location Permission in AndroidManifest e.g. "android.permission.ACCESS_COARSE_LOCATION"
     * You can then use the returned Location value to update the user profile location in CleverTap via {@link #setLocation(Location)}
     *
     * @return android.location.Location
     */

    public Location getLocation() {
        return _getLocation();
    }

    /**
     * set the user profile location in CleverTap
     * location can then be used for geo-segmentation etc.
     *
     * @param location android.location.Location
     */

    public void setLocation(Location location) {
        _setLocation(location);
    }

    /**
     * @deprecated use {@link #setLocation(Location)} ()} instead.
     */

    @Deprecated
    public void updateLocation(Location location) {
        _setLocation(location);
    }

    private void _setLocation(Location location) {
        if (location == null) return;
        locationFromUser = location;
        Logger.v("Location updated (" + location.getLatitude() + ", " + location.getLongitude() + ")");

        // only queue the location ping if we are in the foreground
        if (!isAppForeground()) return;

        // Queue the ping event to transmit location update to server
        // min 10 second interval between location pings
        final int now = (int) (System.currentTimeMillis() / 1000);
        if (now > (lastLocationPingTime + Constants.LOCATION_PING_INTERVAL_IN_SECONDS)) {
            QueueManager.queueEvent(context, new JSONObject(), Constants.PING_EVENT);
            lastLocationPingTime = now;
            Logger.v("Queuing location ping event for location (" + location.getLatitude() + ", " + location.getLongitude() + ")");
        }
    }

    private Location _getLocation() {
        try {
            LocationManager lm = (LocationManager) context.getSystemService(Context.LOCATION_SERVICE);
            List<String> providers = lm.getProviders(true);
            Location bestLocation = null;
            Location l = null;
            for (String provider : providers) {
                try {
                    l = lm.getLastKnownLocation(provider);
                } catch (SecurityException e) {
                    //no-op
                    Logger.v("Location security exception", e);
                }

                if (l == null) {
                    continue;
                }
                if (bestLocation == null || l.getAccuracy() < bestLocation.getAccuracy()) {
                    bestLocation = l;
                }
            }

            return bestLocation;
        } catch (Throwable t) {
            Logger.v("Couldn't get user's location", t);
            return null;
        }
    }

    /**
     * Returns a unique identifier by which CleverTap identifies this user.
     *
     * @return The user identifier currently being used to identify this user.
     */
    public String getCleverTapID() {
        return DeviceInfo.getDeviceID();
    }

    /**
     * Returns a unique CleverTap identifier suitable for use with install attribution providers.
     *
     * @return The attribution identifier currently being used to identify this user.
     */
    public String getCleverTapAttributionIdentifier() {
        return DeviceInfo.getAttributionID();
    }

    /**
     * Returns the device push token or null
     *
     * @param type com.clevertap.android.sdk.PushType (FCM or GCM)
     * @return String device token or null
     * NOTE: on initial install calling getDevicePushToken may return null, as the device token is
     * not yet available
     * Implement CleverTapAPI.DevicePushTokenRefreshListener to get a callback once the token is
     * available
     */
    public String getDevicePushToken(final PushType type) {
        switch (type) {
            case GCM:
                return GcmManager.getDeviceToken();
            case FCM:
                return FcmManager.getDeviceToken();
            default:
                return null;
        }
    }

    private DevicePushTokenRefreshListener tokenRefreshListener;

    void deviceTokenDidRefresh(String token, PushType type) {
        if (tokenRefreshListener != null) {
            Logger.d("Notifying devicePushTokenDidRefresh: " + token);
            tokenRefreshListener.devicePushTokenDidRefresh(token, type);
        }
    }

    /**
     * Implement to get called back when the device push token is refreshed
     */
    public interface DevicePushTokenRefreshListener {
        /**
         * @param token the device token
         * @param type  the token type com.clevertap.android.sdk.PushType (FCM or GCM)
         */
        void devicePushTokenDidRefresh(String token, PushType type);
    }

    public void setDevicePushTokenRefreshListener(DevicePushTokenRefreshListener listener) {
        tokenRefreshListener = listener;
    }

    // helpers

    private static boolean isServiceAvailable(Context context, String action, Class clazz) {
        final PackageManager packageManager = context.getPackageManager();
        final Intent intent = new Intent(action);
        List resolveInfo =
                packageManager.queryIntentServices(intent, 0);
        if (resolveInfo.size() > 0) {
            Logger.v("" + clazz.getName() + " is available");
            return true;
        }
        Logger.v("" + clazz.getName() + " is NOT available");
        return false;
    }
}
