/*
 * Author: Jude Pereira
 * Copyright (c) 2014
 */

package com.clevertap.android.sdk;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.SharedPreferences;
import com.clevertap.android.sdk.exceptions.CleverTapMetaDataNotFoundException;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * Provides methods to manipulate the event queue.
 */
final class QueueManager {
    private static final Boolean lock = true;
    private static Runnable commsRunnable = null;

    /**
     * Returns and empties the event queue.
     *
     * @param context The Android context
     * @return The queue, of type {@link JSONArray}
     */
    @SuppressLint("CommitPrefEdits")
    static JSONArray getAndRemoveAll(Context context) {
        if (CleverTapAPI.isUsingSegment()) {
            final String deviceID = StorageHelper.getString(context, Constants.DEVICE_ID_TAG, null);
            if (deviceID == null || deviceID.trim().equals("")) {
                // Return an empty array, if the ID has not been set yet
                Logger.logFine("No device ID available from Segment yet");
                return new JSONArray();
            }
        }

        synchronized (lock) {
            SharedPreferences prefs = StorageHelper.getPreferences(context);
            try {
                JSONArray all = new JSONArray(prefs.getString("events", "[]"));
                SharedPreferences.Editor editor = prefs.edit().remove("events");
                StorageHelper.persist(editor);
                return all;
            } catch (JSONException e) {
                return new JSONArray();
            }
        }
    }

    /**
     * Adds all the events present in the argument all.
     *
     * @param context The Android context
     * @param all     A {@link JSONArray} containing all the events
     */
    @SuppressLint("CommitPrefEdits")
    static void restoreQueue(Context context, JSONArray all) {
        synchronized (lock) {
            SharedPreferences prefs = StorageHelper.getPreferences(context);
            try {
                JSONArray existingQueue = new JSONArray(prefs.getString("events", "[]"));
                // Since "all" must be definitely old, keep them at the beginning of the queue
                int len = existingQueue.length();
                for (int i = 0; i < len; i++)
                    all.put(existingQueue.get(i));
                SharedPreferences.Editor editor = prefs.edit().putString("events", all.toString());
                StorageHelper.persist(editor);
            } catch (JSONException e) {
                // We won't get here
            }
        }
    }

    private static String mDeviceID = null;

    /**
     * Adds a new event to the queue, to be sent later.
     *
     * @param context   The Android context
     * @param event     The event to be queued
     * @param eventType The type of event to be queued
     */
    @SuppressLint("CommitPrefEdits")
    static void addToQueue(final Context context, JSONObject event, int eventType) {
        synchronized (lock) {
            SharedPreferences prefs = StorageHelper.getPreferences(context);
            try {
                // Create session first
                int session = SessionManager.getCurrentSession();
                if (session == 0) {
                    session = SessionManager.createSession(context);
                }

                // Raise App Launched if not raised yet
                if (CleverTapAPI.isAppForeground() && eventType == Constants.RAISED_EVENT && !SessionManager.isAppLaunchedBeenPushed()) {
                    String evtName = event.getString("evtName");
                    if (!evtName.equals(Constants.NOTIFICATION_CLICKED_EVENT_NAME)) {
                        CleverTapAPI wr = CleverTapAPI.getInstance(context);
                        wr.pushAppLaunchedEvent("Queue Manager - addToQueue");
                    }
                }

                if (mDeviceID == null) {
                    mDeviceID = prefs.getString(Constants.DEVICE_ID_TAG, null);
                }
                String accountId, token;
                try {
                    accountId = ManifestMetaData.getMetaData(context, Constants.LABEL_ACCOUNT_ID);
                    token = ManifestMetaData.getMetaData(context, Constants.LABEL_TOKEN);
                } catch (CleverTapMetaDataNotFoundException e) {
                    // No account ID/token, be gone now!
                    Logger.logFine("Account ID/token not found, will not add to queue");
                    return;
                }
                int activityCount = CleverTapAPI.activityCount;
                activityCount = activityCount == 0 ? 1 : activityCount;
                String type;
                if (eventType == Constants.PAGE_EVENT) {
                    type = "page";
                } else if (eventType == Constants.PING_EVENT) {
                    type = "ping";
                    attachMeta(event, context);
                } else if (eventType == Constants.PROFILE_EVENT) {
                    type = "profile";
                } else if (eventType == Constants.DATA_EVENT) {
                    type = "data";
                } else {
                    type = "event";
                }

                // Complete the received event with the other params
                event.put("id", accountId);

                String currentActivityName = CleverTapAPI.getInstance(context).getCurrentActivityName();
                if (currentActivityName != null) {
                    event.put("n", currentActivityName);
                }

                event.put("s", session);
                if (mDeviceID != null && !mDeviceID.equals("")) {
                    event.put("g", mDeviceID);
                }
                event.put("pg", activityCount);
                event.put("tk", token);
                event.put("type", type);
                event.put("ep", System.currentTimeMillis() / 1000);
                event.put("f", SessionManager.isFirstSession());
                event.put("lsl", SessionManager.getLastSessionLength());
                attachLastSessionActivityTrailAndPackageNameIfRequired(context, event);

                // Report any pending validation error
                if (CleverTapAPI.pendingValidationResult != null) {
                    event.put(Constants.ERROR_KEY, getErrorObject(CleverTapAPI.pendingValidationResult));
                    // Reset the pending error to null
                    CleverTapAPI.pendingValidationResult = null;
                }

                LocalDataStore.setDataSyncFlag(context, event);

                // Add the event to the queue, and save it back
                JSONArray queuedEvents = new JSONArray(prefs.getString("events", "[]"));
                int len = queuedEvents.length();
                if (len > 50) {
                    JSONArray trimmedQueuedEvents = new JSONArray();
                    // Trim down the list to 40, so that this loop needn't run for the next 10 events
                    // Hence, skip the first 10 elements
                    for (int i = 10; i < len; i++)
                        trimmedQueuedEvents.put(queuedEvents.get(i));
                    trimmedQueuedEvents.put(event);
                    queuedEvents = trimmedQueuedEvents;
                } else {
                    queuedEvents.put(event);
                }

                SharedPreferences.Editor editor = prefs.edit();
                editor.putString("events", queuedEvents.toString());
                StorageHelper.persist(editor);
                if (CleverTapAPI.getDebugLevel() == Constants.DEBUG_FINEST) {
                    Logger.logFine("New event queued: " + event.toString());
                }

                updateLocalStoreAsync(context, event, eventType);
                setupSend(context);
            } catch (Throwable e) {
                Logger.error("FATAL: Queuing events failed!", e);
            }
        }
    }

    private static void updateLocalStoreAsync(final Context context, final JSONObject event, final int type) {
        if (type == Constants.PROFILE_EVENT || type == Constants.RAISED_EVENT) {
            CleverTapAPI.postAsyncSafely("QueueManager#updateLocalStoreAsync", context, new Runnable() {
                @Override
                public void run() {
                    LocalDataStore.persistEvent(context, event, type);
                }
            });
        }
    }

    private static void setupSend(final Context context) {
        if (commsRunnable == null)
            commsRunnable = new Runnable() {
                @Override
                public void run() {
                    new CommsManager().drainQueueAsync(context);
                }
            };
        // Cancel any outstanding send runnables, and issue a new delayed one
        CleverTapAPI.getHandlerUsingMainLooper().removeCallbacks(commsRunnable);
        CleverTapAPI.getHandlerUsingMainLooper().postDelayed(commsRunnable, Constants.PUSH_DELAY_MS);

        Logger.logFine("Posted delayed runnable to send queue");
    }

    private static JSONObject getErrorObject(ValidationResult vr) {
        JSONObject error = new JSONObject();
        try {
            error.put("c", vr.getErrorCode());
            error.put("d", vr.getErrorDesc());
        } catch (JSONException e) {
            // Won't reach here
        }
        return error;
    }

    /**
     * Attaches meta info about the current state of the device to an event.
     * Typically, this meta is added only to the ping event.
     */
    private static void attachMeta(final JSONObject o, final Context context) {
        // Memory consumption
        try {
            o.put("mc", Utils.getMemoryConsumption());
        } catch (Throwable t) {
            // Ignore
        }

        // Attach how many services this app has
        try {
            o.put("sc", Utils.getServiceCount(context));
        } catch (Throwable t) {
            // Ignore
        }

        // Attach the network type
        try {
            o.put("nt", Utils.getCurrentNetworkType(context));
        } catch (Throwable t) {
            // Ignore
        }

        // Attach the installed apps count
        try {
            o.put("oa", Utils.getInstalledAppsCount(context));
        } catch (Throwable t) {
            // Ignore
        }
    }

    private static void attachLastSessionActivityTrailAndPackageNameIfRequired(final Context context, final JSONObject event) {
        try {
            final String type = event.getString("type");
            // Send it only for app launched events
            if ("event".equals(type) && Constants.APP_LAUNCHED_EVENT.equals(event.getString("evtName"))) {
                final JSONArray lastSessionActivityTrail = SessionManager.getLastSessionActivityTrail();
                // The above will never be null
                event.put("lsat", lastSessionActivityTrail);
                event.put("pai", context.getPackageName());
            }
        } catch (Throwable t) {
            // Ignore
        }
    }
}
