package com.clevertap.android.sdk;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.SharedPreferences;
import org.json.JSONArray;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * User: Jude Pereira
 * Date: 07/07/2015
 * Time: 12:23
 */
final class LocalDataStore {
    private static final String profileNamespace = "local_profile", eventNamespace = "local_events";
    private static final String PERSONALISATION_ENABLED = "personalisationEnabledBool";

    /**
     * Whenever a profile field is updated, in the session, put it here.
     * The value must be an epoch until how long it is valid for (using existing TTL).
     * <p/>
     * When upstream updates come in, check whether or not to update the field.
     */
    private static final HashMap<String, Integer> PROFILE_EXPIRY_MAP = new HashMap<String, Integer>();

    static void setPersonalisationEnabled(final Context context, boolean enable) {
        if (CleverTapAPI.isUsingSegment()) return;

        StorageHelper.putBoolean(context, PERSONALISATION_ENABLED, enable);
    }

    static boolean isPersonalisationEnabled(final Context context) {
        return StorageHelper.getBoolean(context, PERSONALISATION_ENABLED, false);
    }

    static void setDataSyncFlag(Context context, JSONObject event) {
        try {
            // Check the personalisation flag
            boolean enablePersonalisation = StorageHelper.getBoolean(context, PERSONALISATION_ENABLED, false);
            if (!enablePersonalisation) {
                event.put("dsync", false);
                return;
            }

            // Always request a dsync when the App Launched event is recorded
            final String eventType = event.getString("type");
            if ("event".equals(eventType)) {
                final String evtName = event.getString("evtName");
                if (Constants.APP_LAUNCHED_EVENT.equals(evtName)) {
                    Logger.logFine("Local cache needs to be updated (triggered by App Launched)");
                    event.put("dsync", true);
                    return;
                }
            }

            // If a profile event, then blindly set it to true
            if ("profile".equals(eventType)) {
                event.put("dsync", true);
                Logger.logFine("Local cache needs to be updated (profile event)");
                return;
            }

            // Default to expire in 20 minutes
            final int now = (int) (System.currentTimeMillis() / 1000);
            int expiresIn = StorageHelper.getInt(context, "local_cache_expires_in", 20 * 60);
            int lastUpdate = StorageHelper.getInt(context, "local_cache_last_update", now);

            if (lastUpdate + expiresIn < now) {
                event.put("dsync", true);
                Logger.logFine("Local cache needs to be updated");
            } else {
                event.put("dsync", false);
                Logger.logFine("Local cache doesn't need to be updated");
            }
        } catch (Throwable t) {
            Logger.error("Failed to sync with upstream", t);
        }
    }

    private static int getTTL(final Context context) {
        return StorageHelper.getInt(context, "local_cache_expires_in", 20 * 60);
    }

    private static void setTTL(final Context context, final int ttl) {
        StorageHelper.putInt(context, "local_cache_expires_in", ttl);
    }

    static void persistEvent(Context context, JSONObject event, int type) {
        if (CleverTapAPI.isUsingSegment()) return;

        try {
            if (event == null) return;

            if (type == Constants.RAISED_EVENT) {
                persistEvent(context, event);
            } else if (type == Constants.PROFILE_EVENT) {
                persistProfile(context, event, false);
            }
        } catch (Throwable t) {
            Logger.error("Failed to sync with upstream", t);
        }
    }

    static void syncWithUpstream(Context context, JSONObject response) {
        try {
            JSONObject eventUpdates = null;
            JSONObject profileUpdates = null;

            if (!response.has("evpr")) return;

            JSONObject evpr = response.getJSONObject("evpr");
            if (evpr.has("profile")) {
                JSONObject profile = evpr.getJSONObject("profile");
                if (profile.has("_custom")) {
                    JSONObject custom = profile.getJSONObject("_custom");
                    profile.remove("_custom");
                    Iterator keys = custom.keys();
                    while (keys.hasNext()) {
                        String next = keys.next().toString();
                        profile.put(next, custom.get(next));
                    }
                }

                JSONObject wrapper = new JSONObject();
                wrapper.put("profile", profile);
                profileUpdates = persistProfile(context, wrapper, true);
            }

            if (evpr.has("events")) {
                eventUpdates = syncEventsFromUpstream(context, evpr.getJSONObject("events"));
            }

            if (evpr.has("expires_in")) {
                int expiresIn = evpr.getInt("expires_in");
                setTTL(context, expiresIn);
            }

            StorageHelper.putInt(context, "local_cache_last_update", (int) (System.currentTimeMillis() / 1000));

            if (profileUpdates != null || eventUpdates != null) {
                JSONObject updates = new JSONObject();

                if (profileUpdates != null) {
                    updates.put("profile", profileUpdates);
                }

                if (eventUpdates != null) {
                    updates.put("events", eventUpdates);
                }

                SyncListener syncListener = CleverTapAPI.getInstance(context).getSyncListener();
                if (syncListener != null) {
                    try {
                        syncListener.profileDataUpdated(updates);
                    } catch (Throwable t) {
                        Logger.error("Execution of sync listener failed", t);
                    }
                }
            }
        } catch (Throwable t) {
            Logger.error("Failed to sync with upstream", t);
        }
    }

    @SuppressLint("CommitPrefEdits")
    private static JSONObject syncEventsFromUpstream(Context context, JSONObject events) {
        try {
            JSONObject eventUpdates = null;
            SharedPreferences prefs = StorageHelper.getPreferences(context, eventNamespace);
            Iterator keys = events.keys();
            SharedPreferences.Editor editor = prefs.edit();
            while (keys.hasNext()) {
                String event = keys.next().toString();
                String encoded = prefs.getString(event, encodeEventDetails(0, 0, 0));

                EventDetail ed = decodeEventDetails(event, encoded);

                JSONArray upstream = events.getJSONArray(event);
                if (upstream == null || upstream.length() < 3) {
                    Logger.error("Corrupted upstream event detail");
                    continue;
                }

                int upstreamCount, first, last;
                try {
                    upstreamCount = upstream.getInt(0);
                    first = upstream.getInt(1);
                    last = upstream.getInt(2);
                } catch (Throwable t) {
                    Logger.error("Failed to parse upstream event message: " + upstream.toString());
                    continue;
                }

                if (upstreamCount > ed.getCount()) {
                    editor.putString(event, encodeEventDetails(first, last, upstreamCount));
                    Logger.logFine("Accepted update for event " + event + " from upstream");

                    try {
                        if (eventUpdates == null) {
                            eventUpdates = new JSONObject();
                        }

                        JSONObject evUpdate = new JSONObject();

                        JSONObject countUpdate = new JSONObject();
                        countUpdate.put("oldValue", ed.getCount());
                        countUpdate.put("newValue", upstreamCount);
                        evUpdate.put("count", countUpdate);

                        JSONObject firstUpdate = new JSONObject();
                        firstUpdate.put("oldValue", ed.getFirstTime());
                        firstUpdate.put("newValue", upstream.getInt(1));
                        evUpdate.put("firstTime", firstUpdate);

                        JSONObject lastUpdate = new JSONObject();
                        lastUpdate.put("oldValue", ed.getLastTime());
                        lastUpdate.put("newValue", upstream.getInt(2));
                        evUpdate.put("lastTime", lastUpdate);

                        eventUpdates.put(event, evUpdate);

                    } catch (Throwable t) {
                        Logger.error("Couldn't set event updates", t);
                    }

                } else {
                    Logger.logFine("Rejected update for event " + event + " from upstream");
                }
            }
            StorageHelper.persist(editor);
            return eventUpdates;
        } catch (Throwable t) {
            Logger.error("Couldn't sync events from upstream", t);
            return null;
        }
    }

    @SuppressLint("CommitPrefEdits")
    private static JSONObject persistProfile(Context context, JSONObject event, final boolean syncFromUpstream) {
        try {
            JSONObject profileUpdates = null;

            JSONObject profile = event.getJSONObject("profile");
            if (profile == null || profile.length() == 0) return null;

            SharedPreferences prefs = StorageHelper.getPreferences(context, profileNamespace);
            SharedPreferences.Editor editor = prefs.edit();

            final int now = (int) (System.currentTimeMillis() / 1000);
            final int validUntil = now + getTTL(context);
            final Iterator keys = profile.keys();
            while (keys.hasNext()) {
                try {
                    String key = keys.next().toString();
                    String oldValue = prefs.getString(key, null);
                    String newValue = profile.get(key).toString();

                    if (syncFromUpstream) {
                        // Check for validity
                        Integer keyValidUntil = PROFILE_EXPIRY_MAP.get(key);
                        if (keyValidUntil != null && now < keyValidUntil) {
                            // We shouldn't accept the upstream value, as our map
                            // forces us to use the local
                            Logger.logFine("Rejecting upstream value for key " + key + " " +
                                    "because our local cache prohibits it");
                            continue;
                        }
                    } else {
                        // Update the map
                        PROFILE_EXPIRY_MAP.put(key, validUntil);
                    }

                    editor.putString(key, profile.get(key).toString());

                    if (oldValue == null || !newValue.equals(oldValue)) {
                        try {

                            if (profileUpdates == null) {
                                profileUpdates = new JSONObject();
                            }

                            JSONObject keyUpdates = new JSONObject();
                            if(oldValue != null) {
                                keyUpdates.put("oldValue", oldValue);
                            }    

                            keyUpdates.put("newValue", newValue);

                            profileUpdates.put(key, keyUpdates);

                        } catch (Throwable t) {
                            Logger.logFine("Failed to set profile updates", t);
                        }
                    }

                } catch (Throwable t) {
                    Logger.logFine("Failed to update profile field", t);
                }
            }
            StorageHelper.persist(editor);
            return profileUpdates;
        } catch (Throwable t) {
            Logger.error("Failed to persist profile locally", t);
            return null;
        }
    }

    @SuppressLint("CommitPrefEdits")
    private static void persistEvent(Context context, JSONObject event) {
        try {
            String evtName = event.getString("evtName");
            if (evtName == null) return;

            SharedPreferences prefs = StorageHelper.getPreferences(context, eventNamespace);

            int now = (int) (System.currentTimeMillis() / 1000);

            String encoded = prefs.getString(evtName, encodeEventDetails(now, now, 0));
            EventDetail ed = decodeEventDetails(evtName, encoded);

            String updateEncoded = encodeEventDetails(ed.getFirstTime(), now, ed.getCount() + 1);
            SharedPreferences.Editor editor = prefs.edit();
            editor.putString(evtName, updateEncoded);
            StorageHelper.persist(editor);
        } catch (Throwable t) {
            Logger.error("Failed to persist event locally", t);
        }
    }

    private static String encodeEventDetails(int first, int last, int count) {
        return count + "|" + first + "|" + last;
    }

    private static EventDetail decodeEventDetails(String name, String encoded) {
        if (encoded == null) return null;

        String[] parts = encoded.split("\\|");
        return new EventDetail(Integer.parseInt(parts[0]), Integer.parseInt(parts[1]),
                Integer.parseInt(parts[2]), name);
    }

    static Map<String, EventDetail> getEventHistory(Context context) {
        try {
            SharedPreferences prefs = StorageHelper.getPreferences(context, eventNamespace);
            Map<String, ?> all = prefs.getAll();
            Map<String, EventDetail> out = new HashMap<String, EventDetail>();
            for (String eventName : all.keySet()) {
                out.put(eventName, decodeEventDetails(eventName, all.get(eventName).toString()));
            }
            return out;
        } catch (Throwable t) {
            Logger.error("Failed to retrieve local event history", t);
            return null;
        }
    }

    static EventDetail getEventDetail(Context context, String eventName) {
        try {
            if (!isPersonalisationEnabled(context)) return null;

            SharedPreferences prefs = StorageHelper.getPreferences(context, eventNamespace);
            return decodeEventDetails(eventName, prefs.getString(eventName, null));
        } catch (Throwable t) {
            Logger.error("Failed to retrieve local event detail", t);
            return null;
        }
    }

    static String getProfileProperty(Context context, String key) {
        try {
            if (!isPersonalisationEnabled(context)) return null;

            SharedPreferences prefs = StorageHelper.getPreferences(context, profileNamespace);
            return prefs.getString(key, null);
        } catch (Throwable t) {
            Logger.error("Failed to retrieve local profile property", t);
            return null;
        }
    }
}
