/*
 * Author: Jude Pereira
 * Copyright (c) 2014
 */

package com.clevertap.android.sdk;

import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.os.Bundle;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;

/**
 * Manages in-app notifications.
 */
public final class InAppManager {
    private InAppManager() {
        // No instantiation
    }

    private static final HashMap<Object, Integer> targetCounts = new HashMap<Object, Integer>(3);

    static void clearTargetCounts() {
        targetCounts.clear();
    }

    static HashMap<Object, Integer> getTargetCounts() {
        return targetCounts;
    }

    static int getTotalShown() {
        Collection<Integer> values = targetCounts.values();
        Iterator<Integer> i = values.iterator();
        int total = 0;
        while (i.hasNext()) {
            total += i.next();
        }

        return total;
    }

    static int getCountsForTarget(String key) {
        Integer c = targetCounts.get(key);
        if (c == null) return 0;

        return c;
    }

    static void incrementCountForTarget(Object key) {
        if (key == null) {
            Logger.logFine("Null key for targetCounts increment");
            return;
        }
        Integer c = targetCounts.get(key);
        if (c == null) c = 0;

        c++;

        targetCounts.put(key, c);
    }

    static void doNotShowTarget(Object key) {
        targetCounts.put(key, -1);
    }

    private static void createAndShow(Context context, Bundle notif) {
        if (!CleverTapAPI.isAppForeground()) {
            Logger.error("Cannot show in-app notification. App isn't in foreground. " +
                    "Please check your integration if your app is actually in foreground.");
            return;
        }

        Logger.logFine("Will show new notification shortly: " + notif.toString());
        if (!validateNotifBundle(notif)) {
            // Discard - the format wasn't as expected

            Logger.logFine("Notification was not formatted correctly. Discarding");
            return;
        }

        Object key = notif.get(Constants.NOTIFICATION_ID_TAG);
        incrementCountForTarget(key);

        Intent i = new Intent(context, InAppNotificationActivity.class);
        i.putExtras(notif);
        // Start our transparent activity
        try {
            if (CleverTapAPI.currentActivity == null) {
                throw new IllegalStateException("Current activity reference not found");
            }

            CleverTapAPI.currentActivity.startActivity(i);
        } catch (Throwable t) {
            Logger.error("InAppManager: Please verify the integration of your app." +
                    " It is not setup to support in-app notifications yet.", t);
        }
    }

    static void processResponseAsync(final JSONObject response, final Context context) {
        CleverTapAPI.postAsyncSafely("InAppManager#processResponseAsync", context, new Runnable() {
            @Override
            public void run() {
                try {
                    processResponse(response, context);
                } catch (Throwable t) {
                    Logger.error("Failed to process response!", t);
                }
            }
        });
    }

    private static void processResponse(final JSONObject response, final Context context) {
        try {
            Logger.logFine("InAppManager: Processing response");

            if (!response.has("inapp_notifs")) {
                Logger.logFine("InAppManager: Response JSON object doesn't contain the inapp key, bailing");
                return;
            }

            if (response.has(Constants.INAPP_MAX_PER_SESSION) && response.get(Constants.INAPP_MAX_PER_SESSION) instanceof Integer) {
                int imc = response.getInt(Constants.INAPP_MAX_PER_SESSION);
                StorageHelper.putInt(context, Constants.INAPP_MAX_PER_SESSION, imc);
            }

            JSONArray inappNotifs;
            try {
                inappNotifs = response.getJSONArray(Constants.INAPP_JSON_RESPONSE_KEY);
            } catch (JSONException e) {
                Logger.logFine("InAppManager: In-app key didn't contain a valid JSON array");
                return;
            }

            // Add all the new notifications to the queue
            SharedPreferences prefs = StorageHelper.getPreferences(context);
            SharedPreferences.Editor editor = prefs.edit();
            try {
                JSONArray inappsFromPrefs = new JSONArray(prefs.getString(Constants.PREFS_INAPP_KEY, "[]"));

                // Now add the rest of them :)
                if (inappNotifs != null && inappNotifs.length() > 0) {
                    for (int i = 0; i < inappNotifs.length(); i++) {
                        try {
                            JSONObject inappNotif = inappNotifs.getJSONObject(i);
                            inappsFromPrefs.put(inappNotif);
                        } catch (JSONException e) {
                            Logger.logFine("InAppManager: Malformed inapp notification");
                        }
                    }
                }

                // Commit all the changes
                editor.putString(Constants.PREFS_INAPP_KEY, inappsFromPrefs.toString());
                StorageHelper.persist(editor);
            } catch (Throwable e) {
                Logger.logFine("InAppManager: Failed to parse the in-app notifications properly");
                Logger.error("InAppManager: Reason: " + e.getMessage(), e);
            }
            // Fire the most recent notification, if any
            showNotificationIfAvailable(context);
        } catch (Throwable t) {
            Logger.error("InAppManager: Failed to parse response", t);
        }
    }

    public static void showNotificationIfAvailable(Context context) {
        SharedPreferences prefs = StorageHelper.getPreferences(context);
        try {
            JSONArray inapps = new JSONArray(prefs.getString(Constants.PREFS_INAPP_KEY, "[]"));
            if (inapps.length() < 1) {
                return;
            }

            ArrayList<Integer> indicesToIgnore = new ArrayList<Integer>();
            int qualifiedIndex = -1;

            for (int i = 0; i < inapps.length(); i++) {
                JSONObject inapp = inapps.getJSONObject(i);
                JSONObject w = inapp.getJSONObject(Constants.INAPP_WINDOW);

                String key = "test_notification";
                if (inapp.has(Constants.NOTIFICATION_ID_TAG)) {
                    key = inapp.getString(Constants.NOTIFICATION_ID_TAG);
                }

                int mdc;
                if (w.has(Constants.INAPP_MAX_DISPLAY_COUNT) && w.get(Constants.INAPP_MAX_DISPLAY_COUNT) instanceof Integer) {
                    mdc = w.getInt(Constants.INAPP_MAX_DISPLAY_COUNT);
                } else {
                    mdc = Integer.MAX_VALUE;
                }

                // mdc = -1 when this target is to be shown always
                if (mdc == -1) {
                    qualifiedIndex = i;
                    break;
                }

                int imc = StorageHelper.getInt(context, Constants.INAPP_MAX_PER_SESSION, 1);

                int targetShownCount = getCountsForTarget(key);
                if (targetShownCount != -1 && getTotalShown() < imc && targetShownCount < mdc) {
                    qualifiedIndex = i;
                    break;
                } else {
                    indicesToIgnore.add(i);
                }
            }

            if (qualifiedIndex != -1) {
                createAndShow(context, getBundleFromJsonObject(inapps.getJSONObject(qualifiedIndex)));
            }

            // JSON array doesn't have the feature to remove a single element,
            // so we have to copy over the entire array, but the first element
            JSONArray inappsUpdated = new JSONArray();
            for (int i = 0; i < inapps.length(); i++) {
                if (indicesToIgnore.contains(i) || qualifiedIndex == i) continue;

                inappsUpdated.put(inapps.get(i));
            }
            SharedPreferences.Editor editor = prefs.edit().putString(Constants.PREFS_INAPP_KEY, inappsUpdated.toString());
            StorageHelper.persist(editor);
        } catch (Throwable t) {
            // We won't get here
            Logger.logFine("InAppManager: Couldn't parse JSON array string from prefs", t);
        }
    }

    private static boolean validateNotifBundle(Bundle notif) {

        // Check that either xdp or xp is set
        if (!isKeyValid(notif, Constants.INAPP_X_DP, Integer.class))
            if (!isKeyValid(notif, Constants.INAPP_X_PERCENT, Integer.class))
                return false;

        // Check that either ydp or yp is set
        if (!isKeyValid(notif, Constants.INAPP_Y_DP, Integer.class))
            if (!isKeyValid(notif, Constants.INAPP_Y_PERCENT, Integer.class))
                return false;

        // Check that dk is set
        if (!(isKeyValid(notif, Constants.INAPP_NOTIF_DARKEN_SCREEN, Boolean.class)))
            return false;

        // Check that sc is set
        if (!(isKeyValid(notif, Constants.INAPP_NOTIF_SHOW_CLOSE, Boolean.class)))
            return false;

        // Check that html is set
        if (!(isKeyValid(notif, Constants.INAPP_DATA_TAG, String.class)))
            return false;

        // Check that pos contains the right value
        if ((isKeyValid(notif, Constants.INAPP_POSITION, String.class))) {
            char pos = notif.getString(Constants.INAPP_POSITION).charAt(0);
            switch (pos) {
                case Constants.INAPP_POSITION_TOP:
                    break;
                case Constants.INAPP_POSITION_RIGHT:
                    break;
                case Constants.INAPP_POSITION_BOTTOM:
                    break;
                case Constants.INAPP_POSITION_LEFT:
                    break;
                case Constants.INAPP_POSITION_CENTER:
                    break;
                default:
                    return false;
            }
        } else
            return false;

        // All is good
        //noinspection Contract
        return true;
    }

    private static boolean isKeyValid(Bundle b, String key, Class<?> type) {
        return b.containsKey(key) && b.get(key).getClass().equals(type);
    }

    private static Bundle getBundleFromJsonObject(JSONObject notif) {
        Bundle b = new Bundle();
        Iterator iterator = notif.keys();
        while (iterator.hasNext()) {
            String key = (String) iterator.next();
            try {
                Object value = notif.get(key);
                if (value instanceof String)
                    b.putString(key, (String) value);
                else if (value instanceof Character)
                    b.putChar(key, (Character) value);
                else if (value instanceof Integer)
                    b.putInt(key, (Integer) value);
                else if (value instanceof Float)
                    b.putFloat(key, (Float) value);
                else if (value instanceof Double)
                    b.putDouble(key, (Double) value);
                else if (value instanceof Long)
                    b.putLong(key, (Long) value);
                else if (value instanceof Boolean)
                    b.putBoolean(key, (Boolean) value);
                else if (value instanceof JSONObject)
                    b.putAll(getBundleFromJsonObject((JSONObject) value));
            } catch (JSONException e) {
                // Strange value?

                Logger.logFine("InAppManager: Key had foreign object. Discarding");
            }
        }
        return b;
    }
}
