/*
 * Author: Jude Pereira
 * Copyright (c) 2014
 */

package com.clevertap.android.sdk;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.SharedPreferences;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.os.Environment;
import android.support.v4.content.ContextCompat;
import android.telephony.TelephonyManager;
import com.clevertap.android.sdk.exceptions.CleverTapPermissionsNotSatisfied;

import java.io.*;

import static com.clevertap.android.sdk.Logger.logFine;

/**
 * Provides various methods to access the device.
 */
final class DeviceInfo {
    private static final String separator = "\\|";
    private static final File targetDeviceIdFile = new File(Environment.getExternalStorageDirectory()
            + "/Android/data/com.wizrocket/device_id");

    /**
     * Updates the device ID, if required.
     * <p>
     * This is used internally by the SDK, there is no need to call this explicitly.
     * </p>
     *
     * @param context The Android {@link Context}
     */
    @SuppressLint("CommitPrefEdits")
    static void updateDeviceIdIfRequired(Context context) {
        SharedPreferences prefs = StorageHelper.getPreferences(context);

        String deviceID = prefs.getString(Constants.DEVICE_ID_TAG, null);

        // If the device ID is already present, simply write the device ID and return
        if (deviceID != null) {
            writeDeviceIdToFileAsync(context, targetDeviceIdFile, deviceID);
            return;
        }

        // See if another installation or something has already generated a key previously
        try {
            if (targetDeviceIdFile.exists() && targetDeviceIdFile.isFile()) {

                BufferedReader reader = null;

                try {
                    reader = new BufferedReader((new FileReader(targetDeviceIdFile)));
                    // The very first line itself will be the device ID
                    String id = reader.readLine();
                    // The ID read is in the form "deviceId|177"
                    // 177 is the checksum
                    if (id != null && !id.equals("") && !isDeviceIdCorrupted(id)) {
                        // Perfectly safe to do this at this point
                        id = id.split(separator)[0];
                        SharedPreferences.Editor editor = prefs.edit().putString(Constants.DEVICE_ID_TAG, id);
                        StorageHelper.persist(editor);
                        logFine("Restored existing GUID!");
                        return;
                    }
                } catch (FileNotFoundException e) {
                    // Will never reach here
                } catch (Throwable ex) {
                    Logger.logFine("IOException when attempting to read device ID from file: " + ex);
                } finally {
                    if (reader != null) {
                        try {
                            reader.close();
                        } catch (Throwable e) {
                            // ignore
                        }
                    }
                }
            }
        } catch (Throwable t) {
            logFine("Failed to read the GUID from persistent storage");
        }

        // Since it's not present in the local file, nor prefs either, let's try getting an IMEI/ESN
        if (hasPermission(context, "android.permission.READ_PHONE_STATE")) {
            try {
                TelephonyManager tm = (TelephonyManager) context.getSystemService(Context.TELEPHONY_SERVICE);
                String id = tm.getDeviceId();
                if (id != null && !id.equals("")) {
                    // Prefix an IMEI/ESN with "_"
                    id = "_" + id;
                    SharedPreferences.Editor editor = prefs.edit().putString(Constants.DEVICE_ID_TAG, id);
                    StorageHelper.persist(editor);
                    writeDeviceIdToFileAsync(context, targetDeviceIdFile, id);
                    return;
                }
            } catch (Throwable ex) {
                // Cool, no permission to read phone state
                // Proceed with another technique
                logFine("Couldn't get a device ID. Will fallback to GUID after the first request is made.");
            }
        } else {
            logFine("Cannot retrieve device ID due to insufficient permissions");
        }

        // Okay, so we didn't find a device ID in the local file, nor could TM give us an IMEI/ESN
        String id = "";
        SharedPreferences.Editor editor = prefs.edit().putString(Constants.DEVICE_ID_TAG, id);
        StorageHelper.persist(editor);
        writeDeviceIdToFileAsync(context, targetDeviceIdFile, id);
    }

    private static void writeDeviceIdToFileAsync(final Context context, final File f, final String id) {
        CleverTapAPI.postAsyncSafely("DeviceInfo#writeDeviceIdToFileAsync", context, new Runnable() {
            @Override
            public void run() {
                writeDeviceIdToFile(context, f, id);
            }
        });
    }

    private static void writeDeviceIdToFile(final Context context, final File f, final String id) {
        FileWriter writer = null;
        try {
            // Check for WRITE_EXTERNAL_STORAGE permission
            if (!hasPermission(context, "android.permission.WRITE_EXTERNAL_STORAGE")) {
                return;
            }

            // Create the parent path
            //noinspection ResultOfMethodCallIgnored
            f.getParentFile().mkdirs();
            writer = new FileWriter(f);
            writer.write(id + "|" + generateChecksum(id));
            writer.flush();
        } catch (Throwable e) {
            Logger.logFine("Failed when attempting to write device ID to file: " + e);
        } finally {
            if (writer != null) {
                try {
                    writer.close();
                } catch (Throwable e) {
                    Logger.logFine("Failed when attempting to close FileWriter: " + e);
                }
            }
        }
    }

    /**
     * Force updates the device ID, with the ID specified.
     * <p>
     * This is used internally by the SDK, there is no need to call this explicitly.
     * </p>
     *
     * @param context The Android {@link Context}
     * @param id      The new device ID
     */
    @SuppressLint("CommitPrefEdits")
    static void forceUpdateDeviceId(Context context, String id) {
        SharedPreferences prefs = StorageHelper.getPreferences(context);
        SharedPreferences.Editor editor = prefs.edit().putString(Constants.DEVICE_ID_TAG, id);
        StorageHelper.persist(editor);
        writeDeviceIdToFileAsync(context, targetDeviceIdFile, id);
    }

    /**
     * Tests whether a particular permission is available or not.
     *
     * @param context    The Android {@link Context}
     * @param permission The fully qualified Android permission name
     * @throws CleverTapPermissionsNotSatisfied
     */
    static void testPermission(Context context, String permission)
            throws CleverTapPermissionsNotSatisfied {
        if (!hasPermission(context, permission))
            throw new CleverTapPermissionsNotSatisfied("Permission required: " + permission);
    }

    static boolean hasPermission(Context context, String permission) {
        try {
            return PackageManager.PERMISSION_GRANTED == ContextCompat.checkSelfPermission(context, permission);
        } catch (Throwable t) {
            return false;
        }
    }

    /**
     * Returns the integer identifier for the default app icon.
     *
     * @param context The Android context
     * @return The integer identifier for the image resource
     */
    static int getAppIconAsIntId(Context context) {
        ApplicationInfo ai = context.getApplicationInfo();
        return ai.icon;
    }

    private static int generateChecksum(String key) {
        // Simple checksum - x^2 + i^2 + 3ix + 5i + 7
        int sum = 0;
        int len = key.length();
        for (int i = 0; i < len; i++) {
            char x = key.charAt(i);
            sum += x * x + i * i + 3 * i * x + 5 * i + 7;
        }
        return sum;
    }

    private static boolean isDeviceIdCorrupted(String s) {
        try {
            String[] parts = s.split(separator);
            String key = parts[0];
            int checksum = Integer.parseInt(parts[1]);
            int calculatedChecksum = generateChecksum(key);
            return !(calculatedChecksum == checksum);
        } catch (Exception e) {
            return true;
        }
    }
}
