/*
 * Author: Jude Pereira
 * Copyright (c) 2014
 */

package com.clevertap.android.sdk;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.SharedPreferences;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import javax.net.ssl.HttpsURLConnection;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URL;
import java.util.Iterator;

/**
 * Provides various methods to communicate with the CleverTap servers.
 */
final class CommsManager {
    static final String NAMESPACE_ARP = "ARP";

    void drainQueueAsync(final Context context) {
        CleverTapAPI.postAsyncSafely("CommsManager#drainQueueAsync", new Runnable() {
            @Override
            public void run() {
                drainDBQueue(context);
            }
        });
    }

    private void drainDBQueue(final Context context) {

        Logger.logFine("Somebody has invoked me to send the queue to CleverTap servers");

        if (!canSendQueue(context)) {
            Logger.logFine("Unable to send queue, bailing");
        }

        QueueManager.QueueCursor cursor;
        QueueManager.QueueCursor previousCursor = null;
        Boolean loadMore = true;
        while (loadMore) {
            cursor = QueueManager.getQueuedEvents(context, 50, previousCursor);

            if (cursor == null || cursor.isEmpty()) {
                Logger.logFine("No events in the queue, bailing");
                break;
            }

            previousCursor = cursor;
            JSONArray queue = cursor.getData();

            if (queue == null || queue.length() <= 0) {
                Logger.logFine("No events in the queue, bailing");
                break;
            }

            loadMore = (sendQueue(context, queue) == 200);
        }
    }

    private int sendQueue(final Context context, final JSONArray queue) {

        int status = -1;

        if (queue == null || queue.length() <= 0) return status;

        HttpsURLConnection conn = null;
        try {
            URL url = new URL(Constants.getEndpoint(context));
            conn = (HttpsURLConnection) url.openConnection();
            conn.setConnectTimeout(10000);
            conn.setReadTimeout(10000);
            conn.setRequestProperty("Content-Type", "application/json; charset=utf-8");
            conn.setInstanceFollowRedirects(false);

            final String body;

            synchronized (CommsManager.class) {
                String deviceId = StorageHelper.getString(context, Constants.DEVICE_ID_TAG, null);
                // Insert the device ID
                // This happens if any event is raised when the first call for GUID hasn't responded back,
                // and events have been fired
                fillGUID(deviceId, queue);

                // Insert the ParseInstallationId if we have one
                String parseInstallationId = CleverTapAPI.getParseInstallationId();
                if (parseInstallationId != null && parseInstallationId.length() > 0) {
                    addDataToQueueItems("prg", parseInstallationId, queue);
                }

                final String req = insertHeader(context, queue);
                Logger.logFine("Send queue contains " + queue.length() + " items: " + req);
                conn.setDoOutput(true);
                conn.getOutputStream().write(req.getBytes("UTF-8"));

                final int responseCode = conn.getResponseCode();
                status = responseCode;

                // Always check for a 200 OK
                if (responseCode != 200) {
                    throw new IOException("Response code is not 200. It is " + responseCode);
                }

                BufferedReader br = new BufferedReader(new InputStreamReader(conn.getInputStream(), "utf-8"));

                StringBuilder sb = new StringBuilder();
                String line;
                while ((line = br.readLine()) != null) {
                    sb.append(line);
                }
                body = sb.toString();
                processResponse(context, body, deviceId);
                Logger.logFine("Completed successfully");
            }
        } catch (Throwable e) {
            Logger.logFine("An exception occurred while trying to send the queue", e);

        } finally {
            if (conn != null) {
                try {
                    conn.disconnect();
                } catch (Throwable t) {
                    // Ignore
                }
            }
        }
        return status;
    }

    private Boolean canSendQueue(final Context context) {
        return isOnline(context);
    }

    private String insertHeader(Context context, JSONArray arr) {
        try {
            // Insert our header at the first position
            final JSONObject header = new JSONObject();
            header.put("type", "meta");

            JSONObject appFields = CleverTapAPI.getInstance(context).getAppLaunchedFields();
            header.put("af", appFields);

            InAppFCManager.attachToHeader(context, header);

            // Resort to string concat for backward compatibility
            return "[" + header.toString() + ", " + arr.toString().substring(1);
        } catch (Throwable t) {
            Logger.error("CommsManager: Failed to attach header", t);
            return arr.toString();
        }
    }

    private void addDataToQueueItems(String key, String value, JSONArray queue) {
        try {
            for (int i = 0; i < queue.length(); i++) {
                JSONObject o = queue.getJSONObject(i);
                o.put(key, value);
            }
        } catch (Throwable t) {
            Logger.logFine("Couldn't insert value " + value + " for key " + key + " into the events!");
        }
    }

    private void fillGUID(String deviceId, JSONArray queue) {
        if (deviceId != null && !deviceId.equals("")) {
            try {
                for (int i = 0; i < queue.length(); i++) {
                    JSONObject o = queue.getJSONObject(i);
                    o.put("g", deviceId);
                }
            } catch (Throwable t) {
                Logger.logFine("Couldn't insert device ID into the events!");
            }
        }
    }

    private static boolean isOnline(Context context) {
        try {
            ConnectivityManager cm =
                    (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
            NetworkInfo netInfo = cm.getActiveNetworkInfo();
            return netInfo != null && netInfo.isConnected();
        } catch (Throwable ignore) {
            // Can't decide whether or not the net is available, maybe it just is
            // Therefore, let's wait for a timeout or something to create an IOException later
            return true;
        }
    }

    private static void processResponse(final Context context, final String responseStr, String deviceId) {
        if (responseStr == null) return;

        try {
            Logger.logFine("Trying to process response: " + responseStr);
            JSONObject response = new JSONObject(responseStr);
            try {
                if (Constants.ENABLE_INAPP) {
                    InAppManager.processResponseAsync(response, context);
                }
            } catch (Throwable t) {
                Logger.error("Failed to process in-app notifications from the response!", t);
            }

            try {
                if ((deviceId == null || deviceId.equals("")) && response.has("g")) {
                    deviceId = response.getString("g");
                    DeviceInfo.forceUpdateDeviceId(context, deviceId);
                    Logger.logFine("Got a new device ID: " + deviceId);

                    CleverTapAPI clevertap = CleverTapAPI.getInstance(context);

                    //notify application code of the deviceID
                    clevertap.notifyUserProfileInitialized();
                }
            } catch (Throwable t) {
                Logger.error("Failed to update device ID!", t);
            }

            try {
                LocalDataStore.syncWithUpstream(context, response);
            } catch (Throwable t) {
                Logger.error("Failed to sync local cache with upstream", t);
            }

            // Handle "arp" (additional request parameters)
            try {
                if (response.has("arp")) {
                    final JSONObject arp = (JSONObject) response.get("arp");
                    if (arp.length() > 0) {
                        handleARPUpdate(context, arp);
                    }
                }
            } catch (Throwable t) {
                Logger.logFine("Failed to process ARP", t);
            }

            // Handle "console" - print them as info to the console
            try {
                if (response.has("console")) {
                    final JSONArray console = (JSONArray) response.get("console");
                    if (console.length() > 0) {
                        for (int i = 0; i < console.length(); i++) {
                            Logger.log(console.get(i).toString());
                        }
                    }
                }
            } catch (Throwable t) {
                // Ignore
            }

            // Handle server set debug level
            try {
                if (response.has("dbg_lvl")) {
                    final int debugLevel = response.getInt("dbg_lvl");
                    if (debugLevel >= 0) {
                        CleverTapAPI.setDebugLevel(debugLevel);
                        Logger.logFine("Set debug level to " + debugLevel + " for this session (set by upstream)");
                    }
                }
            } catch (Throwable t) {
                // Ignore
            }

            // Handle stale_inapp
            try {
                InAppFCManager.processResponse(context, response);
            } catch (Throwable t) {
                // Ignore
            }

            // Always push the profile defaults once everything else is done
            CleverTapAPI.getInstance(context).pushProfileDefaults();

        } catch (Throwable t) {
            Logger.error("Failed to send events to CleverTap", t);
        }
    }

    @SuppressLint("CommitPrefEdits")
    private static void handleARPUpdate(final Context context, final JSONObject arp) {
        if (arp == null || arp.length() == 0) return;

        final SharedPreferences prefs = StorageHelper.getPreferences(context, NAMESPACE_ARP);
        final SharedPreferences.Editor editor = prefs.edit();

        final Iterator<String> keys = arp.keys();
        while (keys.hasNext()) {
            final String key = keys.next();
            try {
                final Object o = arp.get(key);
                if (o instanceof Number) {
                    final int update = ((Number) o).intValue();
                    editor.putInt(key, update);
                } else if (o instanceof String) {
                    if (((String) o).length() < 100) {
                        editor.putString(key, (String) o);
                    } else {
                        Logger.logFine("ARP update for key " + key + " rejected (string value too long)");
                    }
                } else if (o instanceof Boolean) {
                    editor.putBoolean(key, (Boolean) o);
                } else {
                    Logger.logFine("ARP update for key " + key + " rejected (invalid data type)");
                }
            } catch (JSONException e) {
                // Ignore
            }
        }
        StorageHelper.persist(editor);
    }
}
