package com.cleveroad.bootstrap.kotlin_phone_input.view

import android.content.Context
import android.content.res.TypedArray
import android.graphics.Color
import android.graphics.Typeface
import android.graphics.drawable.Drawable
import android.util.AttributeSet
import android.util.TypedValue
import android.view.LayoutInflater
import android.view.View
import android.view.View.OnFocusChangeListener
import android.widget.EditText
import android.widget.ImageView
import android.widget.LinearLayout
import android.widget.TextView
import androidx.annotation.ColorInt
import androidx.core.content.res.ResourcesCompat
import com.cleveroad.bootstrap.kotlin_ext.hide
import com.cleveroad.bootstrap.kotlin_ext.show
import com.cleveroad.bootstrap.kotlin_phone_input.Constants.EMPTY_STRING
import com.cleveroad.bootstrap.kotlin_phone_input.Constants.PHONE_PREF
import com.cleveroad.bootstrap.kotlin_phone_input.R
import com.cleveroad.bootstrap.kotlin_phone_input.data.models.CountryAsset
import com.cleveroad.bootstrap.kotlin_phone_input.utils.*
import com.cleveroad.bootstrap.kotlin_phone_input.utils.CountryFlag.getCountryFlagByCode
import com.cleveroad.bootstrap.kotlin_phone_input.utils.State.*
import com.google.i18n.phonenumbers.NumberParseException
import com.google.i18n.phonenumbers.PhoneNumberUtil.PhoneNumberFormat.INTERNATIONAL
import com.redmadrobot.inputmask.MaskedTextChangedListener

interface OnPhoneCodeClickListener {
    fun onCodeClick()
    fun onPhoneChanged(phone: String, withMask: Boolean)
}

@Suppress("MemberVisibilityCanBePrivate")
class PhoneView(context: Context, attrs: AttributeSet? = null) :
        LinearLayout(context, attrs),
        View.OnClickListener {

    companion object {
        private const val ICON_SIZE_MIN = 60F
        private const val ICON_SIZE_MAX = 180F
        private const val TEXT_SIZE_MIN = 11
        private const val TEXT_SIZE_MAX = 64
        private const val DEFAULT_TEXT_COLOR: Int = 0xFF000000.toInt()
        private const val DEFAULT_HINT_COLOR: Int = 0xFF808080.toInt()
        private const val DEFAULT_DIVIDER_COLOR: Int = 0xFF808080.toInt()
        private const val DEFAULT_TEXT_SIZE = TEXT_SIZE_MIN
        private const val DEFAULT_ICON_SIZE_TEMP = ICON_SIZE_MIN
    }

    private var typefacePhone: Typeface? = null
    private var textSizePhone = DEFAULT_TEXT_SIZE
    private var textColorPhone = DEFAULT_TEXT_COLOR
    private var hintColorPhone = DEFAULT_HINT_COLOR
    private var backgroundPhone: Drawable? = null

    private var typefaceCode: Typeface? = null
    private var textSizeCode = DEFAULT_TEXT_SIZE
    private var textColorCode = DEFAULT_TEXT_COLOR
    private var hintColorCode = DEFAULT_HINT_COLOR
    private var hintTextCode = resources?.getString(R.string.code) ?: EMPTY_STRING
    private var backgroundCode: Drawable? = null

    private var dividerColor = DEFAULT_DIVIDER_COLOR

    private var iconSizeTemp = DEFAULT_ICON_SIZE_TEMP
    private var iconSize: Float = ICON_SIZE_MIN
    private var code: String = EMPTY_STRING
    private var flagImageVisibility: Boolean = true

    private val llPhone: LinearLayout
    private val etPvCode: EditText
    private val etPvPhone: EditText
    private val ivSeparator: ImageView
    private val tvError: TextView

    private var phone: String? = null

    private var flagPath: String? = EMPTY_STRING

    private var inactiveBackground: Int = 0
    private var activeBackground: Int = 0
    private var errorBackground: Int = 0
    private var errorInactiveBackground: Int = 0
    private var validBackground: Int = 0

    private var phoneLength: Int = 0

    private var validationOn: Boolean = false
    private var errorTextVisible: Boolean = true

    private var state: State = INACTIVE

    var callback: OnPhoneCodeClickListener? = null
    private var maskedTextChangedListener: PhoneMaskedTextChangedListener? = null

    private val maskedOnTextChangedListener = object : MaskedTextChangedListener.ValueListener {
        override fun onTextChanged(maskFilled: Boolean, extractedValue: String) {
            phone = extractedValue
            checkLength(extractedValue)
        }
    }

    init {
        LayoutInflater.from(context).inflate(R.layout.phone_view, this, true)

        val typedArray = getContext().obtainStyledAttributes(attrs, R.styleable.PhoneView)

        llPhone = findViewById(R.id.llPhone)
        tvError = findViewById(R.id.tvError)
        etPvCode = findViewById(R.id.etCode)
        etPvPhone = findViewById(R.id.etPhone)
        ivSeparator = findViewById(R.id.ivSeparator)

        try {
            typedArray?.run {
                iconSizeTemp = getDimension(R.styleable.PhoneView_pcv_icon_size, dpToPx(ICON_SIZE_MIN))

                setTypefaceCode(getLoadedTypeface(context, getResourceId(R.styleable.PhoneView_pcv_preloaded_code_font, 0))
                        ?: Typeface.create(getString(R.styleable.PhoneView_pcv_text_typeface_phone_code), Typeface.NORMAL))
                setTypefacePhone(getLoadedTypeface(context, getResourceId(R.styleable.PhoneView_pcv_preloaded_phone_font, 0))
                        ?: Typeface.create(getString(R.styleable.PhoneView_pcv_text_typeface_phone_code), Typeface.NORMAL))

                setTextSizePhone(getDimensionPixelSize(R.styleable.PhoneView_pcv_text_size_phone_number, textSizePhone))
                setTextColorPhone(getColor(R.styleable.PhoneView_pcv_text_color_phone_number, textColorPhone))
                setHintColorPhone(getColor(R.styleable.PhoneView_pcv_hint_color_phone_number, hintColorPhone))
                setBackgroundPhone(getDrawable(R.styleable.PhoneView_pcv_background_phone_number))

                setTextSizeCode(getDimensionPixelSize(R.styleable.PhoneView_pcv_text_size_phone_code, textSizeCode))
                setTextColorCode(getColor(R.styleable.PhoneView_pcv_text_color_phone_code, textColorCode))
                setHintColorCode(getColor(R.styleable.PhoneView_pcv_hint_color_phone_code, hintColorCode))
                setHintTextCode(getString(R.styleable.PhoneView_pcv_hint_text_phone_code)
                        ?: hintTextCode)
                setBackgroundCode(getDrawable(R.styleable.PhoneView_pcv_background_phone_code))

                setFlag()
                setBackGrounds(getBoolean(R.styleable.PhoneView_pcv_defaults_backgrounds, false))
                setErrorText(context)
                setState(State.getState(getInt(R.styleable.PhoneView_pcv_state, 0)))
                validationOn = getBoolean(R.styleable.PhoneView_pcv_validation_on, false)
                separatorVisibility(getBoolean(R.styleable.PhoneView_pcv_show_divider, false))
                setDividerColor(getColor(R.styleable.PhoneView_pcv_dividerColor, dividerColor))
            }
        } finally {
            typedArray.recycle()
        }

        iconSize = getIconSizeInLimits(iconSizeTemp)
        etPvCode.setOnClickListener(this)
        setData(getDefaultCountryAsset(context))
    }

    override fun onClick(v: View?) {
        callback?.onCodeClick()
    }

    private fun getTextSizeInLimits(size: Int) = when {
        size > TEXT_SIZE_MAX -> TEXT_SIZE_MAX
        size < TEXT_SIZE_MIN -> TEXT_SIZE_MIN
        else -> size
    }

    private fun getIconSizeInLimits(size: Float) = when {
        size > ICON_SIZE_MAX -> ICON_SIZE_MAX
        size < ICON_SIZE_MIN -> ICON_SIZE_MIN
        else -> size
    }

    private fun setupPhoneEditText(phoneMask: String) {
        etPvPhone.removeTextChangedListener(maskedTextChangedListener)
        val focusChangeListener = OnFocusChangeListener { _, hasFocus ->
            if (state != ERROR && state != ERROR_INACTIVE) {
                setState(if (hasFocus) ACTIVE else INACTIVE)
            } else {
                setState(if (hasFocus) ERROR else ERROR_INACTIVE)
            }
        }
        maskedTextChangedListener = PhoneMaskedTextChangedListener(phoneMask,
                false,
                etPvPhone,
                null,
                maskedOnTextChangedListener,
                focusChangeListener)
        with(etPvPhone) {
            addTextChangedListener(maskedTextChangedListener)
            onFocusChangeListener = maskedTextChangedListener
            hint = maskedTextChangedListener?.placeholder()
        }
        phoneLength = phoneMask.length - phoneMask.replace("0", "").length
    }

    private fun checkLength(phone: String) {
        if (validationOn) {
            setState(if (phone.length >= phoneLength) VALID else ERROR)
            if (errorTextVisible) setErrorTextVisible(state == VALID)
        }
        callback?.onPhoneChanged(phone, state == VALID)
    }

    fun setErrorTextVisible(visible: Boolean) {
        tvError.visibility = if (visible) View.VISIBLE else View.INVISIBLE
    }

    fun setData(countryAsset: CountryAsset) {
        setPhoneMask(countryAsset)
        setCountryCode(countryAsset.dialCode.toString())
        setCountryIcon(countryAsset.ab, flagPath)
    }

    fun setPhoneMask(countryAsset: CountryAsset) {
        setupPhoneEditText(PhoneMaskUtils.generatePhoneMask(
                let {
                    try {
                        PhoneFormatUtils.formatPhone(countryAsset.ab, INTERNATIONAL, true)
                    } catch (numberParseException: NumberParseException) {
                        null
                    } catch (illegalStateException: IllegalStateException) {
                        null
                    }
                } ?: countryAsset.phoneFormat, true))
        etPvPhone.setText(phone)
    }

    fun setState(state: State) {
        this.state = state
        llPhone.setBackgroundResource(when (state) {
            INACTIVE -> inactiveBackground
            ACTIVE -> activeBackground
            ERROR -> errorBackground
            ERROR_INACTIVE -> errorInactiveBackground
            VALID -> validBackground
        })
    }

    fun getState() = state

    fun getPhone() = phone.orEmpty() to code

    fun setTextError(text: String) {
        tvError.text = text
    }

    fun setCountryCode(dialCode: String) {
        val text = PHONE_PREF + dialCode
        etPvCode.setText(text)
    }

    fun setCountryIcon(countryCode: String, flagPath: String?) {
        if (flagImageVisibility) {
            code = countryCode
            val countryIcon = getCountryFlagByCode(context, code)
            val drawable = BitmapUtils.resizeIcon(countryIcon, iconSize.toInt(), resources, flagPath)
            etPvCode.setCompoundDrawablesRelativeWithIntrinsicBounds(drawable, null, null, null)
        }
    }

    fun setBackgroundPhone(background: Drawable?) {
        backgroundPhone = background
        etPvPhone.background = background
    }

    fun setBackgroundCode(background: Drawable?) {
        backgroundCode = background
        etPvCode.background = background
    }

    fun setIconSize(size: Float) {
        iconSize = getIconSizeInLimits(size)
        setCountryIcon(code, flagPath)
    }

    fun setTextSizePhone(size: Int) {
        textSizePhone = size
        etPvPhone.setTextSize(TypedValue.COMPLEX_UNIT_PX, getTextSizeInLimits(size).toFloat())
    }

    fun setTextSizeError(size: Int) {
        tvError.setTextSize(TypedValue.COMPLEX_UNIT_PX, size.toFloat())
    }

    fun setErrorTextColor(color: Int) {
        tvError.setTextColor(color)
    }

    fun setTextSizeCode(size: Int) {
        textSizeCode = size
        etPvCode.setTextSize(TypedValue.COMPLEX_UNIT_PX, getTextSizeInLimits(size).toFloat())
    }

    fun setTextColorPhone(@ColorInt color: Int) {
        textColorPhone = color
        etPvPhone.setTextColor(color)
    }

    fun setTextColorCode(@ColorInt color: Int) {
        textColorCode = color
        etPvCode.setTextColor(color)
    }

    fun setHintTextCode(hintText: String) {
        hintTextCode = hintText
        etPvCode.hint = hintText
    }

    fun setHintColorPhone(@ColorInt color: Int) {
        hintColorPhone = color
        etPvPhone.setHintTextColor(color)
    }

    fun setHintColorCode(@ColorInt color: Int) {
        hintColorCode = color
        etPvCode.setHintTextColor(color)
    }

    fun setDividerColor(@ColorInt color: Int) {
        dividerColor = color
        ivSeparator.setBackgroundColor(color)
    }

    fun setTypefaceError(typeface: Typeface) {
        tvError.typeface = typeface
    }

    fun setTypefacePhone(typeface: Typeface) {
        typefacePhone = typeface
        etPvPhone.typeface = typeface
    }

    fun setErrorMargins(left: Float = 0F, top: Float = 0F, right: Float = 0F, bottom: Float = 0F) {
        tvError.layoutParams = LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT)
                .apply {
                    setMargins(dpToPx(left).toInt(), dpToPx(top).toInt(),
                            dpToPx(right).toInt(), dpToPx(bottom).toInt())
                }
    }

    fun setTypefaceCode(typeface: Typeface) {
        typefaceCode = typeface
        etPvCode.typeface = typeface
    }

    fun separatorVisibility(visible: Boolean, gone: Boolean = true) {
        ivSeparator.run { if (visible) show() else hide(gone) }
    }

    fun flagImageVisibility(visible: Boolean) {
        flagImageVisibility = visible
        if (flagImageVisibility) setCountryIcon(code, flagPath)
        else etPvCode.setCompoundDrawablesRelativeWithIntrinsicBounds(null, null, null, null)
    }

    private fun dpToPx(dp: Float): Float =
            TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, dp, resources.displayMetrics)

    private fun getLoadedTypeface(context: Context, codeFontResId: Int) =
            codeFontResId.takeIf { it != 0 }?.let { ResourcesCompat.getFont(context, it) }

    private fun TypedArray.setBackGrounds(defaults: Boolean) {
        if (defaults) {
            inactiveBackground = R.drawable.underline_inactive
            activeBackground = R.drawable.underline_active
            errorBackground = R.drawable.underline_error
            errorInactiveBackground = R.drawable.underline_error_inactive
            validBackground = R.drawable.underline_valid
        } else {
            inactiveBackground = getResourceId(R.styleable.PhoneView_pcv_background_inactive, 0)
            activeBackground = getResourceId(R.styleable.PhoneView_pcv_background_active, 0)
            errorBackground = getResourceId(R.styleable.PhoneView_pcv_background_error, 0)
            errorInactiveBackground = getResourceId(R.styleable.PhoneView_pcv_background_error_inactive, 0)
            validBackground = getResourceId(R.styleable.PhoneView_pcv_background_valid, 0)
        }
    }

    private fun TypedArray.setFlag() {
        flagPath = getFlagPath(getInt(R.styleable.PhoneView_pcv_flag_shape, -1))
                ?.let(context::getString) ?: getString(R.styleable.PhoneView_pcv_custom_flag_path)
    }

    private fun TypedArray.setErrorText(context: Context) {
        setErrorTextColor(getColor(R.styleable.PhoneView_pcv_error_text_color, Color.RED))
        setTextSizeError(getDimensionPixelSize(R.styleable.PhoneView_pcv_error_text_size, textSizeCode))
        setTextError(getString(R.styleable.PhoneView_pcv_error_text) ?: EMPTY_STRING)
        setTypefaceError(getLoadedTypeface(context, getResourceId(R.styleable.PhoneView_pcv_error_text_preloaded_font, 0))
                ?: Typeface.create(getString(R.styleable.PhoneView_pcv_error_text_typeface), Typeface.NORMAL))
        getBoolean(R.styleable.PhoneView_pcv_error_text_visible, true).apply {
            if (!this) tvError.visibility = View.GONE
            errorTextVisible = this
        }
    }
}