package com.cleveroad.bootstrap.kotlin_mvp_loader

import android.content.Context
import android.os.Bundle
import androidx.loader.app.LoaderManager
import androidx.loader.content.Loader

/**
 * Implementation of [LoaderManager.LoaderCallbacks] for MVP pattern.
 *
 * This class holds a link to [BSMvpView]. Every time [AndroidLifecycle.onResume] called,
 * the view links to [BSMvpPresenter] and the presenter links to view. Every time
 * [AndroidLifecycle.onPause] called, the view unlinks from the presenter and the presenter
 * unlinks from the view. So presenter is always linked to proper view.

 * @since 1.0
 */
internal class PresenterLoaderCallbacks<TPresenter : BSMvpPresenter<TView>, TView : BSMvpView<TPresenter>>
private constructor(context: Context,
                    private val view: TView,
                    private val presenterCallable: () -> TPresenter)
    : LoaderManager.LoaderCallbacks<TPresenter>, AndroidLifecycle {

    companion object {
        fun <TPresenter : BSMvpPresenter<TView>, TView : BSMvpView<TPresenter>>
                create(context: Context, view: TView, presenterCallable: () -> TPresenter) = PresenterLoaderCallbacks(context, view, presenterCallable)
    }

    private val appContext = context.applicationContext
    private var presenter: TPresenter? = null
    private var resumed = false

    private var lastSavedState: Bundle? = null

    override fun onResume() {
        resumed = true
        checkAndCallResume()
    }

    override fun onPause() {
        resumed = false
        presenter?.let {
            if (it.view === this.view) {
                this.view.presenter = null
            }
            it.view = null
            it.onPause()
        }
    }

    private fun checkAndCallResume() {
        val presenter = presenter
        presenter?.let {
            this.view.presenter = it
            it.view = this.view
            it.onResume()
        }
    }

    override fun onCreateLoader(id: Int, args: Bundle?): Loader<TPresenter> {
        return AbstractPresenterLoader(appContext, presenterCallable)
    }

    override fun onLoadFinished(loader: Loader<TPresenter>, data: TPresenter) {
        presenter = data
        presenter?.apply {
            lastSavedState.let {
                onRestoreInstanceState(it)
            }
        }
        if (resumed) {
            checkAndCallResume()
        }
    }

    override fun onLoaderReset(loader: Loader<TPresenter>) {
        presenter?.view = null
        presenter = null
    }

    override fun onRestoreInstanceState(savedInstanceState: Bundle?) {
        presenter?.onRestoreInstanceState(savedInstanceState)
                ?: savedInstanceState?.let { lastSavedState = it }
    }

    override fun onSaveInstanceState(outState: Bundle) {
        presenter?.onSaveInstanceState(outState)
    }

}
