package com.cleveroad.bootstrap.kotlin_mvp_loader

import android.content.Context
import android.os.Build
import androidx.loader.content.Loader

/**
 * Implementation of [Loader] that loads and stores [BSMvpPresenter] instance.
 * @param <TPresenter> type of presenter
</TPresenter> */
internal class AbstractPresenterLoader<TPresenter : BSMvpPresenter<*>>
/**
 * Stores away the application context associated with context.
 * Since Loaders can be used across multiple activities it's dangerous to
 * store the context directly; always use [.getContext] to retrieve
 * the Loader's Context, don't use the constructor argument directly.
 * The Context returned by [.getContext] is safe to use across
 * Activity instances.

 * @param context used to retrieve the application context.
 * *
 * @param presenterProvider provider of presenter instances
 */
(context: Context, private val presenterProvider: () -> TPresenter) : Loader<TPresenter>(context) {

    private var presenter: TPresenter? = null

    override fun onStartLoading() {
        super.onStartLoading()
        if (takeContentChanged() || presenter == null) {
            forceLoad()
            return
        }
        deliverResult(presenter)
    }

    override fun onStopLoading() {
        super.onStopLoading()
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
            cancelLoad()
        }
    }

    override fun onForceLoad() {
        super.onForceLoad()
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
            cancelLoad()
        }
        presenter = presenterProvider()
        presenter?.onCreated()
        deliverResult(presenter)
    }

    override fun onReset() {
        super.onReset()
        onStopLoading()
        presenter?.apply {
            onDestroyed()
            presenter = null
        }
    }
}
