package com.cleveroad.bootstrap.kotlin_gps.service

import android.Manifest.permission.ACCESS_COARSE_LOCATION
import android.Manifest.permission.ACCESS_FINE_LOCATION
import android.annotation.SuppressLint
import android.app.Notification
import android.app.Service
import android.content.Context
import android.content.Intent
import android.content.pm.PackageManager.PERMISSION_GRANTED
import android.location.Location
import android.location.LocationListener
import android.location.LocationManager
import android.os.Build
import android.os.Bundle
import android.os.Looper
import android.util.Log
import androidx.core.app.ActivityCompat.checkSelfPermission
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.LifecycleService
import com.cleveroad.bootstrap.kotlin_gps.LocationProcessor
import com.cleveroad.bootstrap.kotlin_gps.LocationProcessorImpl
import com.cleveroad.bootstrap.kotlin_gps.LocationProvider
import com.google.android.gms.common.ConnectionResult
import com.google.android.gms.common.GoogleApiAvailability
import com.google.android.gms.common.api.GoogleApiClient
import com.google.android.gms.location.LocationCallback
import com.google.android.gms.location.LocationRequest
import com.google.android.gms.location.LocationRequest.*
import com.google.android.gms.location.LocationResult
import com.google.android.gms.location.LocationServices


class GpsTrackerService : LifecycleService(),
        LifecycleOwner,
        GoogleApiClient.ConnectionCallbacks,
        GoogleApiClient.OnConnectionFailedListener {

    companion object {
        private const val ON = 1
        private const val OFF = 2
        private const val SERVICE_ID = 666

        // 5 minutes
        private const val LOCATION_INTERVAL_DEFAULT = 1000L * 60L * 5L
        //1 minute
        private const val LOCATION_INTERVAL_MINIMUM = 1000L * 5L
        private const val SMALLEST_DISPLACEMENT = 10F

        private val TAG = GpsTrackerService::class.java.simpleName
        private val EXTRA_IS_ON = "is_on$TAG"
        private var notificationForForeground: Notification? = null
        private var locationInternal = LOCATION_INTERVAL_DEFAULT
        private var isFastestInterval = false
        private var priorityLocation = PRIORITY_HIGH_ACCURACY

        /**
         * Start gps tracking
         * @param context - [Context]
         * @param notificationForForeground [Notification]
         * @param locationInternal Location detection interval
         * @param isFastestInterval Location detection fastes interval
         * @param priorityLocation [PRIORITY_HIGH_ACCURACY], [PRIORITY_BALANCED_POWER_ACCURACY],
         * [PRIORITY_LOW_POWER], [PRIORITY_NO_POWER]
         *
         */
        fun startTracking(context: Context,
                          notificationForForeground: Notification?,
                          locationInternal: Long,
                          isFastestInterval: Boolean = false,
                          priorityLocation: Int) {
            this.locationInternal = Math.max(locationInternal, LOCATION_INTERVAL_MINIMUM)
            this.priorityLocation = priorityLocation
            this.isFastestInterval = isFastestInterval
            this.notificationForForeground = notificationForForeground
            val intent = Intent(context, GpsTrackerService::class.java)
                    .apply { putExtra(EXTRA_IS_ON, ON) }
            with(context) {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                    startForegroundService(intent)
                } else {
                    startService(intent)
                }
            }
        }

        /**
         * Stop gps tracking
         *
         * @param context - [Context]
         */
        fun stopTracking(context: Context) {
            context.startService(Intent(context, GpsTrackerService::class.java)
                    .apply { putExtra(EXTRA_IS_ON, OFF) })
            notificationForForeground = null
        }

        /**
         * Stop gps service
         *
         * @param context - [Context]
         */
        fun stopService(context: Context) {
            context.stopService(Intent(context, GpsTrackerService::class.java))
            notificationForForeground = null
        }
    }

    private val locationCallback = object : LocationCallback() {
        override fun onLocationResult(location: LocationResult?) {
            location?.let { currentLocation ->
                locationProcessor?.locationUpdated(currentLocation.lastLocation)
            }
        }
    }

    private val locationListener = object : LocationListener {

        override fun onLocationChanged(location: Location) {
            locationProcessor?.locationUpdated(location)
        }

        override fun onProviderDisabled(provider: String) {
            LocationProvider.isTrackingLD.postValue(false)
        }

        override fun onProviderEnabled(provider: String) {
            LocationProvider.isTrackingLD.postValue(true)
        }

        override fun onStatusChanged(provider: String, status: Int, extras: Bundle) {
            //no need
        }
    }

    private var locationProcessor: LocationProcessor? = null

    private val apiClient by lazy {
        GoogleApiClient.Builder(this)
                .addConnectionCallbacks(this)
                .addOnConnectionFailedListener(this)
                .addApi(LocationServices.API)
                .build()
    }

    private val locationManager
        get() = getSystemService(Context.LOCATION_SERVICE) as? LocationManager

    override fun onCreate() {
        startForeground()
        super.onCreate()
    }

    private fun startForeground() {
        notificationForForeground?.let {
            startForeground(SERVICE_ID, it)
        }
    }

    override fun onStartCommand(intent: Intent?, flags: Int, startId: Int): Int = intent.let {
        when (it?.getIntExtra(EXTRA_IS_ON, OFF)) {
            ON -> {
                if (checkPlayServices()) connectGoogleApiClient() else startAndroidLocationUpdates()
                Service.START_STICKY
            }
            OFF -> {
                disconnect()
                super.onStartCommand(it, flags, startId)
            }
            else -> super.onStartCommand(it, flags, startId)
        }
    }

    private fun connectGoogleApiClient() {
        if (checkSelfPermission(this, ACCESS_FINE_LOCATION) == PERMISSION_GRANTED
                && checkSelfPermission(this, ACCESS_COARSE_LOCATION) == PERMISSION_GRANTED) {
            with(apiClient) {
                if (!isConnected && !isConnecting) connect() else reconnect()
            }
        }
    }

    private fun disconnect() {
        locationProcessor?.let {
            it.close()
            locationProcessor = null
        }
        locationManager?.removeUpdates(locationListener)
        LocationServices.getFusedLocationProviderClient(this).removeLocationUpdates(locationCallback)
        apiClient.disconnect()
        LocationProvider.isTrackingLD.postValue(false)
        stopForeground(true)
    }

    override fun onConnected(bundle: Bundle?) {
        startGMSLocationUpdates()
    }

    override fun onConnectionSuspended(i: Int) {
        apiClient.reconnect()
    }

    override fun onConnectionFailed(connectionResult: ConnectionResult) {
        connectionResult.errorMessage?.takeIf { it.isNotEmpty() }?.let { Log.e(TAG, it) }
        LocationProvider.isTrackingLD.postValue(false)
    }

    @SuppressLint("MissingPermission")
    private fun startGMSLocationUpdates() {
        if (checkLocationPermissions()) {
            locationProcessor = LocationProcessorImpl
            LocationServices
                    .getFusedLocationProviderClient(this)
                    .requestLocationUpdates(createRequest(), locationCallback, Looper.myLooper())
            LocationProvider.isTrackingLD.postValue(true)
        }
    }

    @SuppressLint("MissingPermission")
    private fun startAndroidLocationUpdates() {
        if (checkLocationPermissions()) {
            locationProcessor = LocationProcessorImpl
            locationManager?.requestLocationUpdates(LocationManager.NETWORK_PROVIDER.takeIf {
                locationManager?.isProviderEnabled(LocationManager.NETWORK_PROVIDER) == true
            } ?: LocationManager.GPS_PROVIDER,
                    locationInternal, SMALLEST_DISPLACEMENT, locationListener)
            LocationProvider.isTrackingLD.postValue(true)
        }
    }

    private fun createRequest() = LocationRequest().apply {
        priority = priorityLocation
        if (isFastestInterval) {
            fastestInterval = locationInternal
        } else {
            interval = locationInternal
        }
        smallestDisplacement = SMALLEST_DISPLACEMENT
    }

    private fun checkPlayServices() = GoogleApiAvailability.getInstance().run {
        isGooglePlayServicesAvailable(this@GpsTrackerService) == ConnectionResult.SUCCESS
    }

    private fun checkLocationPermissions() = checkSelfPermission(this, ACCESS_FINE_LOCATION) == PERMISSION_GRANTED
            && (checkSelfPermission(this, ACCESS_COARSE_LOCATION) == PERMISSION_GRANTED)

    override fun onDestroy() {
        disconnect()
        super.onDestroy()
    }

}