package com.cleveroad.bootstrap.kotlin_gps.service

import android.Manifest.permission.ACCESS_COARSE_LOCATION
import android.Manifest.permission.ACCESS_FINE_LOCATION
import android.app.Notification
import android.app.Service
import android.content.Context
import android.content.Intent
import android.content.pm.PackageManager.PERMISSION_GRANTED
import android.os.Build
import android.os.Bundle
import android.os.Looper
import android.util.Log
import androidx.core.app.ActivityCompat.checkSelfPermission
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.LifecycleService
import com.cleveroad.bootstrap.kotlin_gps.LocationProcessor
import com.cleveroad.bootstrap.kotlin_gps.LocationProcessorImpl
import com.cleveroad.bootstrap.kotlin_gps.LocationProvider
import com.google.android.gms.common.ConnectionResult
import com.google.android.gms.common.api.GoogleApiClient
import com.google.android.gms.location.LocationCallback
import com.google.android.gms.location.LocationRequest
import com.google.android.gms.location.LocationResult
import com.google.android.gms.location.LocationServices


class GpsTrackerService : LifecycleService(),
        LifecycleOwner,
        GoogleApiClient.ConnectionCallbacks,
        GoogleApiClient.OnConnectionFailedListener {

    companion object {
        private const val ON = 1
        private const val OFF = 2
        private const val SERVICE_ID = 666

        // 5 minutes
        private const val LOCATION_INTERVAL_MILLISECONDS_DEFAULT = 1000L * 60 * 5
        //1 minute
        private const val LOCATION_INTERVAL_MILLISECONDS_MIN = 1000L * 5
        private val TAG = GpsTrackerService::class.java.simpleName
        private val EXTRA_IS_ON = "is_on$TAG"
        private var notificationForForeground: Notification? = null
        private var locationInternalMilliseconds = LOCATION_INTERVAL_MILLISECONDS_DEFAULT
        private var isFastestInterval = false
        private var priorityLocation = LocationRequest.PRIORITY_HIGH_ACCURACY

        /**
         * Start gps tracking
         *
         * @param context - [Context]
         */
        fun startTracking(context: Context,
                          notificationForForeground: Notification?,
                          LocationInternalMilliseconds: Long,
                          isFastestInterval: Boolean = false,
                          priorityLocation: Int) {
            this.locationInternalMilliseconds = Math.max(LocationInternalMilliseconds, LOCATION_INTERVAL_MILLISECONDS_MIN)
            this.priorityLocation = priorityLocation
            this.isFastestInterval = isFastestInterval
            this.notificationForForeground = notificationForForeground
            val intent = Intent(context, GpsTrackerService::class.java)
                    .apply { putExtra(EXTRA_IS_ON, ON) }
            with(context) {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                    startForegroundService(intent)
                } else {
                    startService(intent)
                }
            }
        }

        /**
         * Stop gps tracking
         *
         * @param context - [Context]
         */
        fun stopTracking(context: Context) {
            context.startService(Intent(context, GpsTrackerService::class.java)
                    .apply { putExtra(EXTRA_IS_ON, OFF) })
            notificationForForeground = null
        }

        /**
         * Stop gps service
         *
         * @param context - [Context]
         */
        fun stopService(context: Context) {
            context.stopService(Intent(context, GpsTrackerService::class.java))
            notificationForForeground = null
        }
    }

    private val locationCallback = object : LocationCallback() {
        override fun onLocationResult(location: LocationResult?) {
            location?.let { currentLocation ->
                locationProcessor?.locationUpdated(currentLocation.lastLocation)
            }
        }
    }

    private var locationProcessor: LocationProcessor? = null

    private val apiClient by lazy {
        GoogleApiClient.Builder(this)
                .addConnectionCallbacks(this)
                .addOnConnectionFailedListener(this)
                .addApi(LocationServices.API)
                .build()
    }

    override fun onCreate() {
        startForeground()
        super.onCreate()
    }

    private fun startForeground() {
        notificationForForeground?.let {
            startForeground(SERVICE_ID, it)
        }
    }

    override fun onStartCommand(intent: Intent?, flags: Int, startId: Int): Int = intent.let {
        when (it?.getIntExtra(EXTRA_IS_ON, OFF)) {
            ON -> {
                connectGoogleApiClient()
                Service.START_STICKY
            }
            OFF -> {
                disconnect()
                super.onStartCommand(it, flags, startId)
            }
            else -> super.onStartCommand(it, flags, startId)
        }
    }

    private fun connectGoogleApiClient() {
        if (checkSelfPermission(this, ACCESS_FINE_LOCATION) == PERMISSION_GRANTED
                && checkSelfPermission(this, ACCESS_COARSE_LOCATION) == PERMISSION_GRANTED) {
            with(apiClient) {
                if (!isConnected && !isConnecting) connect() else reconnect()
            }
        }
    }

    private fun disconnect() {
        locationProcessor?.let {
            it.close()
            locationProcessor = null
        }
        apiClient.disconnect()
        LocationProvider.isTrackingLD.postValue(false)
        stopForeground(true)
    }

    override fun onConnected(bundle: Bundle?) {
        if (checkSelfPermission(this, ACCESS_FINE_LOCATION) != PERMISSION_GRANTED && checkSelfPermission(this, ACCESS_COARSE_LOCATION) != PERMISSION_GRANTED) {
            return
        }
        locationProcessor = LocationProcessorImpl
        LocationServices
                .getFusedLocationProviderClient(this)
                .requestLocationUpdates(createRequest(), locationCallback, Looper.myLooper())
        LocationProvider.isTrackingLD.postValue(true)
    }

    override fun onConnectionSuspended(i: Int) {
        apiClient.reconnect()
    }

    override fun onConnectionFailed(connectionResult: ConnectionResult) {
        Log.e(TAG, connectionResult.errorMessage)
        LocationProvider.isTrackingLD.postValue(false)
    }

    private fun createRequest() = LocationRequest().apply {
        priority = priorityLocation
        if (isFastestInterval) {
            fastestInterval = locationInternalMilliseconds
        } else {
            interval = locationInternalMilliseconds
        }
    }

    override fun onDestroy() {
        disconnect()
        super.onDestroy()
    }
}