package com.citruspay.citrusbrowser.core;

import android.content.Context;
import android.os.Parcel;
import android.os.Parcelable;

import java.util.HashMap;
import java.util.Map;

/**
 * This class contains all the config params and methods that merchants need to set and which will
 * define how the CitrusBrowser operates and performs the payment.
 * Created by Gautam on 15/6/16.
 */
public class CitrusBrowserConfig implements Parcelable {

    private Context mContext = null;
    private String paymentUrl = null;
    private Map<String, String> headers = null;
    private TxnDetails mTxnDetails = null;
    private CardDetails mCardDetails = null;
    private PaymentStatus mPaymentStatus = null;
    private NativeScreenType mNativeScreenType = null;
    private boolean autoReadOTP = true;
    private boolean autoSubmitOTP = true;
    public static final String KEY_CITRUS_BROWSER_CONFIG = "key_citrus_browser_config";
    private static CitrusBrowserConfig mCitrusBrowserConfig = null;


    private CitrusBrowserConfig(Context ctx) {
        this.mContext = ctx;
    }

    /**
     * Get CitrusBrowserConfig instance.
     *
     * @param context
     * @return
     */
    public static CitrusBrowserConfig getInstance(Context context) {
        if (mCitrusBrowserConfig == null) {
            synchronized (CitrusBrowserConfig.class) {
                if (mCitrusBrowserConfig == null) {
                    mCitrusBrowserConfig = new CitrusBrowserConfig(context);
                }
            }
        }
        return mCitrusBrowserConfig;
    }

    /**
     * The URL in the form of a string on which the redirection will happen for making a payment.
     *
     * @param url
     */
    private void setPaymentUrl(String url) {
        this.paymentUrl = url;
    }

    /**
     * The URL in the form of a string on which the redirection will happen for making a payment.
     * Also a Map containing certain headers required to add to the webview.
     *
     * @param uRL
     * @param headers
     */
    private void setPaymentUrl(String uRL, Map<String, String> headers) {
        this.paymentUrl = uRL;
        this.headers = headers;
    }

    /**
     * The Bank page HTML in the form of String.
     *
     * @param HTML
     */
    private void setPaymentHtml(String HTML) {

    }

    /**
     * Set this method to TRUE to show the Native UI screen, else to view the traditional WebView.
     *
     * @param value
     */
    private void enableNativeView(boolean value) {

    }

    /**
     * TxnDetails class is a POJO class which holds the Transaction details i.e. the amount , currency, txn type.
     * Set this method with Txn details for the current Txn.
     * Together these details are required to Flash UI experience.
     *
     * @param details
     */
    public void setTxnDetails(TxnDetails details) {
        this.mTxnDetails = details;
    }


    /**
     * CardDetails class is a POJO class which holds the Card details i.e. card number, Card prefix, card suffix and card token.
     * Together these details are required to show the card number on the CitrusBrowser UI and for other use cases.
     *
     * @param mCardDetails
     */
    public void setCardDetails(CardDetails mCardDetails) {
        this.mCardDetails = mCardDetails;
    }

    /**
     * PaymentStatus is an enum consisting these values (SUCCESS, FAIL or CANCELLED)
     * This statuses are important in order to track transaction success/failure cases.
     * Accordingly the CitrusBrowser UI will show appropriate indication to the user.
     *
     * @param status
     */
    private void setPaymentStatus(PaymentStatus status) {
        this.mPaymentStatus = status;
    }

    /**
     * NativeScreenType determines the flash screen render type you want.
     *
     * @param mNativeScreenType
     */
    public void setNativeScreenType(NativeScreenType mNativeScreenType) {
        this.mNativeScreenType = mNativeScreenType;
    }

    /**
     * Set Auto read OTP status.
     *
     * @param autoReadOTP
     */
    private void setAutoReadOTP(boolean autoReadOTP) {
        this.autoReadOTP = autoReadOTP;
    }

    /**
     * Set Auto submit OTP status.
     *
     * @param autoSubmitOTP
     */
    public void setAutoSubmitOTP(boolean autoSubmitOTP) {
        this.autoSubmitOTP = autoSubmitOTP;
    }

    /**
     * Get Txn Details.
     *
     * @return
     */
    public TxnDetails getTxnDetails() {
        return this.mTxnDetails;
    }

    /**
     * Get Payment Status.
     *
     * @return
     */
    private PaymentStatus getPaymentStatus() {
        return mPaymentStatus;
    }

    /**
     * Get Auto Read Status.
     *
     * @return
     */
    private boolean isAutoReadOTP() {
        return autoReadOTP;
    }

    /**
     * Get Auto Submit Status.
     *
     * @return
     */
    public boolean isAutoSubmitOTP() {
        return autoSubmitOTP;
    }

    /**
     * Get Native Screen render Type.
     *
     * @return
     */
    public NativeScreenType getNativeScreenType() {
        return mNativeScreenType;
    }

    /**
     * Get Payment URL.
     *
     * @return
     */
    private String getPaymentUrl() {
        return paymentUrl;
    }

    /**
     * Get Webview Headers.
     *
     * @return
     */
    private Map<String, String> getHeaders() {
        return headers;
    }

    /**
     * Get Card Details.
     *
     * @return
     */
    public CardDetails getCardDetails() {
        return mCardDetails;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(this.paymentUrl);
        dest.writeParcelable(this.mTxnDetails, flags);
        dest.writeParcelable(this.mCardDetails, flags);
        dest.writeInt(this.mPaymentStatus == null ? -1 : this.mPaymentStatus.ordinal());
        dest.writeInt(this.mNativeScreenType == null ? -1 : this.mNativeScreenType.ordinal());
        dest.writeByte(this.autoReadOTP ? (byte) 1 : (byte) 0);
        dest.writeByte(this.autoSubmitOTP ? (byte) 1 : (byte) 0);
    }

    protected CitrusBrowserConfig(Parcel in) {
        this.paymentUrl = in.readString();
        this.mTxnDetails = in.readParcelable(TxnDetails.class.getClassLoader());
        this.mCardDetails = in.readParcelable(CardDetails.class.getClassLoader());
        int tmpMPaymentStatus = in.readInt();
        this.mPaymentStatus = tmpMPaymentStatus == -1 ? null : PaymentStatus.values()[tmpMPaymentStatus];
        int tmpMNativeScreenType = in.readInt();
        this.mNativeScreenType = tmpMNativeScreenType == -1 ? null : NativeScreenType.values()[tmpMNativeScreenType];
        this.autoReadOTP = in.readByte() != 0;
        this.autoSubmitOTP = in.readByte() != 0;
    }

    public static final Creator<CitrusBrowserConfig> CREATOR = new Creator<CitrusBrowserConfig>() {
        @Override
        public CitrusBrowserConfig createFromParcel(Parcel source) {
            return new CitrusBrowserConfig(source);
        }

        @Override
        public CitrusBrowserConfig[] newArray(int size) {
            return new CitrusBrowserConfig[size];
        }
    };
}
