/*
 * Decompiled with CFR 0.152.
 */
package com.cedarsoftware.util;

import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

public final class RegexUtilities {
    private static final String SECURITY_ENABLED_PROPERTY = "cedarsoftware.security.enabled";
    private static final String REGEX_TIMEOUT_ENABLED_PROPERTY = "cedarsoftware.regex.timeout.enabled";
    private static final String REGEX_TIMEOUT_MS_PROPERTY = "cedarsoftware.regex.timeout.milliseconds";
    private static final long DEFAULT_TIMEOUT_MS = 5000L;
    private static final Map<String, Pattern> PATTERN_CACHE = new ConcurrentHashMap<String, Pattern>();
    private static final Map<String, Pattern> PATTERN_CACHE_CI = new ConcurrentHashMap<String, Pattern>();
    private static final Map<PatternCacheKey, Pattern> PATTERN_CACHE_FLAGS = new ConcurrentHashMap<PatternCacheKey, Pattern>();
    private static final Set<String> INVALID_PATTERNS = ConcurrentHashMap.newKeySet();
    private static final Set<PatternCacheKey> INVALID_PATTERN_KEYS = ConcurrentHashMap.newKeySet();
    private static final ExecutorService REGEX_EXECUTOR = Executors.newCachedThreadPool(r -> {
        Thread thread = new Thread(r, "RegexUtilities-Timeout-Thread");
        thread.setDaemon(true);
        return thread;
    });

    private RegexUtilities() {
    }

    public static boolean isSecurityEnabled() {
        return Boolean.parseBoolean(System.getProperty(SECURITY_ENABLED_PROPERTY, "true"));
    }

    public static boolean isRegexTimeoutEnabled() {
        return Boolean.parseBoolean(System.getProperty(REGEX_TIMEOUT_ENABLED_PROPERTY, "true"));
    }

    public static long getRegexTimeoutMilliseconds() {
        return Long.parseLong(System.getProperty(REGEX_TIMEOUT_MS_PROPERTY, String.valueOf(5000L)));
    }

    public static Pattern getCachedPattern(String regex) {
        if (regex == null) {
            return null;
        }
        if (INVALID_PATTERNS.contains(regex)) {
            return null;
        }
        return PATTERN_CACHE.computeIfAbsent(regex, r -> {
            try {
                return Pattern.compile(r);
            }
            catch (PatternSyntaxException e) {
                INVALID_PATTERNS.add((String)r);
                return null;
            }
        });
    }

    public static Pattern getCachedPattern(String regex, boolean caseInsensitive) {
        if (!caseInsensitive) {
            return RegexUtilities.getCachedPattern(regex);
        }
        if (regex == null) {
            return null;
        }
        if (INVALID_PATTERNS.contains(regex)) {
            return null;
        }
        return PATTERN_CACHE_CI.computeIfAbsent(regex, r -> {
            try {
                return Pattern.compile(r, 2);
            }
            catch (PatternSyntaxException e) {
                INVALID_PATTERNS.add((String)r);
                return null;
            }
        });
    }

    public static Pattern getCachedPattern(String regex, int flags) {
        if (regex == null) {
            return null;
        }
        if (flags == 0) {
            return RegexUtilities.getCachedPattern(regex);
        }
        if (flags == 2) {
            return RegexUtilities.getCachedPattern(regex, true);
        }
        PatternCacheKey key = new PatternCacheKey(regex, flags);
        if (INVALID_PATTERN_KEYS.contains(key)) {
            return null;
        }
        return PATTERN_CACHE_FLAGS.computeIfAbsent(key, k -> {
            try {
                return Pattern.compile(((PatternCacheKey)k).regex, ((PatternCacheKey)k).flags);
            }
            catch (PatternSyntaxException e) {
                INVALID_PATTERN_KEYS.add((PatternCacheKey)k);
                return null;
            }
        });
    }

    public static void clearPatternCache() {
        PATTERN_CACHE.clear();
        PATTERN_CACHE_CI.clear();
        PATTERN_CACHE_FLAGS.clear();
        INVALID_PATTERNS.clear();
        INVALID_PATTERN_KEYS.clear();
    }

    public static Map<String, Object> getPatternCacheStats() {
        ConcurrentHashMap<String, Object> stats = new ConcurrentHashMap<String, Object>();
        stats.put("cacheSize", PATTERN_CACHE.size());
        stats.put("cacheSizeCaseInsensitive", PATTERN_CACHE_CI.size());
        stats.put("cacheSizeWithFlags", PATTERN_CACHE_FLAGS.size());
        stats.put("invalidPatternCount", INVALID_PATTERNS.size());
        stats.put("invalidPatternKeyCount", INVALID_PATTERN_KEYS.size());
        stats.put("totalCachedPatterns", PATTERN_CACHE.size() + PATTERN_CACHE_CI.size() + PATTERN_CACHE_FLAGS.size());
        return stats;
    }

    public static boolean safeMatches(Pattern pattern, String input) {
        if (pattern == null || input == null) {
            return false;
        }
        if (!RegexUtilities.isSecurityEnabled() || !RegexUtilities.isRegexTimeoutEnabled()) {
            return pattern.matcher(input).matches();
        }
        long timeout = RegexUtilities.getRegexTimeoutMilliseconds();
        Future<Boolean> future = REGEX_EXECUTOR.submit(() -> pattern.matcher(input).matches());
        try {
            return future.get(timeout, TimeUnit.MILLISECONDS);
        }
        catch (TimeoutException e) {
            future.cancel(true);
            throw new SecurityException("Regex operation timed out (>" + timeout + "ms) - possible ReDoS attack", e);
        }
        catch (Exception e) {
            throw new SecurityException("Regex operation failed: " + e.getMessage(), e);
        }
    }

    public static SafeMatchResult safeFind(Pattern pattern, String input) {
        if (pattern == null || input == null) {
            return new SafeMatchResult(null, input);
        }
        if (!RegexUtilities.isSecurityEnabled() || !RegexUtilities.isRegexTimeoutEnabled()) {
            Matcher matcher = pattern.matcher(input);
            if (matcher.find()) {
                return new SafeMatchResult(matcher, input);
            }
            return new SafeMatchResult(null, input);
        }
        long timeout = RegexUtilities.getRegexTimeoutMilliseconds();
        Future<SafeMatchResult> future = REGEX_EXECUTOR.submit(() -> {
            Matcher matcher = pattern.matcher(input);
            if (matcher.find()) {
                return new SafeMatchResult(matcher, input);
            }
            return new SafeMatchResult(null, input);
        });
        try {
            return future.get(timeout, TimeUnit.MILLISECONDS);
        }
        catch (TimeoutException e) {
            future.cancel(true);
            throw new SecurityException("Regex operation timed out (>" + timeout + "ms) - possible ReDoS attack", e);
        }
        catch (Exception e) {
            throw new SecurityException("Regex operation failed: " + e.getMessage(), e);
        }
    }

    public static String safeReplaceFirst(Pattern pattern, String input, String replacement) {
        if (pattern == null || input == null) {
            return input;
        }
        if (replacement == null) {
            replacement = "";
        }
        if (!RegexUtilities.isSecurityEnabled() || !RegexUtilities.isRegexTimeoutEnabled()) {
            return pattern.matcher(input).replaceFirst(replacement);
        }
        long timeout = RegexUtilities.getRegexTimeoutMilliseconds();
        String finalReplacement = replacement;
        Future<String> future = REGEX_EXECUTOR.submit(() -> pattern.matcher(input).replaceFirst(finalReplacement));
        try {
            return future.get(timeout, TimeUnit.MILLISECONDS);
        }
        catch (TimeoutException e) {
            future.cancel(true);
            throw new SecurityException("Regex operation timed out (>" + timeout + "ms) - possible ReDoS attack", e);
        }
        catch (Exception e) {
            throw new SecurityException("Regex operation failed: " + e.getMessage(), e);
        }
    }

    public static String safeReplaceAll(Pattern pattern, String input, String replacement) {
        if (pattern == null || input == null) {
            return input;
        }
        if (replacement == null) {
            replacement = "";
        }
        if (!RegexUtilities.isSecurityEnabled() || !RegexUtilities.isRegexTimeoutEnabled()) {
            return pattern.matcher(input).replaceAll(replacement);
        }
        long timeout = RegexUtilities.getRegexTimeoutMilliseconds();
        String finalReplacement = replacement;
        Future<String> future = REGEX_EXECUTOR.submit(() -> pattern.matcher(input).replaceAll(finalReplacement));
        try {
            return future.get(timeout, TimeUnit.MILLISECONDS);
        }
        catch (TimeoutException e) {
            future.cancel(true);
            throw new SecurityException("Regex operation timed out (>" + timeout + "ms) - possible ReDoS attack", e);
        }
        catch (Exception e) {
            throw new SecurityException("Regex operation failed: " + e.getMessage(), e);
        }
    }

    public static String[] safeSplit(Pattern pattern, String input) {
        if (pattern == null || input == null) {
            return new String[]{input};
        }
        if (!RegexUtilities.isSecurityEnabled() || !RegexUtilities.isRegexTimeoutEnabled()) {
            return pattern.split(input);
        }
        long timeout = RegexUtilities.getRegexTimeoutMilliseconds();
        Future<String[]> future = REGEX_EXECUTOR.submit(() -> pattern.split(input));
        try {
            return future.get(timeout, TimeUnit.MILLISECONDS);
        }
        catch (TimeoutException e) {
            future.cancel(true);
            throw new SecurityException("Regex operation timed out (>" + timeout + "ms) - possible ReDoS attack", e);
        }
        catch (Exception e) {
            throw new SecurityException("Regex operation failed: " + e.getMessage(), e);
        }
    }

    private static class PatternCacheKey {
        private final String regex;
        private final int flags;
        private final int hashCode;

        PatternCacheKey(String regex, int flags) {
            this.regex = regex;
            this.flags = flags;
            this.hashCode = 31 * regex.hashCode() + flags;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            PatternCacheKey that = (PatternCacheKey)o;
            return this.flags == that.flags && this.regex.equals(that.regex);
        }

        public int hashCode() {
            return this.hashCode;
        }
    }

    public static class SafeMatchResult {
        private final String[] groups;
        private final String replacement;
        private final boolean matched;
        private final int start;
        private final int end;

        public SafeMatchResult(Matcher matcher, String originalInput) {
            if (matcher != null) {
                int groupCount = matcher.groupCount();
                this.groups = new String[groupCount + 1];
                for (int i = 0; i <= groupCount; ++i) {
                    this.groups[i] = matcher.group(i);
                }
                this.replacement = matcher.replaceFirst("");
                this.matched = true;
                this.start = matcher.start();
                this.end = matcher.end();
            } else {
                this.groups = new String[0];
                this.replacement = originalInput;
                this.matched = false;
                this.start = -1;
                this.end = -1;
            }
        }

        public String group(int index) {
            if (index < 0 || index >= this.groups.length) {
                return null;
            }
            return this.groups[index];
        }

        public String group() {
            return this.group(0);
        }

        public int groupCount() {
            return this.matched ? this.groups.length - 1 : 0;
        }

        public boolean matched() {
            return this.matched;
        }

        public int start() {
            return this.start;
        }

        public int end() {
            return this.end;
        }

        public String getReplacement() {
            return this.replacement;
        }
    }
}

