/*
 * Decompiled with CFR 0.152.
 */
package com.cedarsoftware.util;

import com.cedarsoftware.util.Convention;
import com.cedarsoftware.util.RegexUtilities;
import com.cedarsoftware.util.StringUtilities;
import java.math.BigDecimal;
import java.time.DateTimeException;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.zone.ZoneRulesException;
import java.util.Collections;
import java.util.Date;
import java.util.Map;
import java.util.TimeZone;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Pattern;

public final class DateUtilities {
    private static final int DEFAULT_MAX_INPUT_LENGTH = 1000;
    private static final int DEFAULT_MAX_EPOCH_DIGITS = 19;
    private static final long DEFAULT_REGEX_TIMEOUT_MILLISECONDS = 1000L;
    private static final Pattern allDigits;
    private static final String days = "monday|mon|tuesday|tues|tue|wednesday|wed|thursday|thur|thu|friday|fri|saturday|sat|sunday|sun";
    private static final String mos = "January|Jan|February|Feb|March|Mar|April|Apr|May|June|Jun|July|Jul|August|Aug|September|Sept|Sep|October|Oct|November|Nov|December|Dec";
    private static final String yr = "[+-]?\\d{4,9}\\b";
    private static final String d1or2 = "\\d{1,2}";
    private static final String d2 = "\\d{2}";
    private static final String ord = "st|nd|rd|th";
    private static final String sep = "[./-]";
    private static final String ws = "\\s+";
    private static final String wsOp = "\\s*";
    private static final String wsOrComma = "[ ,]+";
    private static final String tzUnix = "[A-Z]{1,3}";
    private static final String tz_Hh_MM = "[+-]\\d{1,2}:\\d{2}";
    private static final String tz_Hh_MM_SS = "[+-]\\d{1,2}:\\d{2}:\\d{2}";
    private static final String tz_HHMM = "[+-]\\d{4}";
    private static final String tz_Hh = "[+-]\\d{1,2}";
    private static final String tzNamed = "\\s*\\[?(?:GMT[+-]\\d{2}:\\d{2}|[A-Za-z][A-Za-z0-9~/._+-]{1,50})]?";
    private static final String nano = "\\.\\d{1,9}";
    private static final Pattern isoDatePattern;
    private static final Pattern alphaMonthPattern;
    private static final Pattern unixDateTimePattern;
    private static final Pattern timePattern;
    private static final Pattern zonePattern;
    private static final Pattern dayPattern;
    private static final Map<String, Integer> months;
    public static final Map<String, String> ABBREVIATION_TO_TIMEZONE;

    private static void initializeSystemPropertyDefaults() {
        if (System.getProperty("dateutilities.max.input.length") == null) {
            System.setProperty("dateutilities.max.input.length", String.valueOf(1000));
        }
        if (System.getProperty("dateutilities.max.epoch.digits") == null) {
            System.setProperty("dateutilities.max.epoch.digits", String.valueOf(19));
        }
        if (System.getProperty("dateutilities.regex.timeout.milliseconds") == null) {
            System.setProperty("dateutilities.regex.timeout.milliseconds", String.valueOf(1000L));
        }
    }

    private static boolean isSecurityEnabled() {
        return Boolean.parseBoolean(System.getProperty("dateutilities.security.enabled", "false"));
    }

    private static boolean isInputValidationEnabled() {
        return Boolean.parseBoolean(System.getProperty("dateutilities.input.validation.enabled", "false"));
    }

    private static boolean isRegexTimeoutEnabled() {
        return Boolean.parseBoolean(System.getProperty("dateutilities.regex.timeout.enabled", "false"));
    }

    private static boolean isMalformedStringProtectionEnabled() {
        return Boolean.parseBoolean(System.getProperty("dateutilities.malformed.string.protection.enabled", "false"));
    }

    private static int getMaxInputLength() {
        String maxLengthProp = System.getProperty("dateutilities.max.input.length");
        if (maxLengthProp != null) {
            try {
                return Math.max(1, Integer.parseInt(maxLengthProp));
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
        }
        return DateUtilities.isSecurityEnabled() ? 1000 : Integer.MAX_VALUE;
    }

    private static int getMaxEpochDigits() {
        String maxDigitsProp = System.getProperty("dateutilities.max.epoch.digits");
        if (maxDigitsProp != null) {
            try {
                return Math.max(1, Integer.parseInt(maxDigitsProp));
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
        }
        return DateUtilities.isSecurityEnabled() ? 19 : Integer.MAX_VALUE;
    }

    private static long getRegexTimeoutMilliseconds() {
        String timeoutProp = System.getProperty("dateutilities.regex.timeout.milliseconds");
        if (timeoutProp != null) {
            try {
                return Math.max(1L, Long.parseLong(timeoutProp));
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
        }
        return DateUtilities.isSecurityEnabled() ? 1000L : Long.MAX_VALUE;
    }

    private static void validateMalformedInput(String input) {
        if (input.matches(".*(.{10,})\\1{5,}.*")) {
            throw new SecurityException("Input contains excessive repetition patterns that could cause ReDoS");
        }
        int openParens = 0;
        int maxNesting = 0;
        for (char c : input.toCharArray()) {
            if (c == '(') {
                maxNesting = Math.max(maxNesting, ++openParens);
                continue;
            }
            if (c != ')') continue;
            --openParens;
        }
        if (maxNesting > 20) {
            throw new SecurityException("Input contains excessive nesting that could cause parsing issues");
        }
        if (input.matches(".*[<>&\"'\\x00-\\x08\\x0B\\x0C\\x0E-\\x1F\\x7F].*")) {
            throw new SecurityException("Input contains invalid characters for date parsing");
        }
    }

    private DateUtilities() {
    }

    public static Date parseDate(String dateStr) {
        if (StringUtilities.isEmpty(dateStr)) {
            return null;
        }
        ZonedDateTime dateTime = DateUtilities.parseDate(dateStr, ZoneId.systemDefault(), true);
        Instant instant = Instant.from(dateTime);
        return Date.from(instant);
    }

    public static ZonedDateTime parseDate(String dateStr, ZoneId defaultZoneId, boolean ensureDateTimeAlone) {
        ZoneId zoneId;
        String remains;
        String day;
        int month;
        String year;
        if ((dateStr = StringUtilities.trimToNull(dateStr)) == null) {
            return null;
        }
        Convention.throwIfNull(defaultZoneId, "ZoneId cannot be null.  Use ZoneId.of(\"America/New_York\"), ZoneId.systemDefault(), etc.");
        if (DateUtilities.isSecurityEnabled() && DateUtilities.isInputValidationEnabled()) {
            int maxLength = DateUtilities.getMaxInputLength();
            if (dateStr.length() > maxLength) {
                throw new SecurityException("Date string too long (max " + maxLength + " characters): " + dateStr.length());
            }
        }
        if (DateUtilities.isSecurityEnabled() && DateUtilities.isMalformedStringProtectionEnabled()) {
            DateUtilities.validateMalformedInput(dateStr);
        }
        if (RegexUtilities.safeMatches(allDigits, dateStr)) {
            long epochMillis;
            if (DateUtilities.isSecurityEnabled() && DateUtilities.isInputValidationEnabled()) {
                int maxEpochDigits = DateUtilities.getMaxEpochDigits();
                if (dateStr.length() > maxEpochDigits) {
                    throw new SecurityException("Epoch milliseconds value too large (max " + maxEpochDigits + " digits): " + dateStr.length());
                }
            }
            try {
                epochMillis = Long.parseLong(dateStr);
            }
            catch (NumberFormatException e) {
                throw new IllegalArgumentException("Invalid epoch milliseconds: " + dateStr, e);
            }
            return Instant.ofEpochMilli(epochMillis).atZone(defaultZoneId);
        }
        String tz = null;
        RegexUtilities.SafeMatchResult result = RegexUtilities.safeFind(isoDatePattern, dateStr);
        String remnant = result.getReplacement();
        if (remnant.length() < dateStr.length()) {
            if (result.group(1) != null) {
                year = result.group(1);
                month = Integer.parseInt(result.group(3));
                day = result.group(4);
            } else {
                year = result.group(8);
                month = Integer.parseInt(result.group(5));
                day = result.group(7);
            }
            remains = remnant;
            if (remnant.startsWith("T") ? RegexUtilities.safeMatches(zonePattern, remnant.substring(1)) : RegexUtilities.safeMatches(zonePattern, remnant)) {
                throw new IllegalArgumentException("Time zone information without time is invalid: " + dateStr);
            }
        } else {
            String mon;
            result = RegexUtilities.safeFind(alphaMonthPattern, dateStr);
            remnant = result.getReplacement();
            if (remnant.length() < dateStr.length()) {
                if (result.group(1) != null) {
                    mon = result.group(1);
                    day = result.group(2);
                    year = result.group(4);
                    remains = remnant;
                } else if (result.group(7) != null) {
                    mon = result.group(7);
                    day = result.group(5);
                    year = result.group(8);
                    remains = remnant;
                } else {
                    year = result.group(9);
                    mon = result.group(10);
                    day = result.group(11);
                    remains = remnant;
                }
                month = months.get(mon.trim().toLowerCase());
            } else {
                result = RegexUtilities.safeFind(unixDateTimePattern, dateStr);
                if (result.getReplacement().length() == dateStr.length()) {
                    throw new IllegalArgumentException("Unable to parse: " + dateStr + " as a date-time");
                }
                year = result.group(6);
                mon = result.group(2);
                month = months.get(mon.trim().toLowerCase());
                day = result.group(3);
                tz = result.group(5);
                remains = result.group(4);
            }
        }
        String hour = null;
        String min = null;
        String sec = "00";
        String fracSec = "0";
        result = RegexUtilities.safeFind(timePattern, remains = remains.trim());
        remnant = result.getReplacement();
        if (remnant.length() < remains.length()) {
            hour = result.group(1);
            min = result.group(2);
            if (result.group(3) != null) {
                sec = result.group(3);
            }
            if (result.group(4) != null) {
                fracSec = "0" + result.group(4);
            }
            if (result.group(5) != null) {
                tz = result.group(5).trim();
            }
            if (result.group(6) != null) {
                tz = DateUtilities.stripBrackets(result.group(6).trim());
            }
        }
        if (ensureDateTimeAlone) {
            DateUtilities.verifyNoGarbageLeft(remnant);
        }
        try {
            zoneId = StringUtilities.isEmpty(tz) ? defaultZoneId : DateUtilities.getTimeZone(tz);
        }
        catch (Exception e) {
            if (ensureDateTimeAlone) {
                throw e;
            }
            zoneId = defaultZoneId;
        }
        return DateUtilities.getDate(dateStr, zoneId, year, month, day, hour, min, sec, fracSec);
    }

    private static ZonedDateTime getDate(String dateStr, ZoneId zoneId, String year, int month, String day, String hour, String min, String sec, String fracSec) {
        int y = Integer.parseInt(year);
        int d = Integer.parseInt(day);
        if (y < -999999999 || y > 999999999) {
            throw new IllegalArgumentException("Year must be between -999999999 and 999999999 inclusive, date: " + dateStr);
        }
        if (month < 1 || month > 12) {
            throw new IllegalArgumentException("Month must be between 1 and 12 inclusive, date: " + dateStr);
        }
        if (d < 1 || d > 31) {
            throw new IllegalArgumentException("Day must be between 1 and 31 inclusive, date: " + dateStr);
        }
        if (hour == null) {
            return ZonedDateTime.of(y, month, d, 0, 0, 0, 0, zoneId);
        }
        int h = Integer.parseInt(hour);
        int mn = Integer.parseInt(min);
        int s = Integer.parseInt(sec);
        long nanoOfSec = DateUtilities.convertFractionToNanos(fracSec);
        if (h > 23) {
            throw new IllegalArgumentException("Hour must be between 0 and 23 inclusive, time: " + dateStr);
        }
        if (mn > 59) {
            throw new IllegalArgumentException("Minute must be between 0 and 59 inclusive, time: " + dateStr);
        }
        if (s > 59) {
            throw new IllegalArgumentException("Second must be between 0 and 59 inclusive, time: " + dateStr);
        }
        return ZonedDateTime.of(y, month, d, h, mn, s, (int)nanoOfSec, zoneId);
    }

    private static long convertFractionToNanos(String fracSec) {
        if (StringUtilities.isEmpty(fracSec)) {
            return 0L;
        }
        BigDecimal fractional = new BigDecimal(fracSec);
        BigDecimal nanos = fractional.movePointRight(9);
        if (nanos.compareTo(BigDecimal.ZERO) < 0 || nanos.compareTo(BigDecimal.valueOf(1000000000L)) >= 0) {
            throw new IllegalArgumentException("Invalid fractional second: " + fracSec);
        }
        return nanos.longValue();
    }

    private static ZoneId getTimeZone(String tz) {
        if (tz == null || tz.isEmpty()) {
            return ZoneId.systemDefault();
        }
        if (tz.length() > 100) {
            throw new IllegalArgumentException("Timezone string too long (max 100 characters): " + tz.length());
        }
        for (int i = 0; i < tz.length(); ++i) {
            char c = tz.charAt(i);
            if (c >= ' ' && c != '\u007f') continue;
            throw new IllegalArgumentException("Invalid timezone string contains control characters");
        }
        if (tz.startsWith("-") || tz.startsWith("+")) {
            try {
                ZoneOffset offset = ZoneOffset.of(tz);
                return ZoneId.ofOffset("GMT", offset);
            }
            catch (DateTimeException e) {
                throw e;
            }
            catch (Exception e) {
                throw new IllegalArgumentException("Invalid timezone offset format: " + tz.substring(0, Math.min(tz.length(), 20)));
            }
        }
        if (tz.equalsIgnoreCase("GMT")) {
            return ZoneId.of("Etc/GMT");
        }
        String mappedZone = ABBREVIATION_TO_TIMEZONE.get(tz.toUpperCase());
        if (mappedZone != null) {
            try {
                return ZoneId.of(mappedZone);
            }
            catch (Exception e) {
                throw new IllegalArgumentException("Invalid timezone abbreviation: " + tz.substring(0, Math.min(tz.length(), 10)));
            }
        }
        try {
            return ZoneId.of(tz);
        }
        catch (ZoneRulesException zoneRulesEx) {
            try {
                TimeZone timeZone = TimeZone.getTimeZone(tz);
                if (timeZone.getID().equals("GMT") && !tz.equalsIgnoreCase("GMT")) {
                    throw zoneRulesEx;
                }
                String timeZoneId = timeZone.getID();
                if (timeZoneId.length() > 50) {
                    throw new IllegalArgumentException("Invalid timezone ID returned by system");
                }
                return timeZone.toZoneId();
            }
            catch (ZoneRulesException ex) {
                throw ex;
            }
            catch (Exception fallbackEx) {
                throw zoneRulesEx;
            }
        }
        catch (Exception otherEx) {
            try {
                TimeZone timeZone = TimeZone.getTimeZone(tz);
                if (timeZone.getID().equals("GMT") && !tz.equalsIgnoreCase("GMT")) {
                    throw new IllegalArgumentException("Unrecognized timezone: " + tz.substring(0, Math.min(tz.length(), 20)));
                }
                String timeZoneId = timeZone.getID();
                if (timeZoneId.length() > 50) {
                    throw new IllegalArgumentException("Invalid timezone ID returned by system");
                }
                return timeZone.toZoneId();
            }
            catch (Exception fallbackEx) {
                throw new IllegalArgumentException("Invalid timezone format: " + tz.substring(0, Math.min(tz.length(), 20)));
            }
        }
    }

    private static void verifyNoGarbageLeft(String remnant) {
        if (StringUtilities.length(remnant) > 0) {
            RegexUtilities.SafeMatchResult dayResult = RegexUtilities.safeFind(dayPattern, remnant);
            remnant = dayResult.getReplacement().trim();
        }
        if (StringUtilities.length(remnant) > 0 && !(remnant = remnant.replaceAll("[T,]", "").trim()).isEmpty()) {
            throw new IllegalArgumentException("Issue parsing date-time, other characters present: " + remnant);
        }
    }

    private static String stripBrackets(String input) {
        if (input == null || input.isEmpty()) {
            return input;
        }
        return input.replaceAll("^\\[|]$", "");
    }

    static {
        DateUtilities.initializeSystemPropertyDefaults();
        allDigits = Pattern.compile("^-?\\d+$", 256);
        isoDatePattern = Pattern.compile("([+-]?\\d{4,9}\\b)([./-])(\\d{1,2})\\2(\\d{1,2})|(\\d{1,2})([./-])(\\d{1,2})\\6([+-]?\\d{4,9}\\b)", 256);
        alphaMonthPattern = Pattern.compile("\\b(January|Jan|February|Feb|March|Mar|April|Apr|May|June|Jun|July|Jul|August|Aug|September|Sept|Sep|October|Oct|November|Nov|December|Dec)\\b[ ,]+(\\d{1,2})(st|nd|rd|th)?[ ,]+([+-]?\\d{4,9}\\b)|(\\d{1,2})(st|nd|rd|th)?[ ,]+\\b(January|Jan|February|Feb|March|Mar|April|Apr|May|June|Jun|July|Jul|August|Aug|September|Sept|Sep|October|Oct|November|Nov|December|Dec)\\b[ ,]+([+-]?\\d{4,9}\\b)|([+-]?\\d{4,9}\\b)[ ,]+\\b(January|Jan|February|Feb|March|Mar|April|Apr|May|June|Jun|July|Jul|August|Aug|September|Sept|Sep|October|Oct|November|Nov|December|Dec\\b)[ ,]+(\\d{1,2})(st|nd|rd|th)?", 258);
        unixDateTimePattern = Pattern.compile("(?:\\b(monday|mon|tuesday|tues|tue|wednesday|wed|thursday|thur|thu|friday|fri|saturday|sat|sunday|sun)\\b\\s+)?\\b(January|Jan|February|Feb|March|Mar|April|Apr|May|June|Jun|July|Jul|August|Aug|September|Sept|Sep|October|Oct|November|Nov|December|Dec)\\b\\s+(\\d{1,2})\\s+(\\d{2}:\\d{2}:\\d{2})\\s*([A-Z]{1,3})?\\s*([+-]?\\d{4,9}\\b)", 258);
        timePattern = Pattern.compile("(\\d{2}):(\\d{2})(?::(\\d{2})(\\.\\d{1,9})?)?([+-]\\d{1,2}:\\d{2}:\\d{2}|[+-]\\d{1,2}:\\d{2}|[+-]\\d{4}|[+-]\\d{1,2}|Z)?(\\s*\\[?(?:GMT[+-]\\d{2}:\\d{2}|[A-Za-z][A-Za-z0-9~/._+-]{1,50})]?)?", 258);
        zonePattern = Pattern.compile("([+-]\\d{1,2}:\\d{2}|[+-]\\d{4}|[+-]\\d{1,2}:\\d{2}:\\d{2}|[+-]\\d{1,2}|Z|\\s*\\[?(?:GMT[+-]\\d{2}:\\d{2}|[A-Za-z][A-Za-z0-9~/._+-]{1,50})]?)", 258);
        dayPattern = Pattern.compile("\\b(monday|mon|tuesday|tues|tue|wednesday|wed|thursday|thur|thu|friday|fri|saturday|sat|sunday|sun)\\b", 258);
        months = new ConcurrentHashMap<String, Integer>();
        months.put("jan", 1);
        months.put("january", 1);
        months.put("feb", 2);
        months.put("february", 2);
        months.put("mar", 3);
        months.put("march", 3);
        months.put("apr", 4);
        months.put("april", 4);
        months.put("may", 5);
        months.put("jun", 6);
        months.put("june", 6);
        months.put("jul", 7);
        months.put("july", 7);
        months.put("aug", 8);
        months.put("august", 8);
        months.put("sep", 9);
        months.put("sept", 9);
        months.put("september", 9);
        months.put("oct", 10);
        months.put("october", 10);
        months.put("nov", 11);
        months.put("november", 11);
        months.put("dec", 12);
        months.put("december", 12);
        ConcurrentHashMap<String, String> timezoneBuilder = new ConcurrentHashMap<String, String>();
        timezoneBuilder.put("EST", "America/New_York");
        timezoneBuilder.put("EDT", "America/New_York");
        timezoneBuilder.put("CST", "America/Chicago");
        timezoneBuilder.put("CDT", "America/Chicago");
        timezoneBuilder.put("MST", "America/Denver");
        timezoneBuilder.put("MDT", "America/Denver");
        timezoneBuilder.put("PST", "America/Los_Angeles");
        timezoneBuilder.put("PDT", "America/Los_Angeles");
        timezoneBuilder.put("AKST", "America/Anchorage");
        timezoneBuilder.put("AKDT", "America/Anchorage");
        timezoneBuilder.put("HST", "Pacific/Honolulu");
        timezoneBuilder.put("GMT", "Europe/London");
        timezoneBuilder.put("BST", "Europe/London");
        timezoneBuilder.put("WET", "Europe/Lisbon");
        timezoneBuilder.put("WEST", "Europe/Lisbon");
        timezoneBuilder.put("CET", "Europe/Berlin");
        timezoneBuilder.put("CEST", "Europe/Berlin");
        timezoneBuilder.put("EET", "Europe/Kiev");
        timezoneBuilder.put("EEST", "Europe/Kiev");
        timezoneBuilder.put("AEST", "Australia/Brisbane");
        timezoneBuilder.put("AEDT", "Australia/Sydney");
        timezoneBuilder.put("ACST", "Australia/Darwin");
        timezoneBuilder.put("ACDT", "Australia/Adelaide");
        timezoneBuilder.put("AWST", "Australia/Perth");
        timezoneBuilder.put("NZST", "Pacific/Auckland");
        timezoneBuilder.put("NZDT", "Pacific/Auckland");
        timezoneBuilder.put("CLT", "America/Santiago");
        timezoneBuilder.put("CLST", "America/Santiago");
        timezoneBuilder.put("PYT", "America/Asuncion");
        timezoneBuilder.put("PYST", "America/Asuncion");
        timezoneBuilder.put("ART", "America/Argentina/Buenos_Aires");
        timezoneBuilder.put("IST", "Asia/Kolkata");
        timezoneBuilder.put("IDT", "Asia/Jerusalem");
        timezoneBuilder.put("IRST", "Asia/Tehran");
        timezoneBuilder.put("IRDT", "Asia/Tehran");
        timezoneBuilder.put("WAT", "Africa/Lagos");
        timezoneBuilder.put("CAT", "Africa/Harare");
        timezoneBuilder.put("JST", "Asia/Tokyo");
        timezoneBuilder.put("KST", "Asia/Seoul");
        timezoneBuilder.put("HKT", "Asia/Hong_Kong");
        timezoneBuilder.put("SGT", "Asia/Singapore");
        timezoneBuilder.put("MYT", "Asia/Kuala_Lumpur");
        timezoneBuilder.put("MSK", "Europe/Moscow");
        timezoneBuilder.put("MSD", "Europe/Moscow");
        timezoneBuilder.put("EAT", "Africa/Nairobi");
        timezoneBuilder.put("ICT", "Asia/Bangkok");
        timezoneBuilder.put("COT", "America/Bogota");
        timezoneBuilder.put("PET", "America/Lima");
        timezoneBuilder.put("PKT", "Asia/Karachi");
        timezoneBuilder.put("WIB", "Asia/Jakarta");
        ABBREVIATION_TO_TIMEZONE = Collections.unmodifiableMap(timezoneBuilder);
    }
}

