/*
 * Decompiled with CFR 0.152.
 */
package com.caucho.db.block;

import com.caucho.db.block.BlockReadWrite;
import com.caucho.db.block.BlockState;
import com.caucho.db.block.BlockStore;
import com.caucho.util.FreeList;
import com.caucho.util.L10N;
import com.caucho.util.SyncCacheListener;
import java.io.IOException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.logging.Level;
import java.util.logging.Logger;

public final class Block
implements SyncCacheListener {
    private static final L10N L = new L10N(Block.class);
    private static final Logger log = Logger.getLogger(Block.class.getName());
    private static final long INIT_DIRTY = 8192L;
    private static final FreeList<byte[]> _freeBuffers = new FreeList(256);
    private static final FreeList<ReadWriteLock> _freeLocks = new FreeList(256);
    private final BlockStore _store;
    private final long _blockId;
    private ReadWriteLock _rwLock;
    private final Lock _readLock;
    private final Lock _writeLock;
    private final AtomicReference<BlockState> _state = new AtomicReference<BlockState>(BlockState.INIT);
    private final AtomicInteger _useCount = new AtomicInteger(1);
    private final AtomicLong _dirtyRange = new AtomicLong(8192L);
    private final boolean _isLogFine = log.isLoggable(Level.FINE);
    private volatile boolean _isFreeBuffer = true;
    private boolean _isFlushDirtyOnCommit;
    private byte[] _buffer;
    private boolean _isCopy;
    private boolean _isLoad;
    private boolean _isDirty;

    Block(BlockStore store, long blockId) {
        store.validateBlockId(blockId);
        this._store = store;
        this._blockId = blockId;
        this._rwLock = Block.allocateReadWriteLock();
        this._readLock = this._rwLock.readLock();
        this._writeLock = this._rwLock.writeLock();
        this._isFlushDirtyOnCommit = this._store.isFlushDirtyBlocksOnCommit();
        this._buffer = Block.allocateBuffer();
        if (log.isLoggable(Level.ALL)) {
            log.log(Level.ALL, this + " create");
        }
    }

    private static ReadWriteLock allocateReadWriteLock() {
        ReadWriteLock lock = (ReadWriteLock)_freeLocks.allocate();
        if (lock == null) {
            lock = new ReentrantReadWriteLock();
        }
        return lock;
    }

    public boolean isFlushDirtyOnCommit() {
        return this._isFlushDirtyOnCommit;
    }

    public void setFlushDirtyOnCommit(boolean isFlush) {
        this._isFlushDirtyOnCommit = isFlush;
    }

    public boolean isIndex() {
        long blockIndex = BlockStore.blockIdToIndex(this.getBlockId());
        return this.getStore().getAllocation(blockIndex) == 4;
    }

    public void validateIsIndex() {
        long blockIndex = BlockStore.blockIdToIndex(this.getBlockId());
        int allocCode = this.getStore().getAllocation(blockIndex);
        if (allocCode != 4) {
            IllegalStateException exn = new IllegalStateException(L.l("block {0} is not an index code={1}", (Object)this, allocCode));
            exn.fillInStackTrace();
            throw exn;
        }
    }

    public BlockStore getStore() {
        return this._store;
    }

    public long getBlockId() {
        return this._blockId;
    }

    public final Lock getReadLock() {
        return this._readLock;
    }

    public final Lock getWriteLock() {
        return this._writeLock;
    }

    public final boolean isValid() {
        return this._state.get().isValid();
    }

    public boolean isDirty() {
        return this._dirtyRange.get() != 8192L;
    }

    public final boolean allocate() {
        int useCount;
        do {
            if ((useCount = this._useCount.get()) >= 1) continue;
            return false;
        } while (!this._useCount.compareAndSet(useCount, useCount + 1));
        if (this._isLogFine) {
            if (this.getBuffer() == null) {
                this._useCount.decrementAndGet();
                Thread.dumpStack();
                log.fine(this + " null buffer " + this + " " + this._useCount.get());
                return false;
            }
            if (log.isLoggable(Level.ALL)) {
                log.log(Level.ALL, this + " allocate (" + useCount + ")");
            }
            if (useCount > 32 && log.isLoggable(Level.FINE)) {
                log.fine("using " + this + " " + useCount + " times");
                if (log.isLoggable(Level.FINER)) {
                    Thread.dumpStack();
                }
            }
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void read() throws IOException {
        if (this._state.get().isValid()) {
            return;
        }
        Block block = this;
        synchronized (block) {
            if (!this._state.get().isValid()) {
                if (this._state.get().isDestroyed()) {
                    throw new IllegalStateException(this.toString());
                }
                if (!this._store.getBlockManager().copyDirtyBlock(this)) {
                    if (log.isLoggable(Level.ALL)) {
                        log.log(Level.ALL, "read db-block " + this);
                    }
                    this._isLoad = true;
                    BlockReadWrite readWrite = this._store.getReadWrite();
                    this.clearDirty();
                    readWrite.readBlock(this._blockId & 0xFFFFFFFFFFFFE000L, this.getBuffer(), 0, 8192);
                    this.toValid();
                }
            }
        }
    }

    public final byte[] getBuffer() {
        return this._buffer;
    }

    public final void setDirty(int min, int max) {
        int dirtyMin;
        int dirtyMax;
        long newDirty;
        long oldDirty;
        if (8192 < max || min < 0 || max < min) {
            throw new IllegalStateException("min=" + min + ", max=" + max);
        }
        this._isDirty = true;
        do {
            oldDirty = this._dirtyRange.get();
            dirtyMax = (int)(oldDirty >> 32);
            dirtyMin = (int)oldDirty;
            if (min < dirtyMin) {
                dirtyMin = min;
            }
            if (dirtyMax >= max) continue;
            dirtyMax = max;
        } while (!this._dirtyRange.compareAndSet(oldDirty, newDirty = ((long)dirtyMax << 32) + (long)dirtyMin));
    }

    public void debugFd() {
        byte[] buffer = this._buffer;
        for (int i = 0; i < buffer.length; ++i) {
            if ((buffer[i] & 0xFF) != 253) continue;
            System.err.println("ID: " + i + " " + this);
            Thread.dumpStack();
        }
    }

    public final void setDirtyExact(int min, int max) {
        long newDirty = ((long)max << 32) + (long)min;
        this._dirtyRange.set(newDirty);
    }

    private void clearDirty() {
        this._dirtyRange.set(8192L);
    }

    public void commit() throws IOException {
        if (this._isFlushDirtyOnCommit) {
            this.save();
        }
    }

    public void commitNoWake() throws IOException {
        if (this._isFlushDirtyOnCommit) {
            this.saveNoWake();
        }
    }

    public int getUseCount() {
        return this._useCount.get();
    }

    public void deallocate() throws IOException {
        try {
            this.getStore().deallocateBlock(this.getBlockId());
        }
        catch (Exception e) {
            log.log(Level.FINER, e.toString(), e);
        }
    }

    public void saveAllocation() throws IOException {
        this.getStore().saveAllocation();
    }

    public final void free() {
        if (log.isLoggable(Level.ALL)) {
            log.log(Level.ALL, this + " free (" + this._useCount.get() + ")");
        }
        this.releaseUse();
    }

    public boolean startLruRemove() {
        return this._useCount.compareAndSet(1, 0);
    }

    public final void syncLruRemoveEvent() {
        this.syncRemoveEvent();
    }

    public final void syncRemoveEvent() {
        if (this._useCount.get() > 1 || this._dirtyRange.get() != 8192L) {
            this.saveNoWake();
        }
        this.releaseUse();
    }

    public boolean save() {
        if (this._dirtyRange.get() == 8192L) {
            return false;
        }
        if (this.toWriteQueued()) {
            this._store.getWriter().addDirtyBlock(this);
        }
        return true;
    }

    private void saveNoWake() {
        if (this.toWriteQueued()) {
            this._store.getWriter().addDirtyBlockNoWake(this);
        }
    }

    void writeFromBlockWriter() throws IOException {
        do {
            int dirtyMax;
            long dirty;
            int dirtyMin;
            if ((dirtyMin = (int)(dirty = this._dirtyRange.getAndSet(8192L))) < (dirtyMax = (int)(dirty >> 32))) {
                if (log.isLoggable(Level.ALL)) {
                    log.log(Level.ALL, "write db-block " + this + " [" + dirtyMin + ", " + dirtyMax + "]");
                }
                boolean isPriority = false;
                this.writeImpl(dirtyMin, dirtyMax - dirtyMin, isPriority);
            }
            if (this._dirtyRange.get() != 8192L || this.isDestroyed()) continue;
            this.toValid();
        } while (this._dirtyRange.get() != 8192L);
        if (this._useCount.get() <= 0) {
            this.freeImpl();
        }
    }

    private void writeImpl(int offset, int length, boolean isPriority) throws IOException {
        BlockReadWrite readWrite = this._store.getReadWrite();
        readWrite.writeBlock((this._blockId & 0xFFFFFFFFFFFFE000L) + (long)offset, this.getBuffer(), offset, length, isPriority);
    }

    boolean copyToBlock(Block block) {
        boolean isValid;
        if (block == this) {
            return true;
        }
        this._isFreeBuffer = false;
        block._isFreeBuffer = false;
        byte[] buffer = this._buffer;
        byte[] targetBuffer = block.getBuffer();
        if (buffer != null && targetBuffer != null) {
            System.arraycopy(buffer, 0, block.getBuffer(), 0, buffer.length);
        }
        boolean bl = isValid = this.isValid() && this._buffer != null && block.getBuffer() != null;
        if (isValid) {
            block.toValid();
            block._isCopy = true;
        }
        return isValid;
    }

    private void releaseUse() {
        if (this._useCount.decrementAndGet() <= 0) {
            this.freeImpl();
        }
    }

    private void freeImpl() {
        if (this._useCount.get() > 0) {
            throw new IllegalStateException("freeImpl");
        }
        if (this.toDestroy()) {
            byte[] buffer = this._buffer;
            this._buffer = null;
            if (buffer != null && this._isFreeBuffer) {
                _freeBuffers.free((Object)buffer);
            }
            ReadWriteLock lock = this._rwLock;
            this._rwLock = null;
            if (lock != null) {
                _freeLocks.free((Object)lock);
            }
        }
    }

    public boolean isDestroyed() {
        return this._state.get().isDestroyed();
    }

    private boolean toDestroy() {
        BlockState newState;
        BlockState oldState;
        while (!this._state.compareAndSet(oldState = this._state.get(), newState = oldState.toDestroy())) {
        }
        return newState.isDestroyed() && !oldState.isDestroyed();
    }

    private boolean toWriteQueued() {
        BlockState newState;
        BlockState oldState;
        AtomicReference<BlockState> state = this._state;
        while (!state.compareAndSet(oldState = state.get(), newState = oldState.toWrite())) {
        }
        return newState.isWrite() && !oldState.isWrite();
    }

    void toValid() {
        BlockState newState;
        BlockState oldState;
        while (!this._state.compareAndSet(oldState = this._state.get(), newState = oldState.toValid())) {
        }
    }

    private static byte[] allocateBuffer() {
        byte[] buffer = (byte[])_freeBuffers.allocate();
        if (buffer == null) {
            buffer = new byte[8192];
        }
        return buffer;
    }

    private BlockState toState(BlockState toState) {
        BlockState newState;
        BlockState oldState;
        while (!this._state.compareAndSet(oldState = this._state.get(), newState = oldState.toState(toState))) {
        }
        return oldState;
    }

    public String toString() {
        return this.getClass().getSimpleName() + "[" + this._store + "," + Long.toHexString(this._blockId) + ",copy=" + this._isCopy + ",load=" + this._isLoad + ",dirty=" + this._isDirty + "]";
    }
}

