package com.bugsnag.android;

import static com.bugsnag.android.ConfigFactory.MF_BUILD_UUID;
import static com.bugsnag.android.MapUtils.getStringFromMap;

import com.bugsnag.android.NativeInterface.Message;

import android.app.Activity;
import android.app.Application;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.SharedPreferences;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;
import android.view.OrientationEventListener;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.Map;
import java.util.Observable;
import java.util.Observer;
import java.util.concurrent.RejectedExecutionException;

/**
 * A Bugsnag Client instance allows you to use Bugsnag in your Android app.
 * Typically you'd instead use the static access provided in the Bugsnag class.
 * <p/>
 * Example usage:
 * <p/>
 * Client client = new Client(this, "your-api-key");
 * client.notify(new RuntimeException("something broke!"));
 *
 * @see Bugsnag
 */
@SuppressWarnings("checkstyle:JavadocTagContinuationIndentation")
public class Client extends Observable implements Observer {

    private static final boolean BLOCKING = true;
    private static final String SHARED_PREF_KEY = "com.bugsnag.android";
    private static final String USER_ID_KEY = "user.id";
    private static final String USER_NAME_KEY = "user.name";
    private static final String USER_EMAIL_KEY = "user.email";

    @NonNull
    protected final Configuration config;
    final Context appContext;

    @NonNull
    protected final DeviceData deviceData;

    @NonNull
    protected final AppData appData;

    @NonNull
    final Breadcrumbs breadcrumbs;

    @NonNull
    private final User user = new User();

    @NonNull
    protected final ErrorStore errorStore;

    final SessionStore sessionStore;

    private final EventReceiver eventReceiver;
    final SessionTracker sessionTracker;
    SharedPreferences sharedPrefs;

    private final OrientationEventListener orientationListener;

    /**
     * Initialize a Bugsnag client
     *
     * @param androidContext an Android context, usually <code>this</code>
     */
    public Client(@NonNull Context androidContext) {
        this(androidContext, null, true);
    }

    /**
     * Initialize a Bugsnag client
     *
     * @param androidContext an Android context, usually <code>this</code>
     * @param apiKey         your Bugsnag API key from your Bugsnag dashboard
     */
    public Client(@NonNull Context androidContext, @Nullable String apiKey) {
        this(androidContext, apiKey, true);
    }

    /**
     * Initialize a Bugsnag client
     *
     * @param androidContext         an Android context, usually <code>this</code>
     * @param apiKey                 your Bugsnag API key from your Bugsnag dashboard
     * @param enableExceptionHandler should we automatically handle uncaught exceptions?
     */
    public Client(@NonNull Context androidContext,
                  @Nullable String apiKey,
                  boolean enableExceptionHandler) {
        this(androidContext,
            ConfigFactory.createNewConfiguration(androidContext, apiKey, enableExceptionHandler));
    }

    /**
     * Initialize a Bugsnag client
     *
     * @param androidContext an Android context, usually <code>this</code>
     * @param configuration  a configuration for the Client
     */
    public Client(@NonNull Context androidContext, @NonNull Configuration configuration) {
        warnIfNotAppContext(androidContext);
        appContext = androidContext.getApplicationContext();
        config = configuration;
        sessionStore = new SessionStore(config, appContext);

        ConnectivityManager cm =
            (ConnectivityManager) appContext.getSystemService(Context.CONNECTIVITY_SERVICE);

        //noinspection ConstantConditions
        if (configuration.getDelivery() == null) {
            configuration.setDelivery(new DefaultDelivery(cm));
        }

        sessionTracker =
            new SessionTracker(configuration, this, sessionStore);
        eventReceiver = new EventReceiver(this);

        // Set up and collect constant app and device diagnostics
        sharedPrefs = appContext.getSharedPreferences(SHARED_PREF_KEY, Context.MODE_PRIVATE);

        appData = new AppData(this);
        deviceData = new DeviceData(this);

        // Set up breadcrumbs
        breadcrumbs = new Breadcrumbs(configuration);

        // Set sensible defaults
        setProjectPackages(appContext.getPackageName());

        String deviceId = getStringFromMap("id", deviceData.getDeviceData());

        if (config.getPersistUserBetweenSessions()) {
            // Check to see if a user was stored in the SharedPreferences
            user.setId(sharedPrefs.getString(USER_ID_KEY, deviceId));
            user.setName(sharedPrefs.getString(USER_NAME_KEY, null));
            user.setEmail(sharedPrefs.getString(USER_EMAIL_KEY, null));
        } else {
            user.setId(deviceId);
        }

        if (appContext instanceof Application) {
            Application application = (Application) appContext;
            application.registerActivityLifecycleCallbacks(sessionTracker);
        } else {
            Logger.warn("Bugsnag is unable to setup automatic activity lifecycle "
                + "breadcrumbs on API Levels below 14.");
        }

        // populate from manifest (in the case where the constructor was called directly by the
        // User or no UUID was supplied)
        if (config.getBuildUUID() == null) {
            String buildUuid = null;
            try {
                PackageManager packageManager = appContext.getPackageManager();
                String packageName = appContext.getPackageName();
                ApplicationInfo ai =
                    packageManager.getApplicationInfo(packageName, PackageManager.GET_META_DATA);
                buildUuid = ai.metaData.getString(MF_BUILD_UUID);
            } catch (Exception ignore) {
                Logger.warn("Bugsnag is unable to read build UUID from manifest.");
            }
            if (buildUuid != null) {
                config.setBuildUUID(buildUuid);
            }
        }

        // Create the error store that is used in the exception handler
        errorStore = new ErrorStore(config, appContext);

        // Install a default exception handler with this client
        if (config.getEnableExceptionHandler()) {
            enableExceptionHandler();
        }

        // register a receiver for automatic breadcrumbs

        try {
            Async.run(new Runnable() {
                @Override
                public void run() {
                    appContext.registerReceiver(eventReceiver, EventReceiver.getIntentFilter());
                    appContext.registerReceiver(new ConnectivityChangeReceiver(),
                        new IntentFilter(ConnectivityManager.CONNECTIVITY_ACTION));
                }
            });
        } catch (RejectedExecutionException ex) {
            Logger.warn("Failed to register for automatic breadcrumb broadcasts", ex);
        }

        boolean isNotProduction = !AppData.RELEASE_STAGE_PRODUCTION.equals(
            appData.guessReleaseStage());
        Logger.setEnabled(isNotProduction);

        config.addObserver(this);
        breadcrumbs.addObserver(this);
        sessionTracker.addObserver(this);
        user.addObserver(this);

        final Client client = this;
        orientationListener = new OrientationEventListener(appContext) {
            @Override
            public void onOrientationChanged(int orientation) {
                client.setChanged();
                client.notifyObservers(new Message(
                    NativeInterface.MessageType.UPDATE_ORIENTATION, orientation));
            }
        };
        orientationListener.enable();

        // Flush any on-disk errors
        errorStore.flushOnLaunch();
    }

    private class ConnectivityChangeReceiver extends BroadcastReceiver {
        @Override
        public void onReceive(Context context, Intent intent) {
            ConnectivityManager cm =
                (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
            if (cm == null) {
                return;
            }

            NetworkInfo networkInfo = cm.getActiveNetworkInfo();
            boolean retryReports = networkInfo != null && networkInfo.isConnectedOrConnecting();

            if (retryReports) {
                errorStore.flushAsync();
            }
        }
    }

    void sendNativeSetupNotification() {
        setChanged();
        super.notifyObservers(new Message(NativeInterface.MessageType.INSTALL, config));
        try {
            Async.run(new Runnable() {
                @Override
                public void run() {
                    enqueuePendingNativeReports();
                }
            });
        } catch (RejectedExecutionException ex) {
            Logger.warn("Failed to enqueue native reports, will retry next launch: ", ex);
        }
    }

    private void enqueuePendingNativeReports() {
        setChanged();
        notifyObservers(new Message(
            NativeInterface.MessageType.DELIVER_PENDING, null));
    }

    @Override
    public void update(Observable observable, Object arg) {
        if (arg instanceof Message) {
            setChanged();
            super.notifyObservers(arg);
        }
    }

    /**
     * Manually starts tracking a new session.
     *
     * Automatic session tracking can be enabled via
     * {@link Configuration#setAutoCaptureSessions(boolean)}, which will automatically create a new
     * session everytime the app enters the foreground.
     */
    public void startSession() {
        sessionTracker.startNewSession(new Date(), user, false);
    }

    /**
     * Starts tracking a new session only if no sessions have yet been tracked
     *
     * This is an integration point for custom libraries implementing automatic session capture
     * which differs from the default activity-based initialization.
     */
    public void startFirstSession(Activity activity) {
        sessionTracker.startFirstSession(activity);
    }

    /**
     * Set the application version sent to Bugsnag. By default we'll pull this
     * from your AndroidManifest.xml
     *
     * @param appVersion the app version to send
     */
    public void setAppVersion(String appVersion) {
        config.setAppVersion(appVersion);
    }

    /**
     * Gets the context to be sent to Bugsnag.
     *
     * @return Context
     */
    public String getContext() {
        return config.getContext();
    }

    /**
     * Set the context sent to Bugsnag. By default we'll attempt to detect the
     * name of the top-most activity at the time of a report, and use this
     * as the context, but sometime this is not possible.
     *
     * @param context set what was happening at the time of a crash
     */
    public void setContext(String context) {
        config.setContext(context);
    }

    /**
     * Set the endpoint to send data to. By default we'll send reports to
     * the standard https://notify.bugsnag.com endpoint, but you can override
     * this if you are using Bugsnag Enterprise to point to your own Bugsnag
     * endpoint.
     *
     * @param endpoint the custom endpoint to send report to
     * @deprecated use {@link com.bugsnag.android.Configuration#setEndpoints(String, String)}
     * instead.
     */
    @Deprecated
    public void setEndpoint(String endpoint) {
        config.setEndpoint(endpoint);
    }

    /**
     * Set the buildUUID to your own value. This is used to identify proguard
     * mapping files in the case that you publish multiple different apps with
     * the same appId and versionCode. The default value is read from the
     * com.bugsnag.android.BUILD_UUID meta-data field in your app manifest.
     *
     * @param buildUuid the buildUuid.
     */
    @SuppressWarnings("checkstyle:AbbreviationAsWordInName")
    public void setBuildUUID(final String buildUuid) {
        config.setBuildUUID(buildUuid);
    }


    /**
     * Set which keys should be filtered when sending metaData to Bugsnag.
     * Use this when you want to ensure sensitive information, such as passwords
     * or credit card information is stripped from metaData you send to Bugsnag.
     * Any keys in metaData which contain these strings will be marked as
     * [FILTERED] when send to Bugsnag.
     * <p/>
     * For example:
     * <p/>
     * client.setFilters("password", "credit_card");
     *
     * @param filters a list of keys to filter from metaData
     */
    public void setFilters(String... filters) {
        config.setFilters(filters);
    }

    /**
     * Set which exception classes should be ignored (not sent) by Bugsnag.
     * <p/>
     * For example:
     * <p/>
     * client.setIgnoreClasses("java.lang.RuntimeException");
     *
     * @param ignoreClasses a list of exception classes to ignore
     */
    public void setIgnoreClasses(String... ignoreClasses) {
        config.setIgnoreClasses(ignoreClasses);
    }

    /**
     * Set for which releaseStages errors should be sent to Bugsnag.
     * Use this to stop errors from development builds being sent.
     * <p/>
     * For example:
     * <p/>
     * client.setNotifyReleaseStages("production");
     *
     * @param notifyReleaseStages a list of releaseStages to notify for
     * @see #setReleaseStage
     */
    public void setNotifyReleaseStages(String... notifyReleaseStages) {
        config.setNotifyReleaseStages(notifyReleaseStages);
    }

    /**
     * Set which packages should be considered part of your application.
     * Bugsnag uses this to help with error grouping, and stacktrace display.
     * <p/>
     * For example:
     * <p/>
     * client.setProjectPackages("com.example.myapp");
     * <p/>
     * By default, we'll mark the current package name as part of you app.
     *
     * @param projectPackages a list of package names
     */
    public void setProjectPackages(String... projectPackages) {
        config.setProjectPackages(projectPackages);
    }

    /**
     * Set the current "release stage" of your application.
     * By default, we'll set this to "development" for debug builds and
     * "production" for non-debug builds.
     *
     * @param releaseStage the release stage of the app
     * @see #setNotifyReleaseStages
     */
    public void setReleaseStage(String releaseStage) {
        config.setReleaseStage(releaseStage);
        Logger.setEnabled(!AppData.RELEASE_STAGE_PRODUCTION.equals(releaseStage));
    }

    /**
     * Set whether to send thread-state with report.
     * By default, this will be true.
     *
     * @param sendThreads should we send thread-state with report?
     */
    public void setSendThreads(boolean sendThreads) {
        config.setSendThreads(sendThreads);
    }


    /**
     * Sets whether or not Bugsnag should automatically capture and report User sessions whenever
     * the app enters the foreground.
     * <p>
     * By default this behavior is disabled.
     *
     * @param autoCapture whether sessions should be captured automatically
     */
    public void setAutoCaptureSessions(boolean autoCapture) {
        config.setAutoCaptureSessions(autoCapture);

        if (autoCapture) { // track any existing sessions
            sessionTracker.onAutoCaptureEnabled();
        }
    }

    /**
     * Set details of the user currently using your application.
     * You can search for this information in your Bugsnag dashboard.
     * <p/>
     * For example:
     * <p/>
     * client.setUser("12345", "james@example.com", "James Smith");
     *
     * @param id    a unique identifier of the current user (defaults to a unique id)
     * @param email the email address of the current user
     * @param name  the name of the current user
     */
    public void setUser(String id, String email, String name) {
        setUserId(id);
        setUserEmail(email);
        setUserName(name);
    }

    /**
     * Retrieves details of the user currently using your application.
     * You can search for this information in your Bugsnag dashboard.
     *
     * @return the current user
     */
    @NonNull
    public User getUser() {
        return user;
    }

    @NonNull
    @InternalApi
    public Collection<Breadcrumb> getBreadcrumbs() {
        return new ArrayList<>(breadcrumbs.store);
    }

    @NonNull
    @InternalApi
    public AppData getAppData() {
        return appData;
    }

    @NonNull
    @InternalApi
    public DeviceData getDeviceData() {
        return deviceData;
    }

    /**
     * Removes the current user data and sets it back to defaults
     */
    public void clearUser() {
        user.setId(getStringFromMap("id", deviceData.getDeviceData()));
        user.setEmail(null);
        user.setName(null);

        SharedPreferences sharedPref =
            appContext.getSharedPreferences(SHARED_PREF_KEY, Context.MODE_PRIVATE);
        sharedPref.edit()
            .remove(USER_ID_KEY)
            .remove(USER_EMAIL_KEY)
            .remove(USER_NAME_KEY)
            .apply();
    }

    /**
     * Set a unique identifier for the user currently using your application.
     * By default, this will be an automatically generated unique id
     * You can search for this information in your Bugsnag dashboard.
     *
     * @param id a unique identifier of the current user
     */
    public void setUserId(String id) {
        user.setId(id);

        if (config.getPersistUserBetweenSessions()) {
            storeInSharedPrefs(USER_ID_KEY, id);
        }
    }

    /**
     * Set the email address of the current user.
     * You can search for this information in your Bugsnag dashboard.
     *
     * @param email the email address of the current user
     */
    public void setUserEmail(String email) {
        user.setEmail(email);

        if (config.getPersistUserBetweenSessions()) {
            storeInSharedPrefs(USER_EMAIL_KEY, email);
        }
    }

    /**
     * Set the name of the current user.
     * You can search for this information in your Bugsnag dashboard.
     *
     * @param name the name of the current user
     */
    public void setUserName(String name) {
        user.setName(name);

        if (config.getPersistUserBetweenSessions()) {
            storeInSharedPrefs(USER_NAME_KEY, name);
        }
    }

    DeliveryCompat getAndSetDeliveryCompat() {
        Delivery current = config.getDelivery();

        if (current instanceof DeliveryCompat) {
            return (DeliveryCompat)current;
        } else {
            DeliveryCompat compat = new DeliveryCompat();
            config.setDelivery(compat);
            return compat;
        }
    }

    @SuppressWarnings("ConstantConditions")
    @Deprecated
    void setErrorReportApiClient(@NonNull ErrorReportApiClient errorReportApiClient) {
        if (errorReportApiClient == null) {
            throw new IllegalArgumentException("ErrorReportApiClient cannot be null.");
        }
        DeliveryCompat compat = getAndSetDeliveryCompat();
        compat.errorReportApiClient = errorReportApiClient;
    }

    @SuppressWarnings("ConstantConditions")
    @Deprecated
    void setSessionTrackingApiClient(@NonNull SessionTrackingApiClient apiClient) {
        if (apiClient == null) {
            throw new IllegalArgumentException("SessionTrackingApiClient cannot be null.");
        }
        DeliveryCompat compat = getAndSetDeliveryCompat();
        compat.sessionTrackingApiClient = apiClient;
    }

    /**
     * Add a "before notify" callback, to execute code before sending
     * reports to Bugsnag.
     * <p/>
     * You can use this to add or modify information attached to an error
     * before it is sent to your dashboard. You can also return
     * <code>false</code> from any callback to prevent delivery. "Before
     * notify" callbacks do not run before reports generated in the event
     * of immediate app termination from crashes in C/C++ code.
     * <p/>
     * For example:
     * <p/>
     * client.beforeNotify(new BeforeNotify() {
     * public boolean run(Error error) {
     * error.setSeverity(Severity.INFO);
     * return true;
     * }
     * })
     *
     * @param beforeNotify a callback to run before sending errors to Bugsnag
     * @see BeforeNotify
     */
    public void beforeNotify(BeforeNotify beforeNotify) {
        config.beforeNotify(beforeNotify);
    }

    /**
     * Add a "before breadcrumb" callback, to execute code before every
     * breadcrumb captured by Bugsnag.
     * <p>
     * You can use this to modify breadcrumbs before they are stored by Bugsnag.
     * You can also return <code>false</code> from any callback to ignore a breadcrumb.
     * <p>
     * For example:
     * <p>
     * Bugsnag.beforeRecordBreadcrumb(new BeforeRecordBreadcrumb() {
     * public boolean shouldRecord(Breadcrumb breadcrumb) {
     * return false; // ignore the breadcrumb
     * }
     * })
     *
     * @param beforeRecordBreadcrumb a callback to run before a breadcrumb is captured
     * @see BeforeRecordBreadcrumb
     */
    public void beforeRecordBreadcrumb(BeforeRecordBreadcrumb beforeRecordBreadcrumb) {
        config.beforeRecordBreadcrumb(beforeRecordBreadcrumb);
    }

    /**
     * Notify Bugsnag of a handled exception
     *
     * @param exception the exception to send to Bugsnag
     */
    public void notify(@NonNull Throwable exception) {
        Error error = new Error.Builder(config, exception, sessionTracker.getCurrentSession(),
            Thread.currentThread(), false)
            .severityReasonType(HandledState.REASON_HANDLED_EXCEPTION)
            .build();
        notify(error, !BLOCKING);
    }

    /**
     * Notify Bugsnag of a handled exception
     *
     * @param exception the exception to send to Bugsnag
     * @param callback  callback invoked on the generated error report for
     *                  additional modification
     */
    public void notify(@NonNull Throwable exception, Callback callback) {
        Error error = new Error.Builder(config, exception, sessionTracker.getCurrentSession(),
            Thread.currentThread(), false)
            .severityReasonType(HandledState.REASON_HANDLED_EXCEPTION)
            .build();
        notify(error, DeliveryStyle.ASYNC, callback);
    }

    /**
     * Notify Bugsnag of an error
     *
     * @param name       the error name or class
     * @param message    the error message
     * @param stacktrace the stackframes associated with the error
     * @param callback   callback invoked on the generated error report for
     *                   additional modification
     */
    public void notify(@NonNull String name,
                       @NonNull String message,
                       @NonNull StackTraceElement[] stacktrace,
                       Callback callback) {
        Error error = new Error.Builder(config, name, message, stacktrace,
            sessionTracker.getCurrentSession(), Thread.currentThread())
            .severityReasonType(HandledState.REASON_HANDLED_EXCEPTION)
            .build();
        notify(error, DeliveryStyle.ASYNC, callback);
    }

    /**
     * Notify Bugsnag of a handled exception
     *
     * @param exception the exception to send to Bugsnag
     * @param severity  the severity of the error, one of Severity.ERROR,
     *                  Severity.WARNING or Severity.INFO
     */
    public void notify(@NonNull Throwable exception, Severity severity) {
        Error error = new Error.Builder(config, exception, sessionTracker.getCurrentSession(),
            Thread.currentThread(), false)
            .severity(severity)
            .build();
        notify(error, !BLOCKING);
    }

    /**
     * Notify Bugsnag of a handled exception
     *
     * @param exception the exception to send to Bugsnag
     * @param metaData  additional information to send with the exception
     * @deprecated Use {@link #notify(Throwable, Callback)} to send and modify error reports
     */
    @Deprecated
    public void notify(@NonNull Throwable exception,
                       @NonNull MetaData metaData) {
        Error error = new Error.Builder(config, exception, sessionTracker.getCurrentSession(),
            Thread.currentThread(), false)
            .metaData(metaData)
            .severityReasonType(HandledState.REASON_HANDLED_EXCEPTION)
            .build();
        notify(error, !BLOCKING);
    }

    /**
     * Notify Bugsnag of a handled exception
     *
     * @param exception the exception to send to Bugsnag
     * @param severity  the severity of the error, one of Severity.ERROR,
     *                  Severity.WARNING or Severity.INFO
     * @param metaData  additional information to send with the exception
     * @deprecated Use {@link #notify(Throwable, Callback)} to send and modify error reports
     */
    @Deprecated
    public void notify(@NonNull Throwable exception, Severity severity,
                       @NonNull MetaData metaData) {
        Error error = new Error.Builder(config, exception, sessionTracker.getCurrentSession(),
            Thread.currentThread(), false)
            .metaData(metaData)
            .severity(severity)
            .build();
        notify(error, !BLOCKING);
    }

    /**
     * Notify Bugsnag of an error
     *
     * @param name       the error name or class
     * @param message    the error message
     * @param stacktrace the stackframes associated with the error
     * @param severity   the severity of the error, one of Severity.ERROR,
     *                   Severity.WARNING or Severity.INFO
     * @param metaData   additional information to send with the exception
     * @deprecated Use {@link #notify(String, String, StackTraceElement[], Callback)}
     * to send and modify error reports
     */
    @Deprecated
    public void notify(@NonNull String name, @NonNull String message,
                       @NonNull StackTraceElement[] stacktrace, Severity severity,
                       @NonNull MetaData metaData) {
        Error error = new Error.Builder(config, name, message,
            stacktrace, sessionTracker.getCurrentSession(), Thread.currentThread())
            .severity(severity)
            .metaData(metaData)
            .build();
        notify(error, !BLOCKING);
    }

    /**
     * Notify Bugsnag of an error
     *
     * @param name       the error name or class
     * @param message    the error message
     * @param context    the error context
     * @param stacktrace the stackframes associated with the error
     * @param severity   the severity of the error, one of Severity.ERROR,
     *                   Severity.WARNING or Severity.INFO
     * @param metaData   additional information to send with the exception
     * @deprecated Use {@link #notify(String, String, StackTraceElement[], Callback)}
     * to send and modify error reports
     */
    @Deprecated
    public void notify(@NonNull String name,
                       @NonNull String message,
                       String context,
                       @NonNull StackTraceElement[] stacktrace,
                       Severity severity,
                       @NonNull MetaData metaData) {
        Error error = new Error.Builder(config, name, message,
            stacktrace, sessionTracker.getCurrentSession(), Thread.currentThread())
            .severity(severity)
            .metaData(metaData)
            .build();
        error.setContext(context);
        notify(error, !BLOCKING);
    }

    private void notify(@NonNull Error error, boolean blocking) {
        DeliveryStyle style = blocking ? DeliveryStyle.SAME_THREAD : DeliveryStyle.ASYNC;
        notify(error, style, null);
    }

    void notify(@NonNull Error error,
                @NonNull DeliveryStyle style,
                @Nullable Callback callback) {
        // Don't notify if this error class should be ignored
        if (error.shouldIgnoreClass()) {
            return;
        }

        // generate new object each time, as this can be mutated by end-users
        Map<String, Object> errorAppData = appData.getAppData();

        // Don't notify unless releaseStage is in notifyReleaseStages
        String releaseStage = getStringFromMap("releaseStage", errorAppData);

        if (!config.shouldNotifyForReleaseStage(releaseStage)) {
            return;
        }

        // Capture the state of the app and device and attach diagnostics to the error
        Map<String, Object> errorDeviceData = deviceData.getDeviceData();
        error.setDeviceData(errorDeviceData);
        error.getMetaData().store.put("device", deviceData.getDeviceMetaData());


        // add additional info that belongs in metadata
        error.setAppData(errorAppData);
        error.getMetaData().store.put("app", appData.getAppDataMetaData());

        // Attach breadcrumbs to the error
        error.setBreadcrumbs(breadcrumbs);

        // Attach user info to the error
        error.setUser(user);

        // Run beforeNotify tasks, don't notify if any return true
        if (!runBeforeNotifyTasks(error)) {
            Logger.info("Skipping notification - beforeNotify task returned false");
            return;
        }

        // Build the report
        Report report = new Report(config.getApiKey(), error);

        if (callback != null) {
            callback.beforeNotify(report);
        }

        HandledState handledState = report.getError().getHandledState();

        if (handledState.isUnhandled()) {
            sessionTracker.incrementUnhandledError();
        } else {
            sessionTracker.incrementHandledError();
            if (sessionTracker.getCurrentSession() != null) {
                setChanged();
                notifyObservers(new NativeInterface.Message(
                            NativeInterface.MessageType.NOTIFY_HANDLED, error.getExceptionName()));
            }
        }

        switch (style) {
            case SAME_THREAD:
                deliver(report, error);
                break;
            case ASYNC:
                final Report finalReport = report;
                final Error finalError = error;

                // Attempt to send the report in the background
                try {
                    Async.run(new Runnable() {
                        @Override
                        public void run() {
                            deliver(finalReport, finalError);
                        }
                    });
                } catch (RejectedExecutionException exception) {
                    errorStore.write(error);
                    Logger.warn("Exceeded max queue count, saving to disk to send later");
                }
                break;
            case ASYNC_WITH_CACHE:
                errorStore.write(error);
                errorStore.flushAsync();
                break;
            default:
                break;
        }

        // Add a breadcrumb for this error occurring
        String exceptionMessage = error.getExceptionMessage();
        Map<String, String> message = Collections.singletonMap("message", exceptionMessage);
        breadcrumbs.add(new Breadcrumb(error.getExceptionName(), BreadcrumbType.ERROR, message));
    }

    /**
     * Notify Bugsnag of a handled exception
     *
     * @param exception the exception to send to Bugsnag
     */
    public void notifyBlocking(@NonNull Throwable exception) {
        Error error = new Error.Builder(config, exception, sessionTracker.getCurrentSession(),
            Thread.currentThread(), false)
            .severityReasonType(HandledState.REASON_HANDLED_EXCEPTION)
            .build();
        notify(error, BLOCKING);
    }

    /**
     * Notify Bugsnag of a handled exception
     *
     * @param exception the exception to send to Bugsnag
     * @param callback  callback invoked on the generated error report for
     *                  additional modification
     */
    public void notifyBlocking(@NonNull Throwable exception, Callback callback) {
        Error error = new Error.Builder(config, exception, sessionTracker.getCurrentSession(),
            Thread.currentThread(), false)
            .severityReasonType(HandledState.REASON_HANDLED_EXCEPTION)
            .build();
        notify(error, DeliveryStyle.SAME_THREAD, callback);
    }

    /**
     * Notify Bugsnag of an error
     *
     * @param name       the error name or class
     * @param message    the error message
     * @param stacktrace the stackframes associated with the error
     * @param callback   callback invoked on the generated error report for
     *                   additional modification
     */
    public void notifyBlocking(@NonNull String name,
                               @NonNull String message,
                               @NonNull StackTraceElement[] stacktrace,
                               Callback callback) {
        Error error = new Error.Builder(config, name, message,
            stacktrace, sessionTracker.getCurrentSession(), Thread.currentThread())
            .severityReasonType(HandledState.REASON_HANDLED_EXCEPTION)
            .build();
        notify(error, DeliveryStyle.SAME_THREAD, callback);
    }

    /**
     * Notify Bugsnag of a handled exception
     *
     * @param exception the exception to send to Bugsnag
     * @param metaData  additional information to send with the exception
     * @deprecated Use {@link #notify(Throwable, Callback)} to send and modify error reports
     */
    @Deprecated
    public void notifyBlocking(@NonNull Throwable exception,
                               @NonNull MetaData metaData) {
        Error error = new Error.Builder(config, exception, sessionTracker.getCurrentSession(),
            Thread.currentThread(), false)
            .severityReasonType(HandledState.REASON_HANDLED_EXCEPTION)
            .metaData(metaData)
            .build();
        notify(error, BLOCKING);
    }

    /**
     * Notify Bugsnag of a handled exception
     *
     * @param exception the exception to send to Bugsnag
     * @param severity  the severity of the error, one of Severity.ERROR,
     *                  Severity.WARNING or Severity.INFO
     * @param metaData  additional information to send with the exception
     * @deprecated Use {@link #notifyBlocking(Throwable, Callback)} to send and modify error reports
     */
    @Deprecated
    public void notifyBlocking(@NonNull Throwable exception, Severity severity,
                               @NonNull MetaData metaData) {
        Error error = new Error.Builder(config, exception, sessionTracker.getCurrentSession(),
            Thread.currentThread(), false)
            .metaData(metaData)
            .severity(severity)
            .build();
        notify(error, BLOCKING);
    }

    /**
     * Notify Bugsnag of an error
     *
     * @param name       the error name or class
     * @param message    the error message
     * @param stacktrace the stackframes associated with the error
     * @param severity   the severity of the error, one of Severity.ERROR,
     *                   Severity.WARNING or Severity.INFO
     * @param metaData   additional information to send with the exception
     * @deprecated Use {@link #notifyBlocking(String, String, StackTraceElement[], Callback)}
     * to send and modify error reports
     */
    @Deprecated
    public void notifyBlocking(@NonNull String name,
                               @NonNull String message,
                               @NonNull StackTraceElement[] stacktrace,
                               Severity severity,
                               @NonNull MetaData metaData) {
        Error error = new Error.Builder(config, name, message,
            stacktrace, sessionTracker.getCurrentSession(), Thread.currentThread())
            .severity(severity)
            .metaData(metaData)
            .build();
        notify(error, BLOCKING);
    }

    /**
     * Notify Bugsnag of an error
     *
     * @param name       the error name or class
     * @param message    the error message
     * @param context    the error context
     * @param stacktrace the stackframes associated with the error
     * @param severity   the severity of the error, one of Severity.ERROR,
     *                   Severity.WARNING or Severity.INFO
     * @param metaData   additional information to send with the exception
     * @deprecated Use {@link #notifyBlocking(String, String, StackTraceElement[], Callback)}
     * to send and modify error reports
     */
    @Deprecated
    public void notifyBlocking(@NonNull String name,
                               @NonNull String message,
                               String context,
                               @NonNull StackTraceElement[] stacktrace,
                               Severity severity,
                               @NonNull MetaData metaData) {
        Error error = new Error.Builder(config, name, message,
            stacktrace, sessionTracker.getCurrentSession(), Thread.currentThread())
            .severity(severity)
            .metaData(metaData)
            .build();
        error.setContext(context);
        notify(error, BLOCKING);
    }

    /**
     * Notify Bugsnag of a handled exception
     *
     * @param exception the exception to send to Bugsnag
     * @param severity  the severity of the error, one of Severity.ERROR,
     *                  Severity.WARNING or Severity.INFO
     */
    public void notifyBlocking(@NonNull Throwable exception, Severity severity) {
        Error error = new Error.Builder(config, exception,
            sessionTracker.getCurrentSession(), Thread.currentThread(), false)
            .severity(severity)
            .build();
        notify(error, BLOCKING);
    }

    /**
     * Intended for internal use only
     *
     * @param exception the exception
     * @param clientData the clientdata
     * @param blocking whether to block when notifying
     * @param callback a callback when notifying
     */
    public void internalClientNotify(@NonNull Throwable exception,
                              Map<String, Object> clientData,
                              boolean blocking,
                              Callback callback) {
        String severity = getKeyFromClientData(clientData, "severity", true);
        String severityReason =
            getKeyFromClientData(clientData, "severityReason", true);
        String logLevel = getKeyFromClientData(clientData, "logLevel", false);

        String msg = String.format("Internal client notify, severity = '%s',"
            + " severityReason = '%s'", severity, severityReason);
        Logger.info(msg);

        @SuppressWarnings("WrongConstant")
        Error error = new Error.Builder(config, exception,
            sessionTracker.getCurrentSession(), Thread.currentThread(), false)
            .severity(Severity.fromString(severity))
            .severityReasonType(severityReason)
            .attributeValue(logLevel)
            .build();

        DeliveryStyle deliveryStyle = blocking ? DeliveryStyle.SAME_THREAD : DeliveryStyle.ASYNC;
        notify(error, deliveryStyle, callback);
    }

    @NonNull
    private String getKeyFromClientData(Map<String, Object> clientData,
                                        String key,
                                        boolean required) {
        Object value = clientData.get(key);
        if (value instanceof String) {
            return (String) value;
        } else if (required) {
            throw new IllegalStateException("Failed to set " + key + " in client data!");
        }
        return null;
    }

    /**
     * Add diagnostic information to every error report.
     * Diagnostic information is collected in "tabs" on your dashboard.
     * <p/>
     * For example:
     * <p/>
     * client.addToTab("account", "name", "Acme Co.");
     * client.addToTab("account", "payingCustomer", true);
     *
     * @param tab   the dashboard tab to add diagnostic data to
     * @param key   the name of the diagnostic information
     * @param value the contents of the diagnostic information
     */
    public void addToTab(String tab, String key, Object value) {
        config.getMetaData().addToTab(tab, key, value);
    }

    /**
     * Remove a tab of app-wide diagnostic information
     *
     * @param tabName the dashboard tab to remove diagnostic data from
     */
    public void clearTab(String tabName) {
        config.getMetaData().clearTab(tabName);
    }

    /**
     * Get the global diagnostic information currently stored in MetaData.
     *
     * @see MetaData
     */
    @NonNull public MetaData getMetaData() {
        return config.getMetaData();
    }

    /**
     * Set the global diagnostic information to be send with every error.
     *
     * @see MetaData
     */
    public void setMetaData(@NonNull MetaData metaData) {
        config.setMetaData(metaData);
    }

    /**
     * Leave a "breadcrumb" log message, representing an action that occurred
     * in your app, to aid with debugging.
     *
     * @param breadcrumb the log message to leave (max 140 chars)
     */
    public void leaveBreadcrumb(@NonNull String breadcrumb) {
        Breadcrumb crumb = new Breadcrumb(breadcrumb);

        if (runBeforeBreadcrumbTasks(crumb)) {
            breadcrumbs.add(crumb);
        }
    }

    /**
     * Leave a "breadcrumb" log message, representing an action which occurred
     * in your app, to aid with debugging.
     */
    public void leaveBreadcrumb(@NonNull String name,
                                @NonNull BreadcrumbType type,
                                @NonNull Map<String, String> metadata) {
        Breadcrumb crumb = new Breadcrumb(name, type, metadata);

        if (runBeforeBreadcrumbTasks(crumb)) {
            breadcrumbs.add(crumb);
        }
    }

    /**
     * Set the maximum number of breadcrumbs to keep and sent to Bugsnag.
     * By default, we'll keep and send the 20 most recent breadcrumb log
     * messages.
     *
     * @param numBreadcrumbs number of breadcrumb log messages to send
     * @deprecated use {@link Configuration#setMaxBreadcrumbs(int)} instead
     */
    @Deprecated
    public void setMaxBreadcrumbs(int numBreadcrumbs) {
        config.setMaxBreadcrumbs(numBreadcrumbs);
    }

    /**
     * Clear any breadcrumbs that have been left so far.
     */
    public void clearBreadcrumbs() {
        breadcrumbs.clear();
    }

    /**
     * Enable automatic reporting of unhandled exceptions.
     * By default, this is automatically enabled in the constructor.
     */
    public void enableExceptionHandler() {
        ExceptionHandler.enable(this);
    }

    /**
     * Disable automatic reporting of unhandled exceptions.
     */
    public void disableExceptionHandler() {
        ExceptionHandler.disable(this);
    }

    void deliver(@NonNull Report report, @NonNull Error error) {
        if (!runBeforeSendTasks(report)) {
            Logger.info("Skipping notification - beforeSend task returned false");
            return;
        }
        try {
            config.getDelivery().deliver(report, config);
            Logger.info("Sent 1 new error to Bugsnag");
        } catch (DeliveryFailureException exception) {
            Logger.warn("Could not send error(s) to Bugsnag,"
                + " saving to disk to send later", exception);
            errorStore.write(error);
        } catch (Exception exception) {
            Logger.warn("Problem sending error to Bugsnag", exception);
        }
    }

    /**
     * Caches an error then attempts to notify.
     *
     * Should only ever be called from the {@link ExceptionHandler}.
     */
    void cacheAndNotify(@NonNull Throwable exception, Severity severity, MetaData metaData,
                        @HandledState.SeverityReason String severityReason,
                        @Nullable String attributeValue, Thread thread) {
        Error error = new Error.Builder(config, exception,
            sessionTracker.getCurrentSession(), thread, true)
            .severity(severity)
            .metaData(metaData)
            .severityReasonType(severityReason)
            .attributeValue(attributeValue)
            .build();

        notify(error, DeliveryStyle.ASYNC_WITH_CACHE, null);
    }

    private boolean runBeforeSendTasks(Report report) {
        for (BeforeSend beforeSend : config.getBeforeSendTasks()) {
            try {
                if (!beforeSend.run(report)) {
                    return false;
                }
            } catch (Throwable ex) {
                Logger.warn("BeforeSend threw an Exception", ex);
            }
        }

        // By default, allow the error to be sent if there were no objections
        return true;
    }

    private boolean runBeforeNotifyTasks(Error error) {
        for (BeforeNotify beforeNotify : config.getBeforeNotifyTasks()) {
            try {
                if (!beforeNotify.run(error)) {
                    return false;
                }
            } catch (Throwable ex) {
                Logger.warn("BeforeNotify threw an Exception", ex);
            }
        }

        // By default, allow the error to be sent if there were no objections
        return true;
    }

    private boolean runBeforeBreadcrumbTasks(@NonNull Breadcrumb breadcrumb) {
        Collection<BeforeRecordBreadcrumb> tasks = config.getBeforeRecordBreadcrumbTasks();
        for (BeforeRecordBreadcrumb beforeRecordBreadcrumb : tasks) {
            try {
                if (!beforeRecordBreadcrumb.shouldRecord(breadcrumb)) {
                    return false;
                }
            } catch (Throwable ex) {
                Logger.warn("BeforeRecordBreadcrumb threw an Exception", ex);
            }
        }
        return true;
    }


    /**
     * Stores the given key value pair into shared preferences
     *
     * @param key   The key to store
     * @param value The value to store
     */
    private void storeInSharedPrefs(String key, String value) {
        SharedPreferences sharedPref =
            appContext.getSharedPreferences(SHARED_PREF_KEY, Context.MODE_PRIVATE);
        sharedPref.edit().putString(key, value).apply();
    }

    ErrorStore getErrorStore() {
        return errorStore;
    }

    /**
     * Finalize by removing the receiver
     *
     * @throws Throwable if something goes wrong
     */
    @SuppressWarnings("checkstyle:NoFinalizer")
    protected void finalize() throws Throwable {
        if (eventReceiver != null) {
            try {
                appContext.unregisterReceiver(eventReceiver);
            } catch (IllegalArgumentException exception) {
                Logger.warn("Receiver not registered");
            }
        }
        super.finalize();
    }

    private static void warnIfNotAppContext(Context androidContext) {
        if (!(androidContext instanceof Application)) {
            Logger.warn("Warning - Non-Application context detected! Please ensure that you are "
                + "initializing Bugsnag from a custom Application class.");
        }
    }

    /**
     * Sets whether the SDK should write logs. In production apps, it is recommended that this
     * should be set to false.
     * <p>
     * Logging is enabled by default unless the release stage is set to 'production', in which case
     * it will be disabled.
     *
     * @param loggingEnabled true if logging is enabled
     */
    public void setLoggingEnabled(boolean loggingEnabled) {
        Logger.setEnabled(loggingEnabled);
    }

    /**
     * Returns the configuration used to initialise the client
     * @return the config
     */
    @NonNull
    public Configuration getConfig() {
        return config;
    }

    /**
     * Retrieves the time at which the client was launched
     *
     * @return the ms since the java epoch
     */
    public long getLaunchTimeMs() {
        return AppData.getDurationMs();
    }

}
