package com.ssqandroidweb.utils;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.net.Uri;
import android.provider.MediaStore;
import android.support.annotation.NonNull;


import java.io.File;
import java.io.FileNotFoundException;
import java.io.InputStream;


/**
 * 从相册或拍照获取一张图片，然后截取
 * <p/>
 * Created by Json Ding on 2015/8/7.
 */
public class CaptureCropUtil {

    public static final String CROP_IMAGE_PATH = "/crop/";
    public static final String CAPTURE_IMAGE_PATH = "/capture/";
    public static final String SIGN_IMAGE_PATH = "/sign/";

    public static final int REQUEST_CODE_CAMERA = 0xa00;  //照相的request_code
    public static final int REQUEST_CODE_IMAGE = 0xa01;   //相册的request_code
    public static final int REQUEST_CODE_RESIZE = 0xa02;  //截取的request_code

    private Activity mActivity;
    private long mOpenCameraTime;
    private long mCropTime;

    public CaptureCropUtil(Activity activity) {
        mActivity = activity;
    }

    public void openCamera() {
        mOpenCameraTime = System.currentTimeMillis();
        Intent cameraIntent = new Intent("android.media.action.IMAGE_CAPTURE");
        cameraIntent.putExtra(MediaStore.EXTRA_OUTPUT, getTempCaptureImageUri());
        cameraIntent.putExtra(MediaStore.EXTRA_VIDEO_QUALITY, 1);
        mActivity.startActivityForResult(cameraIntent, REQUEST_CODE_CAMERA);

    }

    public void openAlbum() {
        Intent intent = new Intent(Intent.ACTION_PICK);
        intent.setType("image/*");//相片类型
        mActivity.startActivityForResult(intent, REQUEST_CODE_IMAGE);
    }

    /**
     * 裁剪图片方法实现
     *
     * @param uri
     */
    public void resizeImage(Uri uri, int aspectX, int aspectY, int outputX, int outputY) {
        mCropTime = System.currentTimeMillis();
        Intent intent = new Intent("com.android.camera.action.CROP");
        intent.setDataAndType(uri, "image/*");
        intent.putExtra("crop", "true");
        intent.putExtra("aspectX", aspectX);
        intent.putExtra("aspectY", aspectY);
        intent.putExtra("outputX", outputX);
        intent.putExtra("outputY", outputY);
        intent.putExtra(MediaStore.EXTRA_OUTPUT, getTempCropImageUri());
        intent.putExtra("return-data", false);
        intent.putExtra("outputFormat", Bitmap.CompressFormat.JPEG.toString());
        intent.putExtra("noFaceDetection", true); //关闭人脸检测
        mActivity.startActivityForResult(intent, REQUEST_CODE_RESIZE);
    }

    public Bitmap decodeUriAsBitmap(Uri uri) {
        Bitmap bitmap = null;
        try {
            final InputStream is = mActivity.getContentResolver().openInputStream(uri);
            bitmap = BitmapFactory.decodeStream(is);
        } catch (FileNotFoundException e) {
            e.printStackTrace();
            return null;
        }
        return bitmap;
    }

    /**
     * 使用前提：用户已登陆
     * 获取照片存放的uri
     *
     * @return
     */
    public Uri getTempCaptureImageUri() {
        return Uri.fromFile(getTempCaptureImageFile());
    }

    public File getTempCaptureImageFile() {
        final File path = new File(getCachePath(mActivity) + CAPTURE_IMAGE_PATH);
        if (!path.exists()) {
            path.mkdirs();
        }
        return new File(path.getAbsolutePath(), String.format("CAPTURE_%s.jpg", mOpenCameraTime));
    }


    public Uri getTempCropImageUri() {
        return Uri.fromFile(getTempCropImageFile());
    }

    public File getTempCropImageFile() {
        final File path = new File(getCachePath(mActivity) + CROP_IMAGE_PATH);
        if (!path.exists()) {
            path.mkdirs();
        }
        return new File(path.getAbsolutePath(), String.format("CROP_%s.jpg", mCropTime)).getAbsoluteFile();
    }

    /**
     * 暂存签名bitmap转为的文件uri
     */
    public Uri getTempSignUri() {
        return Uri.fromFile(getTempSignFile());
    }

    public File getTempSignFile() {
        final File path = new File(getCachePath(mActivity) + SIGN_IMAGE_PATH);
        if (!path.exists()) {
            path.mkdirs();
        }
        return new File(path.getAbsolutePath(), "SIGN_TEMP").getAbsoluteFile();
    }

    public static String getCachePath(@NonNull Context context) {
        File externalCacheDir = context.getExternalCacheDir();
        if (null != externalCacheDir) {
            String path = externalCacheDir.getAbsolutePath() + "/uploadImg/";
            setMkdirs(path);
            return path;
        } else {
            File cacheDir = context.getCacheDir();
            String path = cacheDir.getAbsolutePath() + "/uploadImg/";
            setMkdirs(path);
            return path;
        }
    }

    public static void setMkdirs(String path) {
        File file = new File(path);
        if (!file.exists()) {
            file.mkdirs();
        }
    }
}
