/*
 * Copyright (C) 2018 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.sdk_design.widgets

import android.os.CountDownTimer

/**
 * A HideableWidget is a widget which can be hidden automatically after a preset of millis.
 * It also exposes a function to toggle between states
 *
 * @property hidingTimer define a CountDownTimer to use for the hiding actions
 *
 */
interface HideableWidget {

    var hidingTimer: CountDownTimer?

    /**
     * Triggers onHidingTimerFinishedCallback after input milliseconds.
     * @param afterMillis hidingTimer duration in milliseconds.
     */
    fun autoHide(afterMillis: Long) {
        when {
            hidingTimer != null -> hidingTimer!!.cancel()
            else -> hidingTimer = object : CountDownTimer(afterMillis, 100) {

                override fun onFinish() {
                    onHidingTimerFinished()
                }

                override fun onTick(millisUntilFinished: Long) {}
            }
        }
        hidingTimer!!.start()
    }

    /**
     * Disable auto-hide hidingTimer even if has already started.
     */
    fun disableAutoHide() {
        hidingTimer?.cancel()
    }

    /**
     * Called when hidingTimer has finished.
     */
    fun onHidingTimerFinished()
}