/*
 * Copyright (C) 2020 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.sdk_design.widgets

import android.animation.Animator
import android.graphics.drawable.Drawable
import android.view.View
import android.view.ViewGroup
import com.bandyer.android_common.FieldProperty
import com.bandyer.sdk_design.R
import com.bandyer.sdk_design.extensions.getViewIndex

/**
 * Defines a view group that is capable to fade in from black or fade out from black
 * @param T generic viewgroup type
 * @property fadingView View? the view that will fade in/out
 */
interface FadableViewGroup<T> where T : ViewGroup {

    /**
     * FabableViewGroup instance
     */
    companion object {
        /**
         * Fade in/out animation duration in millis
         */
        const val FADE_ANIMATION_DURATION = 300L
    }

    /**
     * Returns the view group's child id behind which the fade view will be placed
     * or View.NO_ID to let the fading view to be put above all view group's children
     * @return Int
     */
    fun getFadePivotViewId(): Int? = 0

    /**
     * Return the background drawable that will be used for the fading view
     * @return Drawable
     */
    fun getFadeBackground(): Drawable? = null

    /**
     * Fade out the fading view
     * @param endAnimationCallback Function0<Unit>? animation end callback
     */
    fun fadeOut(endAnimationCallback: (() -> Unit)? = null, durationMillis: Long? = FADE_ANIMATION_DURATION) =
            fade(this as ViewGroup, true, endAnimationCallback, durationMillis!!)

    /**
     * Fade in the fading view
     * @param endAnimationCallback Function0<Unit>? animation end callback
     */
    fun fadeIn(endAnimationCallback: (() -> Unit)? = null, durationMillis: Long? = FADE_ANIMATION_DURATION) =
            fade(this as ViewGroup, false, endAnimationCallback, durationMillis!!)

    private fun <T> fade(viewGroup: T, out: Boolean, endAnimationCallback: (() -> Unit)? = null, durationMillis: Long) where T : ViewGroup {
        if (fadingView == null)
            fadingView = View(viewGroup.context).apply {
                this.id = R.id.bandyer_id_fading_view
                this.background = getFadeBackground()
                viewGroup.addView(
                        this,
                        viewGroup.getViewIndex(getFadePivotViewId()),
                        ViewGroup.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT))
            }

        fadingView!!.clearAnimation()

        if (durationMillis == 0L) {
            fadingView!!.alpha = getFadeValue(out, false)
            endAnimationCallback?.invoke()
            return
        } else
            fadingView!!.alpha = getFadeValue(out, true)

        fadingView!!.animate().alpha(getFadeValue(out, false)).setDuration(durationMillis).setListener(object : Animator.AnimatorListener {
            override fun onAnimationRepeat(animation: Animator?) {}
            override fun onAnimationEnd(animation: Animator?) {
                endAnimationCallback?.invoke()
            }

            override fun onAnimationCancel(animation: Animator?) {}
            override fun onAnimationStart(animation: Animator?) {}
        }).start()
    }

    private fun getFadeValue(out: Boolean, startValue: Boolean): Float {
        val isCustomFadingView = fadingView!!.id != R.id.bandyer_id_fading_view

        when {
            out -> {
                return if (startValue) {
                    if (isCustomFadingView) 1f
                    else 0f
                } else {
                    if (isCustomFadingView) 0f
                    else 1f
                }
            }
            else -> {
                return if (startValue) {
                    if (isCustomFadingView) 0f
                    else 1f
                } else {
                    if (isCustomFadingView) 1f
                    else 0f
                }
            }
        }
    }
}

/**
 * Property specifying the fading view
 */
var <T> FadableViewGroup<T>.fadingView: View? where T : android.view.ViewGroup by FieldProperty { null }