/*
 * Copyright (C) 2018 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.sdk_design.whiteboard.layout

import android.content.Context
import android.graphics.Point
import android.util.AttributeSet
import android.view.*
import android.widget.LinearLayout
import android.widget.RelativeLayout
import android.widget.TextView
import com.airbnb.paris.annotations.Styleable
import com.airbnb.paris.annotations.StyleableChild
import com.bandyer.sdk_design.R
import com.bandyer.sdk_design.R2
import com.bandyer.sdk_design.buttons.action.BandyerActionButton
import com.bandyer.sdk_design.edittexts.BandyerEditText
import com.bandyer.sdk_design.extensions.StyleCompat
import com.bandyer.sdk_design.extensions.hideKeyboard
import com.bandyer.sdk_design.extensions.showKeyboard
import com.bandyer.sdk_design.textviews.BandyerTextView
import kotlinx.android.synthetic.main.bandyer_whiteboard_text_editor.view.*


/**
 * @suppress
 * @author kristiyan
 */
@Styleable("BandyerSDKDesign_Whiteboard_TextEditor_Layout")
class BandyerWhiteboardTextEditorLayout @JvmOverloads constructor(context: Context, attrs: AttributeSet? = null, defStyleAttr: Int = 0) : RelativeLayout(context, attrs, defStyleAttr) {

    @StyleableChild(R2.styleable.BandyerSDKDesign_Whiteboard_TextEditor_Layout_bandyer_cancelButtonStyle)
    var cancelInputButton: BandyerActionButton? = null
        private set

    @StyleableChild(R2.styleable.BandyerSDKDesign_Whiteboard_TextEditor_Layout_bandyer_confirmButtonStyle)
    var acceptInputButton: BandyerActionButton? = null
        private set

    @StyleableChild(R2.styleable.BandyerSDKDesign_Whiteboard_TextEditor_Layout_bandyer_editTextStyle)
    var inputTextField: BandyerEditText? = null
        private set

    @StyleableChild(R2.styleable.BandyerSDKDesign_Whiteboard_TextEditor_Layout_bandyer_dataLossMessageStyle)
    var dataLossMessage: BandyerTextView? = null
        private set

    var actionMenu: LinearLayout? = null
        private set

    private var tapTextAreaGestureDetector: GestureDetector? = null

    val text: String
        get() = inputTextField!!.text!!.toString()

    init {
        LayoutInflater.from(context).inflate(R.layout.bandyer_whiteboard_text_editor, this)

        acceptInputButton = bandyerAcceptInputButton
        cancelInputButton = bandyerCancelInputButton
        inputTextField = bandyerWhiteboardEditText
        dataLossMessage = whiteboard_data_loss_textView
        actionMenu = whiteboard_actions

        StyleCompat.styleBuilder(this)
                .add(attrs)
                .add(R.style.BandyerSDKDesign_Whiteboard_TextEditor_Layout)
                .apply()

        tapTextAreaGestureDetector = GestureDetector(context, object : GestureDetector.SimpleOnGestureListener() {
            override fun onSingleTapConfirmed(e: MotionEvent): Boolean {
                inputTextField?.performClick()
                inputTextField?.requestFocus()
                inputTextField?.requestFocusFromTouch()
                bandyerWhiteboardEditText.showKeyboard()
                return true
            }
        })

        textArea.setOnTouchListener { v, event ->
            tapTextAreaGestureDetector?.onTouchEvent(event) == true
        }

        bandyerCancelInputButton.button?.setImageResource(R.drawable.ic_bandyer_close)
        bandyerCancelInputButton.label?.text = context.getString(R.string.bandyer_action_dismiss)
    }

    fun showDiscardChangesMode() {
        (textArea.parent as View).animate().alpha(1.0f).duration = 300
        textArea.visibility = View.GONE
        whiteboard_data_loss_textView.visibility = View.VISIBLE
        whiteboard_data_loss_textView.animate().alpha(1.0f).duration = 300
        bandyerCancelInputButton.label?.text = context!!.getString(R.string.bandyer_action_cancel)
        bandyerCancelInputButton.button?.setImageResource(R.drawable.ic_bandyer_cancel)
        bandyerWhiteboardEditText.hideKeyboard()
    }

    fun getDiscardChangesModeHeight(): Int {
        whiteboard_data_loss_textView ?: return 0
        whiteboard_actions ?: return 0
        bandyer_whiteboard_top_space ?: return 0
        val dataLossHeight = getTextViewHeight(whiteboard_data_loss_textView)
        val lp = whiteboard_actions.layoutParams as MarginLayoutParams
        return dataLossHeight + whiteboard_actions.height + lp.bottomMargin + bandyer_whiteboard_top_space.height
    }

    fun setTextAsChanged(changed: Boolean) {
        if (changed) {
            bandyerCancelInputButton.label?.text = context!!.getString(R.string.bandyer_action_discard_changes)
            bandyerCancelInputButton.button?.setImageResource(R.drawable.ic_bandyer_cancel)
        } else {
            bandyerCancelInputButton.label?.text = context!!.getString(R.string.bandyer_action_dismiss)
            bandyerCancelInputButton.button?.setImageResource(R.drawable.ic_bandyer_close)
        }
    }

    /**
     * Get the TextView height before the TextView will render
     * @param textView the TextView to measure
     * @return the height of the textView
     */
    fun getTextViewHeight(textView: TextView): Int {
        val wm = textView.context.getSystemService(Context.WINDOW_SERVICE) as WindowManager
        val display = wm.defaultDisplay

        val deviceWidth: Int

        val size = Point()
        display.getSize(size)
        deviceWidth = size.x

        val widthMeasureSpec = View.MeasureSpec.makeMeasureSpec(deviceWidth, View.MeasureSpec.AT_MOST)
        val heightMeasureSpec = View.MeasureSpec.makeMeasureSpec(0, View.MeasureSpec.UNSPECIFIED)
        textView.measure(widthMeasureSpec, heightMeasureSpec)
        return textView.measuredHeight
    }


    fun showEditMode(withChanges: Boolean = false) {
        (textArea.parent as View).animate().alpha(1.0f).duration = 300
        textArea.visibility = View.VISIBLE
        whiteboard_data_loss_textView.visibility = View.GONE
        textArea.animate().alpha(1.0f).duration = 300
        setTextAsChanged(withChanges)
    }

    fun alphaAnimateTextArea(alpha: Float) {
        (textArea.parent as View).alpha = alpha
        (textArea.parent as View).requestLayout()
    }

}