/*
 * Copyright (C) 2018 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.sdk_design.whiteboard.dialog

import android.content.DialogInterface
import android.content.pm.ActivityInfo
import android.os.Build
import android.os.Bundle
import android.text.Editable
import android.text.TextWatcher
import android.view.KeyEvent
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.appcompat.app.AppCompatActivity
import com.airbnb.paris.utils.setPaddingLeft
import com.airbnb.paris.utils.setPaddingRight
import com.airbnb.paris.utils.setPaddingTop
import com.bandyer.sdk_design.R
import com.bandyer.sdk_design.bottom_sheet.BandyerBottomSheetDialog
import com.bandyer.sdk_design.dialogs.BandyerDialog
import com.bandyer.sdk_design.extensions.hideKeyboard
import com.bandyer.sdk_design.extensions.scanForFragmentActivity
import com.bandyer.sdk_design.extensions.showKeyboard
import com.bandyer.sdk_design.utils.systemviews.SystemViewLayoutObserver
import com.bandyer.sdk_design.utils.systemviews.SystemViewLayoutOffsetListener
import com.bandyer.sdk_design.whiteboard.dialog.BandyerWhiteboardTextEditorDialog.CustomTextEditorDialog.Companion.TEXT_PLACEHOLDER
import com.bandyer.sdk_design.whiteboard.layout.BandyerWhiteboardTextEditorLayout
import com.google.android.material.bottomsheet.BottomSheetBehavior


/**
 * @suppress
 */
class BandyerWhiteboardTextEditorDialog : BandyerDialog<BandyerWhiteboardTextEditorDialog.CustomTextEditorDialog> {

    override var dialog: CustomTextEditorDialog? = null

    override val id: String = "bandyerWhiteboardTextEditorDialog"

    override fun show(activity: androidx.fragment.app.FragmentActivity) {
        if (dialog == null) dialog = CustomTextEditorDialog()
        dialog!!.show(activity.supportFragmentManager, id)
    }

    fun show(activity: androidx.fragment.app.FragmentActivity, oldText: String? = null, listener: BandyerWhiteboardTextEditorWidgetListener): BandyerWhiteboardTextEditorDialog {
        dialog = CustomTextEditorDialog()
        dialog!!.setListener(listener)
        dialog!!.arguments = Bundle()
        dialog!!.arguments!!.putString(TEXT_PLACEHOLDER, oldText)
        show(activity)
        return this
    }

    /**
     * @suppress
     */
    interface BandyerWhiteboardTextEditorWidgetListener {
        fun onTextEditConfirmed(newText: String)
    }

    class CustomTextEditorDialog : BandyerBottomSheetDialog(), TextWatcher, SystemViewLayoutObserver {

        companion object {
            const val TEXT_PLACEHOLDER = "text_placeholder"
        }

        override val style: Int = R.style.BandyerSDK_DialogStyle

        private var mText: String? = null

        private var callback: BandyerWhiteboardTextEditorWidgetListener? = null
        private var collapsed = false

        private var whiteboardTextEditorLayout: BandyerWhiteboardTextEditorLayout? = null

        private var marginTop = 0

        fun setListener(listener: BandyerWhiteboardTextEditorWidgetListener) {
            callback = listener
        }

        override fun onCreate(savedInstanceState: Bundle?) {
            super.onCreate(savedInstanceState)
            mText = arguments?.getString(TEXT_PLACEHOLDER)
        }

        override fun onCreateView(inflater: LayoutInflater, container: ViewGroup?, savedInstanceState: Bundle?): View? {
            whiteboardTextEditorLayout = BandyerWhiteboardTextEditorLayout(context!!)
            return whiteboardTextEditorLayout
        }

        private fun setupText(text: String?) {
            if (text.isNullOrBlank()) whiteboardTextEditorLayout?.showKeyboard()
            else {
                val editText = whiteboardTextEditorLayout?.inputTextField
                editText?.setText(text)
                editText?.setSelection(text.length)
            }
        }

        override fun onDialogWillShow() {
            setupText(mText)
            whiteboardTextEditorLayout?.acceptInputButton?.setOnClickListener {
                val text = whiteboardTextEditorLayout!!.text
                if (!collapsed && textHasChanged()) callback?.onTextEditConfirmed(text)
                dismiss()
            }

            whiteboardTextEditorLayout?.cancelInputButton?.setOnClickListener {
                when {
                    collapsed -> expand()
                    !textHasChanged() -> dismiss()
                    else -> collapse()
                }
            }
            whiteboardTextEditorLayout?.inputTextField?.addTextChangedListener(this)
        }

        override fun afterTextChanged(s: Editable?) {
            val textHasChanged = textHasChanged()
            whiteboardTextEditorLayout?.setTextAsChanged(textHasChanged)
            setCollapsedHeight()
        }

        private fun setCollapsedHeight() {
            whiteboardTextEditorLayout ?: return
            context ?: return
            behavior ?: return
            val parent = whiteboardTextEditorLayout!!.parent as View
            var peek = whiteboardTextEditorLayout!!.getDiscardChangesModeHeight()
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) peek += marginTop
            if (textHasChanged()) behavior!!.peekHeight = peek
            else behavior!!.peekHeight = 0
        }

        override fun beforeTextChanged(s: CharSequence?, start: Int, count: Int, after: Int) {
        }

        override fun onTextChanged(s: CharSequence?, start: Int, before: Int, count: Int) {

        }

        override fun onSlide(offset: Float) {
            val alpha = if (collapsed) 1f - offset else offset
            whiteboardTextEditorLayout?.alphaAnimateTextArea(alpha)
        }

        override fun onStateChanged(@BottomSheetBehavior.State newState: Int) {

        }

        override fun dismiss() {
            whiteboardTextEditorLayout?.inputTextField?.isCursorVisible = false;
            whiteboardTextEditorLayout?.inputTextField?.removeTextChangedListener(this)
            whiteboardTextEditorLayout?.hideKeyboard(true)
            callback = null
            super.dismiss()
        }

        override fun onDismiss(dialog: DialogInterface) {
            whiteboardTextEditorLayout?.hideKeyboard(true)
            callback = null
            super.onDismiss(dialog)
        }

        override fun onResume() {
            super.onResume()

            activity!!.scanForFragmentActivity()?.let {
                SystemViewLayoutOffsetListener.addObserver(it, this)
            }

            activity!!.requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_PORTRAIT
            dialog?.setOnKeyListener { dialog, keyCode, event ->
                val isBack = keyCode == android.view.KeyEvent.KEYCODE_BACK
                if (isBack && event!!.action == KeyEvent.ACTION_UP && !collapsed) {
                    val newText = whiteboardTextEditorLayout!!.text
                    if (mText != newText) collapse()
                    else dismiss()
                }
                isBack
            }
        }

        override fun onPause() {
            super.onPause()
            activity!!.requestedOrientation = ActivityInfo.SCREEN_ORIENTATION_FULL_SENSOR
            activity!!.scanForFragmentActivity()?.let {
                SystemViewLayoutOffsetListener.removeObserver(it as AppCompatActivity, this)
            }
        }

        override fun onDestroy() {
            super.onDestroy()
            dismiss()
        }

        override fun onTopInsetChanged(pixels: Int) {
            marginTop = pixels
            whiteboardTextEditorLayout ?: return
            if (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP) return
            val parent = (whiteboardTextEditorLayout!!.parent as View)
            parent.setPaddingTop(pixels)
        }

        override fun onBottomInsetChanged(pixels: Int) {
            whiteboardTextEditorLayout ?: return
            if (pixels in 0..200) setCollapsedHeight()
        }

        override fun onRightInsetChanged(pixels: Int) {
            whiteboardTextEditorLayout?.setPaddingRight(pixels)
        }

        override fun onLeftInsetChanged(pixels: Int) {
            whiteboardTextEditorLayout?.setPaddingLeft(pixels)
        }

        override fun onCollapsed() {
            collapsed = true
            if (!textHasChanged()) dismiss()
            else whiteboardTextEditorLayout?.showDiscardChangesMode()
        }

        override fun onExpanded() {
            collapsed = false
            whiteboardTextEditorLayout?.showEditMode(textHasChanged())
        }

        private fun textHasChanged(): Boolean {
            val text = whiteboardTextEditorLayout?.text
            return (mText == null && !text.isNullOrBlank()) || (mText != null && mText != text)
        }
    }
}