/*
 * Copyright (C) 2018 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.sdk_design.utils.systemviews.implementation

import android.annotation.SuppressLint
import android.content.ComponentCallbacks
import android.content.res.Configuration
import android.graphics.Rect
import android.os.Build
import android.view.View
import android.view.ViewGroup
import android.view.Window
import androidx.core.view.ViewCompat
import androidx.core.view.WindowInsetsCompat
import androidx.fragment.app.FragmentActivity
import com.bandyer.android_common.LifecycleEvents
import com.bandyer.android_common.LifecyleBinder
import com.bandyer.sdk_design.extensions.checkIsInMultiWindowMode
import com.bandyer.sdk_design.utils.systemviews.SystemViewLayoutObserver
import java.lang.ref.WeakReference
import kotlin.math.abs


internal class SystemViewControlsAware(val finished: () -> Unit) : SystemViewControlsAwareInstance, SystemViewLayoutObserver, ComponentCallbacks, View.OnLayoutChangeListener {
    private var context: WeakReference<FragmentActivity>? = null
    private var isPortrait = true
    private var hasChangedConfiguration = false

    /**
     * Mapping of observers and requests to keep listening on global layout channges
     */
    private var systemUiObservers = mutableListOf<Pair<SystemViewLayoutObserver, Boolean>>()

    private var window: Window? = null

    @SuppressLint("NewApi")
    fun bind(activity: FragmentActivity): SystemViewControlsAware {
        context = WeakReference(activity)

        isPortrait = activity.resources.configuration.orientation == Configuration.ORIENTATION_PORTRAIT

        LifecyleBinder.bind(activity, object : LifecycleEvents {
            override fun destroy() = dispose()
            override fun create() = Unit
            override fun pause() = Unit
            override fun resume() = resetMargins()
            override fun start() = Unit
            override fun stop() = Unit
        })

        this.window = activity.window

        window!!.decorView.post {
            val context = context?.get() ?: return@post
            context.registerComponentCallbacks(this)
            window?.decorView?.addOnLayoutChangeListener(this)
            resetMargins()
        }

        window!!.decorView.setOnSystemUiVisibilityChangeListener {
            window?.decorView?.post { resetMargins() }
        }

        if (Build.VERSION.SDK_INT >= 21) {
            val layout = (window!!.decorView as ViewGroup).getChildAt(0)
            ViewCompat.setOnApplyWindowInsetsListener(layout) { v, insets ->
                val isInPiP = (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) && context?.get()?.isInPictureInPictureMode == true
                val hasInsetsChanged = insets.hasInsetsChanged(oldInsets)
                val isChangingPiPConfiguration = wasInPiP != isInPiP
                wasInPiP = isInPiP
                oldInsets = insets

                if (isChangingPiPConfiguration || !hasInsetsChanged)
                    return@setOnApplyWindowInsetsListener insets.consumeSystemWindowInsets()

                v.post { resetMargins() }
                return@setOnApplyWindowInsetsListener insets.consumeSystemWindowInsets()
            }
        }

        return this
    }

    private var wasInPiP = false

    private fun WindowInsetsCompat.hasInsetsChanged(other: WindowInsetsCompat?): Boolean {
        return other == null ||
                stableInsetTop != other.stableInsetTop ||
                stableInsetLeft != other.stableInsetLeft ||
                stableInsetRight != other.stableInsetRight ||
                stableInsetBottom != other.stableInsetBottom ||
                systemWindowInsetTop != other.systemWindowInsetTop ||
                systemWindowInsetLeft != other.systemWindowInsetLeft ||
                systemWindowInsetRight != other.systemWindowInsetRight ||
                systemWindowInsetBottom != other.systemWindowInsetBottom
    }

    private var oldInsets: WindowInsetsCompat? = null

    override fun addObserver(observer: SystemViewLayoutObserver, removeOnInsetChanged: Boolean): SystemViewControlsAware {
        val addedObserver = systemUiObservers.firstOrNull { it.first == observer }?.first
        if (addedObserver != null) return this
        systemUiObservers.add(Pair(observer, removeOnInsetChanged))
        resetMargins()
        return this
    }


    override fun removeObserver(observer: SystemViewLayoutObserver): SystemViewControlsAware {
        systemUiObservers = systemUiObservers.filterNot { it.first == observer }.toMutableList()
        return this
    }

    override fun getOffsets() {
        resetMargins()
    }

    override fun onLayoutChange(v: View?, left: Int, top: Int, right: Int, bottom: Int, oldLeft: Int, oldTop: Int, oldRight: Int, oldBottom: Int) {
        if (!hasChangedConfiguration && (oldRight == right || oldLeft == left || oldTop == top || oldBottom == bottom)) return
        hasChangedConfiguration = false
        resetMargins()
    }

    private fun resetMargins() {
        val context = context?.get() ?: return
        window ?: return
        val decorView = window!!.decorView as ViewGroup
        if (decorView.width == 0 || decorView.height == 0) return

        val rect = Rect()
        decorView.getWindowVisibleDisplayFrame(rect)
        val height = decorView.height
        val width = decorView.width

        if (!context.checkIsInMultiWindowMode() && (isPortrait && width > height || !isPortrait && height > width)) {
            decorView.post {
                resetMargins()
            }
            return
        }

        if (height == 0 || width == 0) return


        val bottomMargin = oldInsets?.displayCutout?.safeInsetBottom?.takeIf { it > 0 }
                ?: (height - rect.bottom).takeIf { it >= 0 } ?: 0
        val topMargin = oldInsets?.displayCutout?.safeInsetTop?.takeIf { it > 0 }
                ?: rect.top.takeIf { it >= 0 } ?: 0
        val leftMargin = oldInsets?.displayCutout?.safeInsetLeft?.takeIf { it > 0 }
                ?: rect.left.takeIf { it >= 0 } ?: 0
        val rightMargin = oldInsets?.displayCutout?.safeInsetRight?.takeIf { it > 0 } ?: (width - rect.right).takeIf { it >= 0 } ?: 0

        if (rightMargin >= width || leftMargin >= width || bottomMargin >= height || topMargin >= height) return

        onTopInsetChanged(abs(topMargin))
        onBottomInsetChanged(abs(bottomMargin))
        onLeftInsetChanged(abs(leftMargin))
        onRightInsetChanged(abs(rightMargin))

        stopObserversListeningIfNeeded()
    }

    override fun onConfigurationChanged(newConfig: Configuration?) {
        isPortrait = newConfig?.orientation == Configuration.ORIENTATION_PORTRAIT
        hasChangedConfiguration = true
        getOffsets()
    }

    override fun onTopInsetChanged(pixels: Int) {
        systemUiObservers.forEach { it.first.onTopInsetChanged(pixels) }
    }

    override fun onBottomInsetChanged(pixels: Int) {
        systemUiObservers.forEach { it.first.onBottomInsetChanged(pixels) }
    }

    override fun onLeftInsetChanged(pixels: Int) {
        systemUiObservers.forEach { it.first.onLeftInsetChanged(pixels) }
    }

    override fun onRightInsetChanged(pixels: Int) {
        systemUiObservers.forEach { it.first.onRightInsetChanged(pixels) }
    }

    private fun stopObserversListeningIfNeeded() {
        systemUiObservers = systemUiObservers.filter { !it.second }.toMutableList()
    }

    override fun onLowMemory() = Unit

    private fun dispose() {
        systemUiObservers.clear()
        val context = context?.get() ?: return
        window?.decorView?.removeOnLayoutChangeListener(this)
        context.unregisterComponentCallbacks(this)
        this@SystemViewControlsAware.finished()
    }
}