package com.bandyer.sdk_design.utils

import android.animation.Animator
import android.os.CountDownTimer
import android.view.View

/**
 * Represents a view that can be hidden with an animation.
 */
interface ToggleableVisibilityInterface {

    /**
     * Toggle widget visibility
     * @param show true to show the widget, false otherwise
     * @param animationEndCallback callback that will be invoked when showing or hiding animation ends
     */
    fun toggleVisibility(show: Boolean, animationEndCallback: (shown: Boolean) -> Unit)

    /**
     * Cancels timer that hides automatically the widget
     */
    fun cancelTimer()
}

/**
 * Helper class that toggles a view's visibility animating with alpha.
 * @property view View the view to be toggled
 * @constructor
 */
class VisibilityToggle(val view: View): ToggleableVisibilityInterface {

    /**
     * Hide timer constants
     */
    companion object {
        /**
         * Autohide animation millis
         */
        const val AUTOHIDE_MS = 5300L
        /**
         * Show animnation millis
         */
        const val SHOW_MS = 300L
    }

    private var countDownTimer: CountDownTimer? = null

    override fun toggleVisibility(show: Boolean, animationEndCallback: (shown: Boolean) -> Unit) {
        val endValue = if (show) 1f else 0f
        if (view.alpha == endValue) return

        val animationListener = object : Animator.AnimatorListener {
            override fun onAnimationRepeat(p0: Animator?) = Unit
            override fun onAnimationStart(animation: Animator?) = Unit
            override fun onAnimationCancel(animation: Animator?) = cancelTimer()
            override fun onAnimationEnd(p0: Animator?) = animationEndCallback.invoke(view.alpha == 1f)
        }

        view.animate()
                .alpha(endValue)
                .setDuration(SHOW_MS)
                .setListener(animationListener)
                .start()

        cancelTimer()

        countDownTimer = object : CountDownTimer(AUTOHIDE_MS, AUTOHIDE_MS) {
            override fun onFinish() {
                if (view.alpha != 1f) return
                view.animate().alpha(0f)
                        .setDuration(SHOW_MS)
                        .setListener(animationListener)
                        .start()
            }

            override fun onTick(millisUntilFinished: Long) = Unit
        }
        countDownTimer!!.start()
    }
    /**
     * Cancels timer that hides automatically the widget
     */
    override fun cancelTimer() {
        countDownTimer?.cancel()
        countDownTimer = null
    }
}