/*
 * Copyright (C) 2018 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.sdk_design.textviews

import android.content.Context
import android.graphics.PorterDuff
import android.graphics.PorterDuffColorFilter
import android.graphics.Rect
import android.graphics.Typeface.BOLD
import android.graphics.Typeface.NORMAL
import android.graphics.drawable.Drawable
import android.util.AttributeSet
import androidx.annotation.AnyRes
import androidx.annotation.ColorInt
import androidx.annotation.ColorRes
import androidx.appcompat.widget.AppCompatTextView
import androidx.core.content.ContextCompat
import com.airbnb.paris.annotations.Attr
import com.airbnb.paris.annotations.LayoutDimension
import com.airbnb.paris.annotations.Style
import com.airbnb.paris.annotations.Styleable
import com.bandyer.sdk_design.R
import com.bandyer.sdk_design.R2
import com.bandyer.sdk_design.extensions.StyleCompat
import com.bandyer.sdk_design.extensions.getFont

/**
 * Bandyer Design SDK base custom text.
 * Style can be customized with the following styles:
 * 'BandyerSDKDesign.TextView.Title'
 * and
 * 'BandyerSDKDesign.TextView.SubTitle'.
 */
@Styleable("BandyerSDKDesign_TextView")
open class BandyerTextView @JvmOverloads constructor(context: Context, attrs: AttributeSet? = null, defStyleAttr: Int = 0) : AppCompatTextView(context, attrs, defStyleAttr) {

    /**
     * BandyerTextView instance
     */
    companion object {

        /**
         * Title style
         */
        @Style
        val TITLE = R.style.BandyerSDKDesign_TextView_Title

        /**
         * Subtitle style
         */
        @Style
        val SUBTITLE = R.style.BandyerSDKDesign_TextView_Subtitle

    }

    private var mShadowDx: Float? = null
    private var mShadowDy: Float? = null
    private var mShadowRadius: Float? = null
    private var mShadowColor: Int? = null

    private var mCompoundDrawableHeight: Int? = null
    private var mCompoundDrawableWidth: Int? = null
    private var mCompoundDrawableColorTint: Int? = null

    private var fontFamily: Int = 0

    init {
        this.builderStyle().add(attrs).apply()
    }

    /**
     * Method to customize style before it's applied
     * @return BandyerTextViewStyleApplier.StyleBuilder
     */
    open fun builderStyle(): BandyerTextViewStyleApplier.StyleBuilder {
        return StyleCompat.style(this).builder()
    }

    /**
     * Set font typeface given a font resource
     * @param typeface Font resource file
     */
    @Attr(R2.styleable.BandyerSDKDesign_TextView_bandyer_font_name)
    fun setFontName(@AnyRes typeface: Int) {
        this.fontFamily = typeface
        this.typeface = getFont(typeface, NORMAL)
    }

    /**
     * Sets compound drawable height
     * @param height Int desired height
     */
    @Attr(R2.styleable.BandyerSDKDesign_TextView_bandyer_compoundDrawableTint)
    fun setCompoundDrawableTint(@ColorRes color: Int) {
        this.mCompoundDrawableColorTint = color
    }

    /**
     * Sets compound drawable tint
     * @param height Int desired height
     */
    @Attr(R2.styleable.BandyerSDKDesign_TextView_bandyer_compoundDrawableHeight)
    fun setCompoundDrawableHeight(@LayoutDimension height: Int) {
        this.mCompoundDrawableHeight = height
    }

    /**
     * Sets compound drawable width
     * @param height Int desired width
     */
    @Attr(R2.styleable.BandyerSDKDesign_TextView_bandyer_compoundDrawableWidth)
    fun setCompoundDrawableWidth(@LayoutDimension width: Int) {
        this.mCompoundDrawableWidth = width
    }

    private fun initCompoundDrawableSize() {
        if (mCompoundDrawableHeight == null && mCompoundDrawableWidth == null) return
        val drawables = compoundDrawables
        for (drawable in drawables) {
            if (drawable == null) {
                continue
            }
            val realBounds: Rect = drawable.bounds
            val scaleFactor: Float = realBounds.height() / realBounds.width().toFloat()
            var drawableWidth: Float = realBounds.width().toFloat()
            var drawableHeight: Float = realBounds.height().toFloat()
            if (mCompoundDrawableWidth!! > 0) {
                if (drawableWidth > mCompoundDrawableWidth!!) {
                    drawableWidth = mCompoundDrawableWidth!!.toFloat()
                    drawableHeight = drawableWidth * scaleFactor
                }
            }
            if (mCompoundDrawableHeight!! > 0) { // save scale factor of image
                if (drawableHeight > mCompoundDrawableWidth!!) {
                    drawableHeight = mCompoundDrawableWidth!!.toFloat()
                    drawableWidth = drawableHeight / scaleFactor
                }
            }
            realBounds.right = realBounds.left + Math.round(drawableWidth)
            realBounds.bottom = realBounds.top + Math.round(drawableHeight)
            drawable.bounds = realBounds
        }
        setCompoundDrawables(drawables[0], drawables[1], drawables[2], drawables[3])
    }

    private fun setDrawableColorTint() {
        mCompoundDrawableColorTint ?: return
        compoundDrawables.filterNotNull().forEach {
            it.mutate().colorFilter = PorterDuffColorFilter(ContextCompat.getColor(context, mCompoundDrawableColorTint!!), PorterDuff.Mode.SRC_IN)
        }
    }

    /**
     * Select the textView
     * When selected the font will change to Bold if a custom font was set
     * @param selected Boolean true to select, false otherwise
     */
    override fun setSelected(selected: Boolean) {
        super.setSelected(selected)

        post {
            compoundDrawables.forEach { icon: Drawable? ->
                icon?.mutate()?.colorFilter = PorterDuffColorFilter(currentTextColor, PorterDuff.Mode.MULTIPLY)
            }
        }

        typeface = if (selected && fontFamily != 0) {
            getFont(fontFamily, BOLD)
        } else
            getFont(fontFamily, NORMAL)
    }

    /**
     * When the view is attached to window, optional shadow is computed and applied.
     */
    override fun onAttachedToWindow() {
        super.onAttachedToWindow()

        if (mShadowColor != null)
            setShadowLayer(mShadowRadius ?: 0f, mShadowDx ?: 0f, mShadowDy ?: 0f, mShadowColor ?: 0)

        initCompoundDrawableSize()
        setDrawableColorTint()
    }

    /**
     * Set shadow dx
     * @param dimension pixels of shadow in x direction
     */
    @Attr(R2.styleable.BandyerSDKDesign_TextView_bandyer_shadowDx)
    fun setShadowDx(@LayoutDimension dimension: Int) {
        mShadowDx = dimension.toFloat()
    }

    /**
     * Set color of links to highlight
     * @param color link color
     */
    @Attr(R2.styleable.BandyerSDKDesign_TextView_bandyer_colorLink)
    fun setLinkColor(@ColorInt color: Int) {
        setLinkTextColor(color)
    }

    /**
     * Set shadow dy
     * @param dimension pixels of shadow in y direction
     */
    @Attr(R2.styleable.BandyerSDKDesign_TextView_bandyer_shadowDy)
    fun setShadowDy(@LayoutDimension dimension: Int) {
        mShadowDy = dimension.toFloat()
    }

    /**
     * Set shadow radius
     * @param dimension pixels of shadow radius
     */
    @Attr(R2.styleable.BandyerSDKDesign_TextView_bandyer_shadowRadius)
    fun setShadowRadius(@LayoutDimension dimension: Int) {
        mShadowRadius = dimension.toFloat()
    }

    /**
     * Set shadow color
     * @param color Color int to apply on shadow
     */
    @Attr(R2.styleable.BandyerSDKDesign_TextView_shadowColor)
    fun setShadowColor(@ColorInt color: Int) {
        mShadowColor = color
    }

    /**
     * Set fading edge type (horizontal, vertical or none)
     * @param value FadingEdge of fading edge to apply
     */
    @Attr(R2.styleable.BandyerSDKDesign_TextView_bandyer_fadingEdge)
    fun setFadingEdgeValue(value: Int) {
        val edge = FadingEdge.values().firstOrNull { it.value == value } ?: FadingEdge.NONE
        setFadingEdgeValue(edge)
    }

    /**
     * Set fading edge type (horizontal, vertical or none)
     * @param edge FadingEdge of fading edge to apply
     */
    fun setFadingEdgeValue(edge: FadingEdge) {
        when (edge) {
            FadingEdge.NONE -> {
                isHorizontalFadingEdgeEnabled = false
                isVerticalFadingEdgeEnabled = false
            }
            FadingEdge.HORIZONTAL -> isHorizontalFadingEdgeEnabled = true
            FadingEdge.VERTICAL -> isVerticalFadingEdgeEnabled = true
        }
    }

    /**
     * Set the size of the faded edge used to indicate that more content in this
     * view is available.  Will not change whether the fading edge is enabled; use
     * setFadingEdgeValue for the vertical or horizontal fading edges.
     *
     * @param dimension The size in pixels of the faded edge used to indicate that more
     *        content in this view is visible.
     */
    @Attr(R2.styleable.BandyerSDKDesign_TextView_bandyer_fadingEdgeLength)
    fun setFadingEdgeLengthValue(@LayoutDimension dimension: Int) {
        setFadingEdgeLength(dimension)
    }

    /**
     * Sets how many times to repeat the marquee animation. Only applied if the
     * TextView has marquee enabled. Set to -1 to repeat indefinitely.
     *
     * @see #getMarqueeRepeatLimit()
     * @param value limit
     */
    @Attr(R2.styleable.BandyerSDKDesign_TextView_bandyer_marqueeRepeatLimit)
    fun setMarqueeRepeatLimitValue(value: Int) {
        marqueeRepeatLimit = value
    }

    /**
     * Enum representing the Fading edge types
     * @property value by default is NONE
     * @constructor
     */
    enum class FadingEdge(val value: Int = 0x00000000) {
        /**
         * No fading edge
         */
        NONE(0x00000000),

        /**
         * Horizontal fading edge
         */
        HORIZONTAL(0x00001000),

        /**
         * Vertical fading edge
         */
        VERTICAL(0x00002000)
    }
}