/*
 * Copyright (C) 2018 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.sdk_design.layout

import android.content.Context
import android.os.Build
import androidx.appcompat.app.AppCompatActivity
import android.util.AttributeSet
import android.widget.FrameLayout
import android.widget.RelativeLayout
import com.airbnb.paris.utils.*
import com.bandyer.sdk_design.extensions.scanForFragmentActivity
import com.bandyer.sdk_design.utils.systemviews.SystemViewLayoutObserver
import com.bandyer.sdk_design.utils.systemviews.SystemViewLayoutOffsetListener

/**
 * A RelativeLayout which follows is aware of the system views dimensions and positions
 *
 * @constructor
 */
open class SystemControlsAwareRelativeLayout @JvmOverloads constructor(
        context: Context, attrs: AttributeSet? = null, defStyleAttr: Int = 0
) : RelativeLayout(context, attrs, defStyleAttr), SystemViewLayoutObserver {


    ///////////////////////////////////////// SYSTEM CONTROLS AWARE OBSERVER //////////////////////////////////////////////////////////////////////////////////

    /**
     * @suppress
     */
    override fun onAttachedToWindow() {
        super.onAttachedToWindow()
        context.scanForFragmentActivity()?.let {
            SystemViewLayoutOffsetListener.addObserver(it, this)
        }
    }

    /**
     * @suppress
     */
    override fun onDetachedFromWindow() {
        super.onDetachedFromWindow()
        context.scanForFragmentActivity()?.let {
            SystemViewLayoutOffsetListener.removeObserver(it as AppCompatActivity, this)
        }
    }

    /**
     * @suppress
     */
    override fun onTopInsetChanged(pixels: Int) {
        post {
            setPaddingTop(pixels)
        }
    }

    /**
     * @suppress
     */
    override fun onBottomInsetChanged(pixels: Int) {
        post {
            setPaddingBottom(pixels)
        }
    }

    /**
     * @suppress
     */
    override fun onRightInsetChanged(pixels: Int) {
        post {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1)
                setPaddingEnd(pixels)
            else
                setPaddingRight(pixels)
        }
    }

    /**
     * @suppress
     */
    override fun onLeftInsetChanged(pixels: Int) {
        post {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1)
                setPaddingStart(pixels)
            else
                setPaddingLeft(pixels)
        }
    }
}