/*
 * Copyright (C) 2018 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.sdk_design.layout

import android.content.Context
import android.os.Build
import androidx.appcompat.app.AppCompatActivity
import android.util.AttributeSet
import android.widget.FrameLayout
import com.airbnb.paris.utils.setPaddingEnd
import com.airbnb.paris.utils.setPaddingLeft
import com.airbnb.paris.utils.setPaddingRight
import com.airbnb.paris.utils.setPaddingStart
import com.bandyer.sdk_design.extensions.scanForFragmentActivity
import com.bandyer.sdk_design.utils.systemviews.SystemViewLayoutObserver
import com.bandyer.sdk_design.utils.systemviews.SystemViewLayoutOffsetListener

/**
 * A FrameLayout which follows is aware of the system views dimensions and positions
 *
 * @constructor
 */
open class SystemControlsAwareFrameLayout @JvmOverloads constructor(
        context: Context, attrs: AttributeSet? = null, defStyleAttr: Int = 0
) : FrameLayout(context, attrs, defStyleAttr), SystemViewLayoutObserver {


    ///////////////////////////////////////// SYSTEM CONTROLS AWARE OBSERVER //////////////////////////////////////////////////////////////////////////////////

    /**
     * @suppress
     */
    override fun onAttachedToWindow() {
        super.onAttachedToWindow()
        context.scanForFragmentActivity()?.let {
            SystemViewLayoutOffsetListener.addObserver(it, this)
        }
    }

    /**
     * @suppress
     */
    override fun onDetachedFromWindow() {
        super.onDetachedFromWindow()
        context.scanForFragmentActivity()?.let {
            SystemViewLayoutOffsetListener.removeObserver(it as AppCompatActivity, this)
        }
    }

    /**
     * @suppress
     */
    override fun onTopInsetChanged(pixels: Int) {}

    /**
     * @suppress
     */
    override fun onBottomInsetChanged(pixels: Int) {}

    /**
     * @suppress
     */
    override fun onRightInsetChanged(pixels: Int) {
        post {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1)
                setPaddingEnd(pixels)
            else
                setPaddingRight(pixels)
        }
    }

    /**
     * @suppress
     */
    override fun onLeftInsetChanged(pixels: Int) {
        post {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1)
                setPaddingStart(pixels)
            else
                setPaddingLeft(pixels)
        }
    }
}