/*
 * Copyright (C) 2018 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.sdk_design.imageviews

import android.content.Context
import android.graphics.Canvas
import android.graphics.drawable.Drawable
import android.net.Uri
import android.util.AttributeSet
import android.view.View
import androidx.annotation.DrawableRes
import androidx.appcompat.widget.AppCompatImageView
import com.bandyer.android_common.Roundable
import com.bandyer.android_common.RoundedView
import com.bandyer.sdk_design.R
import com.squareup.picasso.Picasso

/**
 * Bandyer Rounded Custom Image View
 * @constructor
 */
open class BandyerRoundedImageView @JvmOverloads constructor(
        context: Context, attrs: AttributeSet? = null, defStyleAttr: Int = 0
) : AppCompatImageView(context, attrs, defStyleAttr), RoundedView {

    private var mRounded = false
    private var currentUrl: String? = null
    private var currentUri: Uri? = null
    private var currentDrawableRes: Int? = null

    init {
        if (attrs != null) {
            val typedArray = context.theme.obtainStyledAttributes(attrs, R.styleable.RoundableStyle, defStyleAttr, 0)
            try {
                if (typedArray.getBoolean(R.styleable.RoundableStyle_bandyer_round, true)) {
                    mRounded = true
                } else typedArray.getDimensionPixelSize(R.styleable.RoundableStyle_bandyer_corner_radius, 0).takeIf { it > 0 }?.let {
                    setCornerRadius<BandyerRoundedImageView>(it.toFloat())
                }

            } finally {
                typedArray.recycle()
            }
        }
    }

    /**
     * @suppress
     */
    override fun onDraw(canvas: Canvas?) {
        setRoundClip(canvas)
        super.onDraw(canvas)
    }

    /**
     * Set BandyerRoundedImageView visibility.
     * @param visibility Int visibility
     */
    override fun setVisibility(visibility: Int) {
        val oldAlpha = alpha
        if (visibility == View.VISIBLE) alpha = 0f
        super.setVisibility(visibility)

        if (rounded<BandyerRoundedImageView>()) round<BandyerRoundedImageView>(true)
        if (radius<BandyerRoundedImageView>() > 0f) setCornerRadius<BandyerRoundedImageView>(radius<BandyerRoundedImageView>())
        alpha = oldAlpha
        requestLayout()
    }

    /**
     * Display image given the url
     * @param url image
     */
    fun setImageUrl(url: String) {
        if (currentUrl == url) return
        this.currentUrl = url
        Picasso.get().load(url).also {
            if (mRounded) it.transform(PicassoCircularTransform())
        }.noFade().into(this)
    }

    /**
     * Display image given a resource
     * @param resId image
     */
    fun setImageResId(@DrawableRes resId: Int) {
        if (currentDrawableRes == resId) return
        this.currentDrawableRes = resId
        if (resId == 0) return
        Picasso.get().load(resId).also {
            if (mRounded) it.transform(PicassoCircularTransform())
        }.noFade().into(this)
    }

    /**
     * Display image given a uri
     * @param uri image
     */
    fun setImageUri(uri: Uri) {
        if (this.currentUri == uri) return
        this.currentUri = uri
        Picasso.get().load(uri).also {
            if (mRounded) it.transform(PicassoCircularTransform())
        }.noFade().into(this)
    }
}