/*
 * Copyright (C) 2018 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.sdk_design.extensions

import android.app.Activity
import android.app.AppOpsManager
import android.content.Context
import android.content.ContextWrapper
import android.content.pm.PackageManager
import android.content.res.Configuration
import android.graphics.Point
import android.graphics.drawable.Drawable
import android.os.Build
import android.provider.Settings
import android.util.DisplayMetrics
import android.view.WindowManager
import androidx.appcompat.app.AppCompatActivity
import androidx.core.content.ContextCompat
import java.util.*

private val dipsMap = HashMap<Float, Int>()
private val pixelsMap = HashMap<Float, Int>()

/**
 * Extension function used to retrieve an attribute resource from context.
 * @param attrRes attribute resource
 * @return drawable
 */
internal fun Context.getDrawableFromAttrRes(attrRes: Int): Drawable? {
    val a = obtainStyledAttributes(intArrayOf(attrRes))
    val drawable: Drawable?
    try {
        drawable = ContextCompat.getDrawable(this, a.getResourceId(0, 0))
    } finally {
        a.recycle()
    }
    return drawable
}

/**
 * Check if device is a tablet
 */
fun Context.isTablet(): Boolean {
    return resources.configuration.screenLayout and Configuration.SCREENLAYOUT_SIZE_MASK >= Configuration.SCREENLAYOUT_SIZE_LARGE
}

/**
 * Convert dp value in pixels
 * @param dp value
 * @return value in pixels
 */
fun Context.dp2px(dp: Float): Int {
    dipsMap[dp]?.let { return it }

    val metrics = resources.displayMetrics
    val value = (dp * (metrics.densityDpi.toFloat() / DisplayMetrics.DENSITY_DEFAULT)).toInt()
    dipsMap[dp] = value

    return value
}

/**
 * Convert px value in dp
 * @param px value
 * @return value in dps
 */
fun Context.px2dp(px: Float): Int {
    pixelsMap[px]?.let { return it }

    val metrics = resources.displayMetrics
    val value = (px / (metrics.densityDpi.toFloat() / DisplayMetrics.DENSITY_DEFAULT)).toInt()
    pixelsMap[px] = value

    return value
}

/**
 * Calculates screen's ratio
 * @return ratio
 */
fun Context.getScreenRatio(): Float {
    val metrics = resources.displayMetrics
    return metrics.heightPixels.toFloat() / metrics.widthPixels.toFloat()
}

/**
 * Check if can draw over apps
 * @receiver Context
 * @return Boolean true if can draw overlays, false otherwise
 */
fun Context.canDrawOverlays(): Boolean = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) Settings.canDrawOverlays(this.applicationContext) else true

/**
 * Checks if this context can be or is in picture in picture mode
 * @receiver Context
 * @return Boolean true if currently in picture in picture mode, false otherwise
 */
fun Activity.isInPictureInPictureModeCompat(): Boolean {
    return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
        val canGoInPip = Build.VERSION.SDK_INT >= Build.VERSION_CODES.O && packageManager.hasSystemFeature(PackageManager.FEATURE_PICTURE_IN_PICTURE)
        canGoInPip && (this as? AppCompatActivity)?.isInPictureInPictureMode == true
    } else false
}

/**
 * Interrupts watching permission change with app ops manager
 * @receiver Context
 * @param callback Function2<String, String, Unit> the callback to be stopped.
 */
fun Context.stopAppOpsWatch(callback: ((String, String) -> Unit)) {
    if (Build.VERSION.SDK_INT < Build.VERSION_CODES.M) return
    val appOpsManager = this.getSystemService(Context.APP_OPS_SERVICE) as AppOpsManager
    appOpsManager.stopWatchingMode(callback)
}

/**
 * Starts watch permission change with app ops manager
 * @receiver Context
 * @param operation String
 * @param callback Function2<String, String, Unit> the callback to be called.
 */
fun Context.startAppOpsWatch(operation:String, callback: ((String, String) -> Unit)){
    if (Build.VERSION.SDK_INT < Build.VERSION_CODES.M) return
    val pckName = applicationContext.packageName
    val appOpsManager = this.getSystemService(Context.APP_OPS_SERVICE) as AppOpsManager
    appOpsManager.startWatchingMode(operation, pckName, callback)
}

/**
 * Calculates screen's size
 * @receiver Context
 * @return Point
 */
fun Context.getScreenSize(): Point {
    val wm = getSystemService(Context.WINDOW_SERVICE) as WindowManager
    val display = wm.defaultDisplay
    val size = Point()
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
        display.getRealSize(size)
    } else {
        display.getSize(size)
    }
    return size
}

/**
 * Finds the fragment activity associated to the context if any.
 * @param cont Context?
 * @return Activity?
 */
fun Context.scanForFragmentActivity(): androidx.fragment.app.FragmentActivity? {
    return when (this) {
        null -> null
        is AppCompatActivity -> this
        is androidx.fragment.app.FragmentActivity -> this
        is ContextWrapper -> this.baseContext.scanForFragmentActivity()
        else -> null
    }
}