/*
 * Copyright (C) 2018 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.sdk_design.extensions

import android.app.Activity
import android.content.Context
import android.graphics.PointF
import android.graphics.Rect
import android.util.Log
import android.view.*
import android.view.inputmethod.InputMethodManager
import androidx.appcompat.app.AppCompatActivity

/**
 * Method to check if device is in Landscape or not
 */
fun Activity.isLandscape(): Boolean {
    val rotation = (getSystemService(Context.WINDOW_SERVICE) as WindowManager).defaultDisplay.rotation
    return when (rotation) {
        Surface.ROTATION_0 -> false
        Surface.ROTATION_180 -> false
        else -> true
    }
}

/**
 * Hide keyboard
 * @receiver Activity
 * @param forced true if should be hidden no matter what, false otherwise
 */
fun Activity.hideKeyboard(forced: Boolean = false) {
    val inputManager = getSystemService(Context.INPUT_METHOD_SERVICE) as InputMethodManager
    val hideMethod = if (forced) 0 else InputMethodManager.HIDE_NOT_ALWAYS
    currentFocus?.let {
        inputManager.hideSoftInputFromWindow(it.windowToken, hideMethod)
    }
}

/**
 * Returns screen center coordinates as Pair<Float, Float>
 * @receiver Activity
 * @return Pair<Float, Float>
 */
fun Activity.getScreenCenter(): PointF {
    return PointF(getScreenSize().x / 2f, getScreenSize().y / 2f)
}

/**
 * Returns status bar height in pixels
 * @return value in pixels
 */
fun Activity.getStatusBarHeight(): Int {
    val rect = Rect()
    window.decorView.getWindowVisibleDisplayFrame(rect)
    val statusBarHeight = rect.top
    return if (statusBarHeight != getScreenSize().y) statusBarHeight else 0
}

/**
 * Returns navigation bar height in pixels
 * @return value in pixels
 */
fun Activity.getNavigationBarHeight(): Int {
    val rect = Rect()
    window.decorView.getWindowVisibleDisplayFrame(rect)
    return window.decorView.height - rect.bottom
}

/**
 * Finds next focus navigating with DPAD controller
 * @receiver Activity
 * @param keyCode Int direction
 * @param rootView ViewGroup root view
 * @return (android.view.View..android.view.View?) the next focused view
 */
fun Activity.findNextDPADFocus(keyCode: Int, rootView: ViewGroup): View? = FocusFinder.getInstance().findNextFocus(rootView, rootView.findFocus(), when (keyCode) {
    KeyEvent.KEYCODE_DPAD_LEFT -> {
        View.FOCUS_LEFT
    }
    KeyEvent.KEYCODE_DPAD_UP -> {
        View.FOCUS_UP
    }
    KeyEvent.KEYCODE_DPAD_RIGHT -> {
        View.FOCUS_RIGHT
    }
    KeyEvent.KEYCODE_DPAD_DOWN -> {
        View.FOCUS_DOWN
    }
    KeyEvent.KEYCODE_TAB -> {
        View.FOCUS_FORWARD
    }
    else -> {
        View.FOCUS_FORWARD
    }
})

/**
 * Check if activity is currently displayed in multi window mode.
 * @receiver Activity
 * @return Boolean
 */
fun Activity.checkIsInMultiWindowMode(): Boolean = if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.N) {
    this.isInMultiWindowMode
} else false