/*
 * Copyright (C) 2018 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.sdk_design.edittexts

import android.content.Context
import android.graphics.Typeface
import androidx.annotation.AnyRes
import androidx.annotation.ColorInt
import androidx.appcompat.widget.AppCompatEditText
import android.util.AttributeSet
import android.widget.EditText
import com.airbnb.paris.annotations.Attr
import com.airbnb.paris.annotations.LayoutDimension
import com.airbnb.paris.annotations.Styleable
import com.bandyer.sdk_design.R
import com.bandyer.sdk_design.R2
import com.bandyer.sdk_design.extensions.StyleCompat
import com.bandyer.sdk_design.extensions.getFont

/**
 * Bandyer Design SDK base custom editable text.
 * Style can be customized with the following styles:
 * 'BandyerSDKDesign.EditText.
 */
@Styleable("BandyerSDKDesign_TextView")
class BandyerEditText @JvmOverloads constructor(context: Context, attrs: AttributeSet? = null, defStyleAttr: Int = 0) : AppCompatEditText(context, attrs, defStyleAttr) {

    private var mShadowDx: Float? = null
    private var mShadowDy: Float? = null
    private var mShadowRadius: Float? = null
    private var mShadowColor: Int? = null

    private var fontFamily: Int = 0

    init {
        StyleCompat.style(this).builder().add(R.style.BandyerSDKDesign_EditText).apply()
    }

    /**
     * Set font typeface given a font resource
     * @param typeface Font resource file
     */
    @Attr(R2.styleable.BandyerSDKDesign_TextView_bandyer_font_name)
    fun setFontName(@AnyRes typeface: Int) {
        this.fontFamily = typeface
        this.typeface = getFont(typeface, Typeface.NORMAL)
    }

    /**
     * When the view is attached to window, optional shadow is computed and applied.
     */
    override fun onAttachedToWindow() {
        super.onAttachedToWindow()

        if (mShadowColor != null)
            setShadowLayer(mShadowRadius ?: 0f, mShadowDx ?: 0f, mShadowDy ?: 0f, mShadowColor ?: 0)
    }

    /**
     * Set shadow dx
     * @param dimension pixels of shadow in x direction
     */
    @Attr(R2.styleable.BandyerSDKDesign_TextView_bandyer_shadowDx)
    fun setShadowDx(@LayoutDimension dimension: Int) {
        mShadowDx = dimension.toFloat()
    }

    /**
     * Set shadow dy
     * @param dimension pixels of shadow in y direction
     */
    @Attr(R2.styleable.BandyerSDKDesign_TextView_bandyer_shadowDy)
    fun setShadowDy(@LayoutDimension dimension: Int) {
        mShadowDy = dimension.toFloat()
    }

    /**
     * Set shadow radius
     * @param dimension pixels of shadow radius
     */
    @Attr(R2.styleable.BandyerSDKDesign_TextView_bandyer_shadowRadius)
    fun setShadowRadius(@LayoutDimension dimension: Int) {
        mShadowRadius = dimension.toFloat()
    }

    /**
     * Set shadow color
     * @param color Color int to apply on shadow
     */
    @Attr(R2.styleable.BandyerSDKDesign_TextView_shadowColor)
    fun setShadowColor(@ColorInt color: Int) {
        mShadowColor = color
    }
}