/*
 * Copyright (C) 2018 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.sdk_design.chat.widgets

import android.content.Context
import android.graphics.Color
import android.text.Editable
import android.text.TextWatcher
import android.util.AttributeSet
import android.view.Gravity
import android.view.LayoutInflater
import android.view.ViewGroup
import android.view.inputmethod.EditorInfo
import android.widget.LinearLayout
import com.airbnb.paris.annotations.Attr
import com.airbnb.paris.annotations.Styleable
import com.airbnb.paris.annotations.StyleableChild
import com.bandyer.sdk_design.R
import com.bandyer.sdk_design.R2
import com.bandyer.sdk_design.buttons.BandyerImageButton
import com.bandyer.sdk_design.edittexts.BandyerEditText
import com.bandyer.sdk_design.extensions.StyleCompat
import kotlinx.android.synthetic.main.bandyer_widget_chat_input_layout.view.*

/**
 * @suppress
 */
@Styleable("BandyerSDKDesign_Widget_ChatInputLayout")
class BandyerChatInputLayoutWidget @JvmOverloads constructor(context: Context, attrs: AttributeSet? = null, defStyleAttr: Int? = null)
    : LinearLayout(context, attrs, defStyleAttr ?: 0) {

    @StyleableChild(R2.styleable.BandyerSDKDesign_Widget_ChatInputLayout_bandyer_inputFieldStyle)
    var input: BandyerEditText? = null

    @StyleableChild(R2.styleable.BandyerSDKDesign_Widget_ChatInputLayout_bandyer_iconSendStyle)
    var send: BandyerImageButton? = null

    private var activeIconSendColor = Color.BLACK
    private var inactiveIconSendColor = Color.LTGRAY

    var callback: BandyerChatInputLayoutEventListener? = null

    private var textWatcher: TextWatcher? = null

    init {

        val view = LayoutInflater.from(context).inflate(R.layout.bandyer_widget_chat_input_layout, this) as LinearLayout
        view.orientation = HORIZONTAL
        view.gravity = Gravity.CENTER_VERTICAL

        view.descendantFocusability = ViewGroup.FOCUS_AFTER_DESCENDANTS

        input = write_here_edit_text

        send = send_button
        send?.setOnClickListener {
            if (input?.text != null && input?.text!!.isNotEmpty()) {
                highlightSendButton(false)
                callback?.onSendClicked(input?.text!!.toString().trim()).apply { input?.setText("") }
            }
        }

        input?.setOnEditorActionListener { v, actionId, event ->
            if (actionId == EditorInfo.IME_ACTION_SEND) {
                send?.performClick()
                true
            } else false
        }

        textWatcher = object : TextWatcher {
            override fun afterTextChanged(p0: Editable?) {}
            override fun beforeTextChanged(p0: CharSequence?, p1: Int, p2: Int, p3: Int) {}
            override fun onTextChanged(p0: CharSequence?, p1: Int, p2: Int, p3: Int) {

                if (p0 != null && p0.isNotEmpty())
                    highlightSendButton(true)
                else
                    highlightSendButton(false)

                callback?.onTextChanged(p0.toString())
            }
        }

        input?.addTextChangedListener(textWatcher)

        StyleCompat.styleBuilder(this)
                .add(R.style.BandyerSDKDesign_Widget_ChatInputLayout)
                .add(attrs)
                .apply()

        highlightSendButton(false)
    }

    fun dispose() {
        textWatcher?.let {
            input?.removeTextChangedListener(it)
            textWatcher = null
        }
        callback = null
    }

    override fun onDetachedFromWindow() {
        dispose()
        super.onDetachedFromWindow()
    }

    private fun highlightSendButton(enabled: Boolean) {
        if (enabled)
            send?.setColorFilter(activeIconSendColor)
        else
            send?.setColorFilter(inactiveIconSendColor)
    }

    @Attr(R2.styleable.BandyerSDKDesign_Widget_ChatInputLayout_bandyer_activeIconSendColor)
    fun getActiveIconSendColor(value: Int) {
        activeIconSendColor = value
    }

    @Attr(R2.styleable.BandyerSDKDesign_Widget_ChatInputLayout_bandyer_inactiveIconSendColor)
    fun getInactiveIconSendColor(value: Int) {
        inactiveIconSendColor = value
    }
}

/**
 * @suppress
 */
interface BandyerChatInputLayoutEventListener {
    fun onTextChanged(text: String)
    fun onSendClicked(text: String)
}