/*
 * Copyright (C) 2018 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.sdk_design.chat.widgets

import android.annotation.SuppressLint
import android.content.Context
import android.graphics.Bitmap
import androidx.annotation.DrawableRes
import android.util.AttributeSet
import android.view.LayoutInflater
import android.view.View
import android.widget.LinearLayout
import com.airbnb.paris.annotations.Styleable
import com.airbnb.paris.annotations.StyleableChild
import com.bandyer.sdk_design.R
import com.bandyer.sdk_design.R2
import com.bandyer.sdk_design.call.imageviews.BandyerAvatarImageView
import com.bandyer.sdk_design.extensions.StyleCompat
import com.bandyer.sdk_design.imageviews.BandyerRoundedImageView
import com.bandyer.sdk_design.textviews.BandyerTextViewSubtitle
import com.bandyer.sdk_design.textviews.BandyerTextViewSubtitleBouncingDots
import com.bandyer.sdk_design.textviews.BandyerTextViewTitle
import kotlinx.android.synthetic.main.bandyer_widget_chatinfo.view.*

/**
 * This class represent a widget used to display in-chat informations.
 * It has a tile, a subtitle and bouncing dots for typing action displaying.
 * Text and icon styles can be customized modifying the following styles:
 * 'BandyerSDKDesign.TextView.Title'
 * 'BandyerSDKDesign.TextView.SubTitle'
 * 'BandyerSDKDesign.ChatInfoStyle'
 */
@Styleable("BandyerSDKDesign_Widget_ChatInfo")
class BandyerChatInfoWidget @JvmOverloads constructor(context: Context, attrs: AttributeSet? = null, defStyleAttr: Int? = null)
    : LinearLayout(context, attrs, defStyleAttr ?: 0) {

    /**
     * View used to display contact names
     */
    @StyleableChild(R2.styleable.BandyerSDKDesign_Widget_ChatInfo_bandyer_chatTitleStyle)
    var contactNameView: BandyerTextViewTitle? = null
        private set

    /**
     * View used to display contact chat status ( online, offline, typing )
     */
    @StyleableChild(R2.styleable.BandyerSDKDesign_Widget_ChatInfo_bandyer_chatSubtitleStyle)
    var contactStatusView: BandyerTextViewSubtitle? = null
        private set

    /**
     * View used to display contact image
     */
    @StyleableChild(R2.styleable.BandyerSDKDesign_Widget_ChatInfo_bandyer_avatarStyle)
    var contactImageView: BandyerAvatarImageView? = null
        private set

    /**
     * View effect used in typing status
     */
    @StyleableChild(R2.styleable.BandyerSDKDesign_Widget_ChatInfo_bandyer_typingDotsStyle)
    var typingDotsView: BandyerTextViewSubtitleBouncingDots? = null
        private set

    /**
     * Change or get the current status of the contact
     */
    var state: BandyerChatInfoWidgetState? = BandyerChatInfoWidgetState.WAITING_FOR_NETWORK()
        @SuppressLint("SetTextI18n")
        set(value) {
            field = value
            typingDotsView?.hideAndStop()
            contactStatusView ?: return

            when (value) {
                is BandyerChatInfoWidgetState.OFFLINE -> {
                    StyleCompat
                            .style(contactStatusView)
                            .apply(R.style.BandyerSDKDesign_TextView_Subtitle_Offline_ChatInfo)
                    second_line!!.visibility = View.VISIBLE
                    if (value.lastLogin!=null && !value.lastLogin.isNullOrBlank())
                        status_textView!!.text = context.getString(R.string.bandyer_chat_user_status_last_login) + " " +  value.lastLogin!!
                    else
                        status_textView!!.text = context.getText(R.string.bandyer_chat_user_status_offline)
                }
                is BandyerChatInfoWidgetState.ONLINE -> {
                    StyleCompat
                            .style(contactStatusView)
                            .apply(R.style.BandyerSDKDesign_TextView_Subtitle_Online_ChatInfo)
                    second_line!!.visibility = View.VISIBLE
                }
                is BandyerChatInfoWidgetState.TYPING -> {
                    typingDotsView?.showAndPlay()
                    StyleCompat
                            .style(contactStatusView)
                            .apply(R.style.BandyerSDKDesign_TextView_Subtitle_Typing_ChatInfo)
                    second_line!!.visibility = View.VISIBLE
                }
                is BandyerChatInfoWidgetState.WAITING_FOR_NETWORK -> {
                    StyleCompat
                            .style(contactStatusView)
                            .apply(R.style.BandyerSDKDesign_TextView_Subtitle_WaitingForNetwork_ChatInfo)
                    second_line!!.visibility = View.VISIBLE
                }
                is BandyerChatInfoWidgetState.CONNECTING -> {
                    StyleCompat
                            .style(contactStatusView)
                            .apply(R.style.BandyerSDKDesign_TextView_Subtitle_Connecting_ChatInfo)
                    second_line!!.visibility = View.VISIBLE
                }
                else -> {
                    second_line!!.visibility = View.GONE
                }
            }
            status_textView.isSelected = true
            field = value
        }

    init {
        LayoutInflater.from(context).inflate(R.layout.bandyer_widget_chatinfo, this)

        contactNameView = first_line
        contactNameView?.visibility = View.GONE
        contactStatusView = status_textView
        contactImageView = avatar
        typingDotsView = bouncing_dots

        contactNameView?.isSelected = true // activate marquee

        StyleCompat.styleBuilder(this)
                .add(attrs)
                .add(R.style.BandyerSDKDesign_Widget_ChatInfo)
                .apply()
    }

    /**
     * Set the contact name
     * @param name the name to display
     */
    fun setName(name: String) {
        contactNameView!!.text = name
    }

    /**
     * Display contact image given the url
     * @param url image
     */
    fun setImage(url: String) {
        contactImageView?.setImageUrl(url)
    }

    /**
     * Display contact image given a resource
     * @param resId image
     */
    fun setImage(@DrawableRes resId: Int) {
        contactImageView?.setImageResId(resId)
    }

    /**
     * Display contact image given a bitmap
     * @param bitmap image
     */
    fun setImage(bitmap: Bitmap) {
        contactImageView?.setImageBitmap(bitmap)
    }

    /**
     * Bandyer Chat Info Widget States
     */
    sealed class BandyerChatInfoWidgetState {

        /**
         * When the contact is online
         */
        class ONLINE : BandyerChatInfoWidgetState()

        /**
         * When the contact is offline
         * @property lastLogin last login description text
         */
        class OFFLINE(var lastLogin: String? = null) : BandyerChatInfoWidgetState()

        /**
         * When the contact is typing a message
         */
        class TYPING : BandyerChatInfoWidgetState()

        /**
         * When the contact is in an unknown state
         */
        @Suppress("ClassName")
        class WAITING_FOR_NETWORK : BandyerChatInfoWidgetState()

        /**
         * When the contact is online
         */
        class CONNECTING : BandyerChatInfoWidgetState()

        /**
         * When the contact is in an unknown state
         */
        class UNKNOWN : BandyerChatInfoWidgetState()

        /**
         * @suppress
         * @param other Any?
         * @return Boolean
         */
        override fun equals(other: Any?): Boolean {
            return this === other
        }

        /**
         * @suppress
         * @return Int
         */
        override fun hashCode(): Int {
            return System.identityHashCode(this)
        }
    }
}