/*
 * Copyright (C) 2018 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.sdk_design.chat.adapter_items.message.abstract_message

import androidx.constraintlayout.widget.ConstraintLayout
import androidx.constraintlayout.widget.ConstraintSet
import android.text.format.DateUtils
import android.view.View
import com.bandyer.sdk_design.chat.layout.BandyerBaseChatMessageLayout
import com.bandyer.sdk_design.extensions.dp2px
import com.mikepenz.fastadapter.FastAdapter

/**
 * Base Chat Message View Holder
 * @property F Layout to use for the message view
 * @property T Type of Message Item to display
 * @constructor
 */
@Suppress("UNCHECKED_CAST")
abstract class BandyerBaseChatMessageViewHolder<F, T> constructor(view: View) : FastAdapter.ViewHolder<T>(view) where T : BandyerBaseChatMessageItem<*>, F : BandyerBaseChatMessageLayout {

    private var messageLayout: F = view as F
    private var messageContent: ConstraintLayout = messageLayout.dataViewContainer as ConstraintLayout

    /**
     * @suppress
     */
    final override fun bindView(item: T, payloads: MutableList<Any>) {
        bind(item, messageLayout, payloads)
        val alignment: Int = if (item.data.mine) ConstraintSet.END else ConstraintSet.START

        style(item, messageLayout)

        val pending = item.data.pending
        val seen = item.data.seen.invoke()
        val sent = item.data.sent

        if (pending)
            messageLayout.statusView?.setAsPending()
        if (seen)
            messageLayout.statusView?.setAsSeen()
        if (sent)
            messageLayout.statusView?.setAsSent()

        messageLayout.showStatus(item.data.mine && (pending || seen || sent))

        messageLayout.timestampTextView!!.text = DateUtils.getRelativeTimeSpanString(item.data.timestamp)

        val space16 = messageLayout.context.dp2px(16f)

        ConstraintSet().apply {
            clone(messageLayout)
            connect(messageContent.id, alignment, messageLayout.id, alignment, space16)
        }.applyTo(messageLayout)
    }

    /**
     * Style the message
     * @param item T Message Item
     * @param itemView F  Message Item
     */
    abstract fun style(item: T, itemView: F)

    /**
     * On Bind of the message
     * @param item T  Message Item
     * @param itemView F Message Item
     * @param payloads MutableList<Any> of payload
     */
    abstract fun bind(item: T, itemView: F, payloads: MutableList<Any>)

    /**
     * On unBind of the message
     * @param item T Message Item
     * @param itemView F Message Item
     */
    abstract fun unbind(item: T, itemView: F)

    /**
     * @suppress
     */
    final override fun unbindView(item: T) {
        unbind(item, messageLayout)
        ConstraintSet().apply {
            clone(messageLayout)
            clear(messageContent.id)
        }.applyTo(messageLayout)
        messageLayout.showStatus(false)
    }
}