/*
 * Copyright (C) 2018 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.sdk_design.call.widgets

import android.content.Context
import android.os.CountDownTimer
import android.util.AttributeSet
import android.view.LayoutInflater
import com.airbnb.paris.annotations.Styleable
import com.airbnb.paris.annotations.StyleableChild
import com.airbnb.paris.utils.setPaddingTop
import com.bandyer.sdk_design.R
import com.bandyer.sdk_design.R2
import com.bandyer.sdk_design.buttons.BandyerImageButton
import com.bandyer.sdk_design.extensions.StyleCompat
import com.bandyer.sdk_design.layout.SystemControlsAwareFrameLayout
import com.bandyer.sdk_design.textviews.BandyerTextView
import com.bandyer.sdk_design.utils.ToggleableVisibilityInterface
import com.bandyer.sdk_design.utils.VisibilityToggle
import kotlinx.android.synthetic.main.bandyer_widget_call_user_info.view.*

/**
 * Bandyer user info widget that will be displayed from the top
 * @property fullscreenActionButton BandyerImageButton?
 * @property displayName BandyerTextView?
 * @property currentTopPixels Int pixels from the top of the screen
 * @property considerSystemAwareHints Boolean
 * @property text String?
 * @constructor
 */
@Styleable("BandyerSDKDesign_Widget_CallUserInfo")
class BandyerCallUserInfoWidget @JvmOverloads constructor(context: Context, attrs: AttributeSet? = null, defStyleAttr: Int = 0) :
        SystemControlsAwareFrameLayout(context, attrs, defStyleAttr), ToggleableVisibilityInterface {

    /**
     * Fullscreen action button
     */
    @StyleableChild(R2.styleable.BandyerSDKDesign_Widget_CallUserInfo_bandyer_buttonStyle)
    var fullscreenActionButton: BandyerImageButton? = null

    /**
     * Display name label
     */
    @StyleableChild(R2.styleable.BandyerSDKDesign_Widget_CallUserInfo_bandyer_labelStyle)
    var displayName: BandyerTextView? = null

    private var currentTopPixels = 0

    var considerSystemAwareHints = false
        set(value) {
            field = value
            onTopInsetChanged(if (value) currentTopPixels else 0)
        }

    var text: String? = null
        set(value) {
            display_name?.text = value
            field = value
        }

    private var visibilityToggle = VisibilityToggle(this)

    private var attrs: AttributeSet? = null

    init {
        LayoutInflater.from(context).inflate(R.layout.bandyer_widget_call_user_info, this)
        fullscreenActionButton = fullscreen_action_button
        displayName = display_name

        this.attrs = attrs

        setFullscreenStyle(false)
    }

    override fun toggleVisibility(show: Boolean, animationEndCallback: (shown: Boolean) -> Unit) {
        visibilityToggle.toggleVisibility(show, animationEndCallback)
    }

    /**
     * Enables fullscreen action button style
     * @param fullscreen Boolean true if fullscreen, false otherwise
     */
    fun setFullscreenStyle(fullscreen: Boolean) {
        if (fullscreen) StyleCompat.styleBuilder(this)
                .add(attrs)
                .add(R.style.BandyerSDKDesign_Widget_CallUserInfo_Fullscreen)
                .apply()
            else StyleCompat.styleBuilder(this)
                .add(attrs)
                .add(R.style.BandyerSDKDesign_Widget_CallUserInfo)
                .apply()
    }

    override fun cancelTimer() {
        visibilityToggle?.cancelTimer()
    }

    /**
     * Set alpha value
     * @param alpha alpha
     */
    override fun setAlpha(alpha: Float) {
        super.setAlpha(alpha)
        if (alpha != 0f) return
        cancelTimer()
    }

    override fun onDetachedFromWindow() {
        super.onDetachedFromWindow()
        cancelTimer()
    }

    /**
     * @suppress
     */
    override fun onTopInsetChanged(pixels: Int) {
        if (paddingTop == pixels) return
        if (pixels > 0) currentTopPixels = pixels
        post {
            setPaddingTop(if (considerSystemAwareHints) currentTopPixels else 0)
        }
    }
}