/*
 * Copyright (C) 2018 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.sdk_design.call.widgets


import android.animation.Animator
import android.content.Context
import android.graphics.drawable.Drawable
import android.os.Build
import android.os.CountDownTimer
import android.util.AttributeSet
import android.view.LayoutInflater
import android.view.View
import androidx.appcompat.app.AppCompatActivity
import androidx.constraintlayout.widget.ConstraintLayout
import com.airbnb.paris.annotations.Attr
import com.airbnb.paris.annotations.Styleable
import com.airbnb.paris.annotations.StyleableChild
import com.airbnb.paris.utils.setPaddingLeft
import com.airbnb.paris.utils.setPaddingRight
import com.airbnb.paris.utils.setPaddingTop
import com.bandyer.sdk_design.R
import com.bandyer.sdk_design.R2
import com.bandyer.sdk_design.call.layout.BandyerCallWatermarkLayout
import com.bandyer.sdk_design.extensions.StyleCompat
import com.bandyer.sdk_design.textviews.BandyerTextViewSubtitle
import com.bandyer.sdk_design.textviews.BandyerTextViewTitle
import com.bandyer.sdk_design.utils.systemviews.SystemViewLayoutObserver
import com.bandyer.sdk_design.utils.systemviews.SystemViewLayoutOffsetListener
import com.bandyer.sdk_design.widgets.HideableWidget
import kotlinx.android.synthetic.main.bandyer_widget_callinfo.view.*

/**
 * This class represent a widget used to display in-call informations.
 * It has a tile, a subtitle and an icon that can be used to display the recording state of the video/audio call.
 * Text and icon styles can be customized modifying the following styles:
 * 'BandyerSDKDesign.TextView.Title'
 * 'BandyerSDKDesign.TextView.SubTitle'
 * 'BandyerSDKDesign.CallInfoStyle'
 */
@Styleable("BandyerSDKDesign_Widget_CallInfo")
class BandyerCallInfoWidget @JvmOverloads constructor(context: Context, attrs: AttributeSet? = null, defStyleAttr: Int? = null)
    : ConstraintLayout(context, attrs, defStyleAttr
        ?: 0), SystemViewLayoutObserver, HideableWidget {

    override var hidingTimer: CountDownTimer? = null

    private var drawables: Array<Drawable>? = null

    private var AUTO_HIDE_ANIMATION_DURATION_MS = 250L
    private var AUTO_SHOW_ANIMATION_DURATION_MS = 125L

    /**
     * Title view
     */
    @StyleableChild(R2.styleable.BandyerSDKDesign_Widget_CallInfo_bandyer_titleStyle)
    var titleView: BandyerTextViewTitle? = null
        private set

    /**
     * Subtitle view
     */
    @StyleableChild(R2.styleable.BandyerSDKDesign_Widget_CallInfo_bandyer_subtitleStyle)
    var subtitleView: BandyerTextViewSubtitle? = null
        private set

    /**
     * Recording view
     */
    @StyleableChild(R2.styleable.BandyerSDKDesign_Widget_CallInfo_bandyer_recordingStyle)
    var recordingView: BandyerTextViewSubtitle? = null
        private set


    /**
     * Recording view
     */
    @StyleableChild(R2.styleable.BandyerSDKDesign_Widget_CallInfo_bandyer_watermarkStyle)
    var watermarkView: BandyerCallWatermarkLayout? = null
        private set

    private var initialPaddingStart = -1
    private var initialPaddingEnd = -1
    private var initialPaddingLeft = -1
    private var initialPaddingRight = -1
    private var initialPaddingTop = -1
    private var initialPaddingBottom = -1

    init {
        LayoutInflater.from(context).inflate(R.layout.bandyer_widget_callinfo, this, true)

        titleView = title!!
        subtitleView = subtitle!!
        recordingView = recording!!
        titleView?.isSelected = true // activate marquee
        watermarkView = watermark!!

        StyleCompat.styleBuilder(this)
                .add(attrs)
                .add(R.style.BandyerSDKDesign_Widget_CallInfo)
                .apply()

        if (drawables == null) drawables = recordingView?.compoundDrawables

        initialPaddingLeft = paddingLeft
        initialPaddingTop = paddingTop
        initialPaddingRight = paddingRight
        initialPaddingBottom = paddingBottom
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) initialPaddingEnd = paddingEnd
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) initialPaddingStart = paddingStart
    }

    /**
     * @suppress
     */
    override fun onAttachedToWindow() {
        super.onAttachedToWindow()
        SystemViewLayoutOffsetListener.addObserver(context as AppCompatActivity, this)
    }

    /**
     * @suppress
     */
    override fun onDetachedFromWindow() {
        super.onDetachedFromWindow()
        SystemViewLayoutOffsetListener.removeObserver(context as AppCompatActivity, this)
    }

    override fun onHidingTimerFinished() {
        animate().translationY(-height.toFloat()).setListener(object : Animator.AnimatorListener {
            override fun onAnimationRepeat(animation: Animator?) {}
            override fun onAnimationEnd(animation: Animator?) {
                visibility = View.INVISIBLE
            }

            override fun onAnimationCancel(animation: Animator?) {}
            override fun onAnimationStart(animation: Animator?) {}
        }).setDuration(AUTO_HIDE_ANIMATION_DURATION_MS).start()
    }

    /**
     * Method to show & hide the widget
     */
    fun toggle() {
        disableAutoHide()
        if (visibility == View.VISIBLE) {
            onHidingTimerFinished()
        } else {
            animate().translationY(0f).setDuration(AUTO_SHOW_ANIMATION_DURATION_MS).start()
            visibility = View.VISIBLE
        }
        hidingTimer?.start()
    }

    /**
     * Indefinitely shows the widget.
     */
    fun show() {
        animate().translationY(0f).setListener(object : Animator.AnimatorListener {
            override fun onAnimationRepeat(animation: Animator?) {}
            override fun onAnimationEnd(animation: Animator?) {
                visibility = View.VISIBLE
            }

            override fun onAnimationCancel(animation: Animator?) {}
            override fun onAnimationStart(animation: Animator?) {}
        }).setDuration(AUTO_SHOW_ANIMATION_DURATION_MS).start()
    }

    /**
     * Display title text
     * @param titleText title
     */
    fun setTitle(titleText: String) {
        titleView?.text = titleText
    }

    /**
     * Display subtitle text
     * @param subtitleText subtitle
     */
    fun setSubtitle(subtitleText: String) {
        subtitleView?.text = subtitleText
    }

    /**
     * Hide subtitle view
     */
    fun hideSubtitle() {
        subtitleView?.visibility = View.GONE
    }

    /**
     * Show subtitle view
     */
    fun showSubtitle() {
        subtitleView?.visibility = View.VISIBLE
    }

    /**
     * Show subtitle view
     */
    fun showTitle() {
        titleView?.visibility = View.VISIBLE
    }

    /**
     * Display recording text
     * @param recordingText title
     */
    fun setRecordingText(recordingText: String) {
        recordingView?.text = recordingText
    }

    /**
     * Hide title view
     */
    fun hideTitle() {
        titleView?.visibility = View.GONE
    }

    /**
     * Hide recording view
     */
    fun hideRecording() {
        recordingView?.visibility = View.GONE
    }

    /**
     * Show or hide recording icon
     * @param enabled enabled state
     */
    @Attr(R2.styleable.BandyerSDKDesign_Widget_CallInfo_bandyer_recording)
    fun setRecording(enabled: Boolean) {
        if (drawables == null)
            drawables = recordingView?.compoundDrawables
        if (enabled) {
            drawables?.let {
                recordingView?.setCompoundDrawablesWithIntrinsicBounds(it[0], it[1], it[2], it[3])
                recordingView?.visibility = View.VISIBLE
            }
        } else {
            recordingView?.setCompoundDrawablesWithIntrinsicBounds(null, null, null, null)
            recordingView?.visibility = View.GONE
        }

    }

    /**
     * @suppress
     */
    override fun onTopInsetChanged(pixels: Int) {
        post {
            val padOffset = initialPaddingTop.takeIf { it > -1 } ?: 0
            setPaddingTop(pixels + padOffset)
        }
    }

    /**
     * @suppress
     */
    override fun onBottomInsetChanged(pixels: Int) {}

    /**
     * @suppress
     */
    override fun onRightInsetChanged(pixels: Int) {
        post {
            val padOffset = initialPaddingEnd.takeIf { it > -1 }
                    ?: initialPaddingRight.takeIf { it > -1 } ?: 0
            setPaddingRight(pixels + padOffset)
        }
    }

    /**
     * @suppress
     */
    override fun onLeftInsetChanged(pixels: Int) {
        post {
            val padOffset = initialPaddingStart.takeIf { it > -1 }
                    ?: initialPaddingLeft.takeIf { it > -1 } ?: 0
            setPaddingLeft(pixels)
        }
    }
}