/*
 * Copyright (C) 2018 Bandyer S.r.l. All Rights Reserved.
 * See LICENSE.txt for licensing information
 */

package com.bandyer.sdk_design.call.widgets

import android.os.Bundle
import android.os.CountDownTimer
import android.view.Gravity
import android.view.View
import android.view.ViewGroup
import androidx.annotation.Px
import androidx.appcompat.app.AppCompatActivity
import androidx.coordinatorlayout.widget.CoordinatorLayout
import com.badoo.mobile.util.WeakHandler
import com.bandyer.sdk_design.R
import com.bandyer.sdk_design.bottom_sheet.BandyerActionBottomSheet
import com.bandyer.sdk_design.bottom_sheet.BandyerBottomSheet
import com.bandyer.sdk_design.bottom_sheet.BandyerClickableBottomSheet
import com.bandyer.sdk_design.bottom_sheet.OnStateChangedBottomSheetListener
import com.bandyer.sdk_design.bottom_sheet.behaviours.BandyerBottomSheetBehaviour
import com.bandyer.sdk_design.bottom_sheet.items.ActionItem
import com.bandyer.sdk_design.bottom_sheet.view.BottomSheetLayoutContent
import com.bandyer.sdk_design.call.bottom_sheet.AudioRouteBottomSheet
import com.bandyer.sdk_design.call.bottom_sheet.CallBottomSheet
import com.bandyer.sdk_design.call.bottom_sheet.CallBottomSheet.Companion.MAX_ITEMS_PER_ROW
import com.bandyer.sdk_design.call.bottom_sheet.OnAudioRouteBottomSheetListener
import com.bandyer.sdk_design.call.bottom_sheet.RingingBottomSheet
import com.bandyer.sdk_design.call.bottom_sheet.items.AudioRoute
import com.bandyer.sdk_design.call.bottom_sheet.items.CallAction
import com.bandyer.sdk_design.extensions.getScreenSize
import com.bandyer.sdk_design.widgets.HideableWidget

/**
 * Widget used during a call to perform actions such as mute video/audio, change audioRoute etc.
 * @param context AppCompatActivity context
 * @param callActionItems items to use
 * @constructor
 * @author kristiyan
 */
class BandyerCallActionWidget<T, F>(val context: AppCompatActivity, val callActionItems: List<CallAction>) : HideableWidget where T : ActionItem, F : BandyerBottomSheet {

    override var hidingTimer: CountDownTimer? = null

    /**
     * Click listener for when an item has been clicked
     */
    var onClickListener: OnClickListener? = null

    /**
     * Used to request available audioRoutes
     */
    var onAudioRoutesRequest: OnAudioRouteBottomSheetListener? = null

    private var anchoredViews: MutableList<Pair<View, Int>> = mutableListOf()
    private var currentBottomSheetLayout: BottomSheetLayoutContent? = null

    private var currentShownBottomSheet: BandyerBottomSheet? = null

    /**
     * Sliding listener for when the widget has been slided
     */
    var slidingListener: SlidingListener? = null

    /**
     * Current widget hidden state
     */
    var isHidden = false

    /**
     * Current onShowListener
     */
    private var onShowListener: OnShowListener? = null

    /**
     * Current onHiddenListener
     */
    private var onHiddenListener: OnHiddenListener? = null

    /**
     * Weak handler to safely call delayed tasks
     */
    private val uiHandler by lazy {
        WeakHandler()
    }

    /**
     * Widget states
     */
    enum class BandyerCallActionWidgetState {
        /**
         * Fully expanded
         */
        EXPANDED,

        /**
         * Anchored
         */
        ANCHORED,

        /**
         * Collapsed
         */
        COLLAPSED
    }


    init {

        callActionItems.forEachIndexed { index, callAction ->
            if (index < callActionItems.size - 1) {
                callAction.itemView?.nextFocusForwardId =
                        callActionItems[index + 1].itemView?.id!!
            }
        }
    }

    /**
     * Current widget state
     */
    @Suppress("SetterBackingFieldAssignment")
    var state: BandyerCallActionWidgetState? = null

    /**
     * Save current instance state
     * @param saveInstanceState Bundle? to store in
     * @return Bundle updated
     */
    fun saveInstanceState(saveInstanceState: Bundle?): Bundle? {
        val bundle = audioRouteBottomSheet.saveInstanceState(saveInstanceState)
        return callBottomSheet.saveInstanceState(bundle)
    }

    /**
     * Restore a previously saved instance state
     * @param bundle Bundle? to restore from
     */
    fun restoreInstanceState(bundle: Bundle?) {
        callBottomSheet.restoreInstanceState(bundle)
        audioRouteBottomSheet.restoreInstanceState(bundle)
    }

    /**
     * Toggle the widget.
     */
    fun toggle() {
        if (callBottomSheet.isVisible())
            callBottomSheet.toggle()
        if (audioRouteBottomSheet.isVisible())
            audioRouteBottomSheet.toggle()
    }

    /**
     * Expand the widget.
     */
    fun expand() {
        if (callBottomSheet.isVisible())
            callBottomSheet.expand()
        if (audioRouteBottomSheet.isVisible())
            audioRouteBottomSheet.expand()
    }

    /**
     * Request focus on current bottom sheet shown.
     */
    fun requestFocus(): View? {
        val toBeFocused = when {
            ringingBottomSheet.isVisible() -> ringingBottomSheet.bottomSheetLayoutContent.recyclerView?.layoutManager?.findViewByPosition(1)
            callBottomSheet.isVisible() -> callBottomSheet.bottomSheetLayoutContent.recyclerView?.layoutManager?.findViewByPosition(0)
            audioRouteBottomSheet.isVisible() -> audioRouteBottomSheet.bottomSheetLayoutContent.recyclerView?.layoutManager?.findViewByPosition(0)
            else -> null
        }
        toBeFocused?.requestFocus()
        return toBeFocused
    }

    /**
     * Collapse the widget.
     */
    fun collapse() {
        if (callBottomSheet.isVisible()) {
            if (collapsible) callBottomSheet.collapse()
            else callBottomSheet.anchor()
        }
        if (audioRouteBottomSheet.isVisible())
            audioRouteBottomSheet.hide()
    }

    /**
     * Hides current displayed bottomsheet
     */
    @JvmOverloads
    fun hide(onHiddenListener: OnHiddenListener? = null) {
        if (isHidden) return
        isHidden = true
        if (onHiddenListener != null) this.onHiddenListener = onHiddenListener
        when (currentShownBottomSheet) {
            callBottomSheet -> {
                callBottomSheet.hide(true)
            }
            audioRouteBottomSheet -> {
                removeAnchorFromAnchoredView()
                audioRouteBottomSheet.dispose()
            }
            ringingBottomSheet -> {
                removeAnchorFromAnchoredView()
                ringingBottomSheet.dispose()
            }
        }
    }

    /**
     * Shows bandyer call action widget based on previous last bottom sheet shown
     */
    @JvmOverloads
    fun show(onShowListener: OnShowListener? = null) {
        if (!isHidden) return
        isHidden = false
        if (onShowListener != null) this.onShowListener = onShowListener
        when (currentShownBottomSheet) {
            ringingBottomSheet -> showRingingControls()
            else -> showCallControls(collapsible, fixed, collapsed = true)
        }
    }

    /**
     * Shows bandyer call action widget based on previous last bottom sheet shown
     * delayed by factor
     * @param millis delay factor in milliseconds
     */
    fun showDelayed(millis: Long, onShowListener: OnShowListener? = null) {
        uiHandler.postDelayed({ show(onShowListener) }, millis)
    }

    /**
     * Check expanded status.
     * @return true if widget is expanded, false otherwise.
     */
    fun isExpanded(): Boolean {
        return when {
            ringingBottomSheet.isVisible() -> true
            callBottomSheet.isVisible() -> callBottomSheet.isExpanded()
            audioRouteBottomSheet.isVisible() -> audioRouteBottomSheet.isExpanded()
            else -> false
        }
    }

    /**
     * Check collapsed status.
     * @return true if widget is expanded, false otherwise.
     */
    fun isCollapsed(): Boolean {
        return when {
            ringingBottomSheet.isVisible() -> false
            callBottomSheet.isVisible() -> callBottomSheet.isCollapsed()
            audioRouteBottomSheet.isVisible() -> audioRouteBottomSheet.isCollapsed()
            else -> false
        }
    }

    /**
     * Check anchored status.
     * @return true if widget is anchored, false otherwise.
     */
    fun isAnchored(): Boolean {
        return when {
            ringingBottomSheet.isVisible() -> true
            callBottomSheet.isVisible() -> callBottomSheet.isAnchored()
            audioRouteBottomSheet.isVisible() -> audioRouteBottomSheet.isAnchored()
            else -> false
        }
    }

    /**
     * Show ringing controls like Hangup and Answer
     */
    fun showRingingControls() {
        isHidden = false
        currentBottomSheetLayout = ringingBottomSheet.bottomSheetLayoutContent
        removeAnchorFromAnchoredView()
        ringingBottomSheet.bottomSheetLayoutContent.id = R.id.bandyer_id_ringing_bottom_sheet
        if (callBottomSheet.isVisible() || audioRouteBottomSheet.isVisible()) {
            callBottomSheet.dispose()
            audioRouteBottomSheet.dispose()
        }
        ringingBottomSheet.show()
        anchorViews()
    }

    /**
     * Anchor a view to the widget
     * @param anchoredView View to anchor
     * @param gravity gravity to apply
     * @param forceAnchor true to anchor no matter what, false otherwise
     */
    @JvmOverloads
    fun setAnchoredView(anchoredView: View, gravity: Int, forceAnchor: Boolean = false) {
        var hasAnchoredView = false
        anchoredViews.forEach {
            if (it.first == anchoredView) {
                hasAnchoredView = true
                return@forEach
            }
        }
        if (!hasAnchoredView) {
            anchoredViews.add(Pair(anchoredView, gravity))
            checkAnchoredViewInCoordinatorLayout()
            if (forceAnchor) anchorViews()
        }
    }

    private fun anchorViews() {
        if (currentBottomSheetLayout == null) return

        anchoredViews.forEach { pair ->
            val lp = (pair.first.layoutParams as CoordinatorLayout.LayoutParams)
            lp.anchorId = currentBottomSheetLayout?.id ?: View.NO_ID
            lp.anchorGravity = pair.second
            lp.gravity = pair.second
            pair.first.layoutParams = lp
            pair.first.visibility = View.VISIBLE
        }
    }

    /**
     * Anchors current bottom sheet
     */
    fun anchor() {
        currentShownBottomSheet?.anchor()
    }

    /**
     */
    fun hideCallControls(withTimer: Boolean = false) {
        if (!withTimer) {
            onHidingTimerFinished()
            hidingTimer?.cancel()
            return
        }
        if (currentShownBottomSheet is CallBottomSheet<*>) hidingTimer?.start()
    }

    private fun checkAnchoredViewInCoordinatorLayout() {
        if (currentBottomSheetLayout == null) return
        anchoredViews.forEach { pair ->
            if (pair.second == Gravity.TOP)
                (pair.first.layoutParams as ViewGroup.MarginLayoutParams).bottomMargin = pair.first.height
        }
    }

    private fun removeAnchorFromAnchoredView() {
        anchoredViews.forEach { pair ->
            if (pair.first.parent == null || pair.first.parent !is CoordinatorLayout)
                return
            val lp = (pair.first.layoutParams as CoordinatorLayout.LayoutParams)
            lp.anchorId = View.NO_ID
            lp.anchorGravity = Gravity.TOP
            lp.gravity = Gravity.BOTTOM or Gravity.END
            pair.first.layoutParams = lp
            pair.first.visibility = View.INVISIBLE
        }
    }

    /**
     * Show the call controls
     * @param collapsible true if the bottomSheet should be collapsible, false otherwise
     * @param fixed true if it can not be moved, false otherwise
     * @param collapsed optional initial state collapsed
     */
    @JvmOverloads
    fun showCallControls(collapsible: Boolean, fixed: Boolean = false, collapsed: Boolean = false) {
        isHidden = false
        currentBottomSheetLayout = callBottomSheet.bottomSheetLayoutContent
        removeAnchorFromAnchoredView()
        ringingBottomSheet.bottomSheetLayoutContent.id = R.id.bandyer_id_ringing_bottom_sheet
        if (ringingBottomSheet.isVisible())
            ringingBottomSheet.dispose()
        if (audioRouteBottomSheet.isVisible())
            audioRouteBottomSheet.dispose()

        this.collapsible = collapsible
        this.fixed = fixed
        callBottomSheet.bottomSheetLayoutContent.id = R.id.bandyer_id_call_bottom_sheet
        callBottomSheet.show(collapsible, fixed, collapsed)
        callBottomSheet.updateAudioRouteIcon(audioRouteBottomSheet.mCurrentAudioRoute)
        anchorViews()
    }

    override fun onHidingTimerFinished() {
        if (!collapsible) return
        callBottomSheet.collapse()
    }

    /**
     * Show the audioRoute bottomSheet
     */
    fun showAudioRouteBottomSheet() {
        isHidden = false
        audioRouteBottomSheet.bottomSheetLayoutContent.id = R.id.bandyer_id__audio_route_bottom_sheet
        removeAnchorFromAnchoredView()
        callBottomSheet.hide(true)
        audioRouteBottomSheet.show()
        currentBottomSheetLayout = audioRouteBottomSheet.bottomSheetLayoutContent
        anchorViews()
    }

    /**
     * Add an audioRoute item to the list of available routes
     * @param audioRoute AudioRoute to add
     */
    fun addAudioRouteItem(audioRoute: AudioRoute) {
        audioRouteBottomSheet.addAudioRouteItem(audioRoute)
    }

    /**
     * Select an audioRoute item from the list of available routes
     * @param audioRoute AudioRoute? to select
     */
    fun selectAudioRoute(audioRoute: AudioRoute?) {
        audioRouteBottomSheet.selectAudioRoute(audioRoute)
        callBottomSheet.updateAudioRouteIcon(audioRoute)
    }

    /**
     * Remove an audioRoute item from the list of available routes
     * @param audioRoute AudioRoute to remove
     */
    fun removeAudioRouteItem(audioRoute: AudioRoute) {
        audioRouteBottomSheet.removeAudioRouteItem(audioRoute)
    }

    private val onBottomSheetCallBacks = object : BandyerActionBottomSheet.OnActionBottomSheetListener<ActionItem, BandyerBottomSheet>, OnStateChangedBottomSheetListener<BandyerBottomSheet> {

        override fun onActionClicked(bottomSheet: BandyerBottomSheet, action: ActionItem, position: Int): Boolean {
            when (bottomSheet) {
                is CallBottomSheet<*> -> {
                    val consumed = onClickListener?.onCallActionClicked(action as CallAction, position)
                    if (consumed == true)
                        return false
                    when (action) {
                        is CallAction.AUDIOROUTE -> showAudioRouteBottomSheet()
                        is CallAction.OPTIONS -> bottomSheet.expand()
                    }
                    return true
                }
                is AudioRouteBottomSheet<*> -> {
                    val consumed = onClickListener?.onAudioRouteClicked(action as AudioRoute, position)
                    if (consumed == true) return false
                    callBottomSheet.updateAudioRouteIcon(audioRouteBottomSheet.mCurrentAudioRoute)
                    bottomSheet.mCurrentAudioRoute = action as AudioRoute
                    bottomSheet.hide(true)

                    return true
                }
                is RingingBottomSheet<*> -> {
                    val consumed = onClickListener?.onCallActionClicked(action as CallAction, position)
                    if (consumed == true) return false
                    return true
                }
                else -> return false
            }
        }

        override fun onExpand(bottomSheet: BandyerBottomSheet) {
            disableAutoHide()
            state = BandyerCallActionWidgetState.EXPANDED
            if(bottomSheet is RingingBottomSheet<*>) {
                onShowListener?.onShown()
                onShowListener = null
            }
            slidingListener?.onStateChanged(BandyerCallActionWidgetState.EXPANDED)
        }

        override fun onAnchor(bottomSheet: BandyerBottomSheet) {
            if (bottomSheet is CallBottomSheet<*>) hidingTimer?.start()
            state = BandyerCallActionWidgetState.ANCHORED
            slidingListener?.onSlide(currentBottomSheetLayout?.top
                    ?: context.getScreenSize().y, false)
            if(bottomSheet is CallBottomSheet<*>) {
                onShowListener?.onShown()
                onShowListener = null
            }
            slidingListener?.onStateChanged(BandyerCallActionWidgetState.ANCHORED)
        }

        override fun onShow(bottomSheet: BandyerBottomSheet) {
            currentShownBottomSheet = bottomSheet
            anchorViews()
            (bottomSheet as? CallBottomSheet<*>)?.updateAudioRouteIcon(audioRouteBottomSheet.mCurrentAudioRoute)
        }

        override fun onHide(bottomSheet: BandyerBottomSheet) {
            when (bottomSheet) {
                is CallBottomSheet<*> -> disableAutoHide()
                is AudioRouteBottomSheet<*> -> {
                    if (!isHidden) showCallControls(collapsible, fixed)
                    bottomSheet.dispose()
                }
                is RingingBottomSheet<*> -> bottomSheet.dispose()
            }
            onHiddenListener?.onHidden()
            onHiddenListener = null
        }

        override fun onDragging(bottomSheet: BandyerBottomSheet) {
            disableAutoHide()
        }

        override fun onCollapse(bottomSheet: BandyerBottomSheet) {
            hidingTimer?.onFinish()
            state = BandyerCallActionWidgetState.COLLAPSED
            slidingListener?.onSlide(currentBottomSheetLayout?.top
                    ?: context.getScreenSize().y, true)

            if(bottomSheet is CallBottomSheet<*>) {
                onShowListener?.onShown()
                onShowListener = null
            }
            slidingListener?.onStateChanged(BandyerCallActionWidgetState.COLLAPSED)
        }

        override fun onSlide(bottomSheet: BandyerBottomSheet, slideOffset: Float) {
            if (slideOffset == 0.0f) return
            disableAutoHide()
            val top = currentBottomSheetLayout?.top ?: return
            slidingListener?.onSlide(
                    top,
                    (currentShownBottomSheet?.state == BandyerBottomSheetBehaviour.STATE_COLLAPSED || currentShownBottomSheet?.state == BandyerBottomSheetBehaviour.STATE_HIDDEN)
            )
        }
    }

    private val onAudioRoutesListener = object : OnAudioRouteBottomSheetListener {
        override fun onAudioRoutesRequested(): List<AudioRoute>? {
            return onAudioRoutesRequest?.onAudioRoutesRequested()
        }
    }

    private var callBottomSheet: CallBottomSheet<T> = CallBottomSheet(
            context,
            callActionItems,
            R.style.BandyerSDKDesign_Widget_BottomSheetLayout_Call)

    private var ringingBottomSheet: BandyerClickableBottomSheet<T> = RingingBottomSheet(
            context,
            R.style.BandyerSDKDesign_Widget_BottomSheetLayout_Ringing)


    private var audioRouteBottomSheet: AudioRouteBottomSheet<T> = AudioRouteBottomSheet(
            context = context,
            onAudioRoutesRequest = onAudioRoutesListener,
            audioRouteItems = onAudioRoutesRequest?.onAudioRoutesRequested(),
            initial_selection = -1,
            bottomSheetStyle = R.style.BandyerSDKDesign_Widget_BottomSheetLayout_AudioRoute)

    /**
     * Dispose the widget
     */
    fun dispose() {
        removeAnchorFromAnchoredView()
        onHiddenListener = null
        onShowListener = null
        slidingListener = null
        callBottomSheet.dispose()
        audioRouteBottomSheet.dispose()
        ringingBottomSheet.dispose()
    }

    /**
     * Returns true if the widget is collapsible, false otherwise.
     */
    var collapsible: Boolean = true
    private var fixed: Boolean = true

    init {
        ringingBottomSheet.onStateChangedBottomSheetListener = onBottomSheetCallBacks
        ringingBottomSheet.onActionBottomSheetListener = onBottomSheetCallBacks

        audioRouteBottomSheet.onStateChangedBottomSheetListener = onBottomSheetCallBacks
        audioRouteBottomSheet.onActionBottomSheetListener = onBottomSheetCallBacks

        callBottomSheet.onStateChangedBottomSheetListener = onBottomSheetCallBacks
        callBottomSheet.onActionBottomSheetListener = onBottomSheetCallBacks
    }

    /**
     * Click Listener for the call action widget
     */
    interface OnClickListener {
        /**
         * Called when a call action has been requested
         * @param item CallAction requested
         * @param position position of item
         * @return true if has been handled, false otherwise
         */
        fun onCallActionClicked(item: CallAction, position: Int): Boolean

        /**
         * Called when an audioRoute has been selected
         * @param item AudioRoute requested
         * @param position position of item
         * @return true if has been handled, false otherwise
         */
        fun onAudioRouteClicked(item: AudioRoute, position: Int): Boolean
    }


    /**
     * Sliding Listener for the call action widget
     */
    interface SlidingListener {

        /**
         * Called when the widget slide offset changes.
         * @param top distance from the top of the parent in pixels
         * @param isCollapsed true if call action widget is collapsed
         */
        fun onSlide(@Px top: Int, isCollapsed: Boolean)

        /**
         * Called when the call action widget changes state.
         * @param state BandyerCallActionWidgetState
         */
        fun onStateChanged(state: BandyerCallActionWidgetState)
    }

    /**
     * Listener to be called after on shown process
     */
    interface OnShowListener {
        /**
         * Callback fired when the widget has been programmatically shown
         */
        fun onShown()
    }

    /**
     * Listener to be called after on hidden process
     */
    interface OnHiddenListener {
        /**
         * Callback fired when the widget has been programmatically hidden
         */
        fun onHidden()
    }
}